package ohd.hseb.hefs.utils.plugins;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import ohd.hseb.util.misc.MiscTools;

/**
 * A default factory implementation that has as attributes {@link #_packageName} and {@link #_classSuffix}.
 * 
 * @author hankherr
 * @param <E> The class of objects instantiated by this factory.
 */
public class DefaultFactory<E>
{
    protected final HashMap<String, GeneralPlugInParameters> _xmlTagNameTosInstanceMap = new HashMap<String, GeneralPlugInParameters>();
    private String _packageName = "";
    private String _classSuffix = "";

    public DefaultFactory(final String packageName, final String classSuffix)
    {
        _packageName = packageName;
        _classSuffix = classSuffix;
    }

    /**
     * Loads an instance of the class with the name classNameOrType + {@link #_classSuffix} from the package specified
     * by attribute {@link #_packageName}.
     * 
     * @param classNameOrType Name of class to generate an instance of without the {@link #_classSuffix}.
     * @return Object of that type.
     * @throws DefaultFactoryException Thrown if class is not found, class cannot be instantiated, or it cannot be
     *             accessed for whatever reason.
     */
    @SuppressWarnings("unchecked")
    public E loadInstance(final String classNameOrType) throws DefaultFactoryException
    {
        //Build the class name.  It is either the passed in string or one of the defaults
        //if the passed in class does not exist.
        String className = classNameOrType;
        try
        {
            Class.forName(classNameOrType);
        }
        catch(final ClassNotFoundException cnfe)
        {
            className = _packageName + "." + classNameOrType + _classSuffix;
        }

        //Now attempt to load it.
        try
        {
            final Class<?> classFound = Class.forName(className);
            return (E)classFound.getDeclaredConstructor().newInstance();
        }
        catch(final ClassNotFoundException e)
        {
            throw new DefaultFactoryException("Cannot find plug-in class " + classNameOrType + ": " + e.getMessage());
        }
        catch(final InstantiationException e)
        {
            throw new DefaultFactoryException("Cannot instantiate plug-in class " + classNameOrType + ": "
                + e.getMessage());
        }
        catch(final IllegalAccessException e)
        {
            throw new DefaultFactoryException("Cannot access plug-in class " + classNameOrType + ": " + e.getMessage());
        } 
        catch (IllegalArgumentException e) 
        {
			throw new DefaultFactoryException("Cannot access plug-in class " + classNameOrType + ": " + e.getMessage());
		} 
        catch (InvocationTargetException e) 
        {
			throw new DefaultFactoryException("Cannot access plug-in class " + classNameOrType + ": " + e.getMessage());
		} 
        catch (NoSuchMethodException e) 
        {
			throw new DefaultFactoryException("Cannot access plug-in class " + classNameOrType + ": " + e.getMessage());
		} 
        catch (SecurityException e) 
        {
			throw new DefaultFactoryException("Cannot access plug-in class " + classNameOrType + ": " + e.getMessage());
		}
    }

    /**
     * @return A list of strings where each is a just the list of classes matching the {@link #_classSuffix} with the
     *         package name stripped out.
     */
    public List<String> obtainListOfClassesWithSuffixInDefaultPackage() throws DefaultFactoryException
    {
        final List<String> results = new ArrayList<String>();
        final int suffixLength = _classSuffix.length();

        //+1 accounts for the period at the end
        final int prefixLength = _packageName.length() + 1;

        @SuppressWarnings("rawtypes")
        List<Class> classes;
        try
        {
            classes = MiscTools.getClassesForPackage(_packageName);
        }
        catch(final ClassNotFoundException e)
        {
            throw new DefaultFactoryException("Unable to get list of classes in package " + _packageName + ": "
                + e.getMessage());
        }

        for(int i = 0; i < classes.size(); i++)
        {
            if(classes.get(i).getName().endsWith(_classSuffix))
            {
                final String className = classes.get(i)
                                                .getName()
                                                .substring(prefixLength,
                                                           classes.get(i).getName().length() - suffixLength);
                if(results.indexOf(className) < 0)
                {
                    results.add(className);
                }
            }
        }
        return results;
    }
}
