package ohd.hseb.hefs.utils.plugins;

import java.util.ArrayList;
import java.util.List;

import javax.swing.JPanel;


/**
 * A JPanel designed to allow users to modify a GraphicsGeneratorPlugInParameters instance.
 * 
 * @author herrhd
 */
public abstract class GeneralPlugInPanel extends JPanel
{
    private static final long serialVersionUID = 1L;

    private final List<GeneralPlugInPanelListener> _panelListeners = new ArrayList<GeneralPlugInPanelListener>();

    private GeneralPlugInParameters _managedParameters = null;

    /**
     * Sets the _managedParameters for this panel.
     * 
     * @param parameters GraphicsGeneratorPlugInParameters object being managed.
     */
    public void setManagedParameters(GeneralPlugInParameters parameters)
    {
        _managedParameters = parameters;
    }

    /**
     * @return The managed parameters for this plug-in panel.
     */
    public GeneralPlugInParameters getManagedParameters()
    {
        return this._managedParameters;
    }

    /**
     * @param listener An implementer of GraphicsGeneratorPlugInPanelListener which will receive signals when changes
     *            are made to the parameters via this panel.
     */
    public void addGraphicsGeneratorPlugInPanelListener(GeneralPlugInPanelListener listener)
    {
        if(_panelListeners.indexOf(listener) < 0)
        {
            _panelListeners.add(listener);
        }
    }

    /**
     * @param listener GraphicsGeneratorPlugInPanelListener implementer to be removed.
     */
    public void removeGraphicsGeneratorPlugInPanelListener(GeneralPlugInPanelListener listener)
    {
        _panelListeners.remove(listener);
    }

    /**
     * Eliminates all current GraphicsGeneratorPlugInPanelListener from the list of listeners.
     */
    public void clearGraphicsGeneratorPlugInPanelListeners()
    {
        this._panelListeners.clear();
    }

    /**
     * Calls the parameterChanged method for all listeners passing in _managedParameters.
     */
    public void fireParametersChanged()
    {
        for(int i = 0; i < _panelListeners.size(); i++)
        {
            _panelListeners.get(i).parameterChanged(_managedParameters);
        }
    }

    @Override
    public void setEnabled(boolean b)
    {
        super.setEnabled(b);
        if(b)
        {
            enableSubPanel();
        }
        else
        {
            disableSubPanel();
        }
    }

    /**
     * Disable the subpanel turning off user interaction.
     */
    public abstract void disableSubPanel();

    /**
     * Enable the subpanel allowing for user interaction.
     */
    public abstract void enableSubPanel();

    /**
     * Called by panes when the argument has changed. Override this only if your plug-in panel must react to a change in
     * the managed parameter's ArgumentsProcessor.
     */
    public void argumentsChanged()
    {
    }

    /**
     * Called whenever an undo state is recorded, it should return a status that can be used to determine the current
     * appearance of the panel. For instance, if a table row is selected, you may return the selected row number.
     * 
     * @return Object summarizing the status, processed via resetPanelStatus.
     */
    public Object getPanelStatus()
    {
        return null;
    }

    /**
     * Called whenever an undo is performed, it should assume the Object passed in is an instance of the getPanelStatus
     * return value and react accordingly. It should never be necessary for this to fire parameterChanged events. It is
     * up to the tabbed pane displaying this panel to update everything necessary. This method need only update its own
     * display to reflect the state according to the status passed in.
     * 
     * @param status Object summarizing the status; output by getPanelStatus.
     */
    public void resetPanelStatus(Object status)
    {
    }
}
