package ohd.hseb.hefs.utils.plugins;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLWriterException;
import ohd.hseb.util.misc.HString;
import ohd.hseb.util.misc.SegmentedLine;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.Attributes;

import com.google.common.collect.Lists;

/**
 * An instance of GeneralPlugInParameters that manages a list of GenericParameter instances.
 * 
 * @author hank.herr
 */
public class GenericParameterList extends DefaultPlugInParameters
{

    private String _parameterXMLTagName = "parameter";

    /**
     * Parameters are assumed to NOT be case sensitive!
     */
    private final List<GenericParameter> _parameters = new ArrayList<GenericParameter>();

    public GenericParameterList()
    {
        setXMLTagName("parameters");
    }

    public GenericParameterList(final String parameterXMLTagName)
    {
        setXMLTagName("parameters");
        this._parameterXMLTagName = parameterXMLTagName;
    }

    public void sort()
    {
        Collections.sort(_parameters, new Comparator<GenericParameter>()
        {

            @Override
            public int compare(final GenericParameter o1, final GenericParameter o2)
            {
                return (o1.getName().compareTo(o2.getName()));
            }

        });
    }

    /**
     * For some plugins, it may be allowed for parameters in the list to be defined with pipe-delimited values such that
     * it is equivalent to listing each value in the list as a separate generic parameter. This method will flatten
     * those lists; i.e., it will take out the pipi-delimited values and add a single entry in the list after the entry
     * being flattened. It will then set the first item in the list to the value of the parameter that originally
     * included the list.<br>
     * <br>
     * NOTE: Pipe, |, was chosen becuase commas and semicolons may be part of the parameters to be parsed. For example,
     * commas may show up in numbers and semi-colors in the parameters to replaceable arguments.
     * 
     * @param parameterName The name of the parameters to flatten. Pass in null if the entire list should be flattened.
     */
    public void flattenList(final String parameterName)
    {
        for(int i = _parameters.size() - 1; i >= 0; i--)
        {
            final GenericParameter parm = _parameters.get(i);

            //Check the parameter name.
            if((parameterName == null) || (parm.isForParameter(parameterName)))
            {

                //Segment the value based on comma and only change the list if more than one segment is found.
                //SEPARATOR IS A '|'
                final SegmentedLine segLine = new SegmentedLine(parm.getValue(), "|");
                if(segLine.getNumberOfSegments() > 1)
                {
                    //For each segment... If it is the first, then replace the parameter already found.  Otherwise,
                    //add a new list element in the slot specified by i + j (to maintain order).
                    for(int j = 0; j < segLine.getNumberOfSegments(); j++)
                    {
                        if(j == 0)
                        {
                            parm.setValue(segLine.getSegment(j).trim());
                        }
                        else
                        {
                            _parameters.add(i + j, new GenericParameter(parm.getName(), segLine.getSegment(j).trim()));
                        }
                    }
                }
            }
        }
    }

    public String getParameterXMLTagName()
    {
        return this._parameterXMLTagName;
    }

    public List<GenericParameter> getParameters()
    {
        return this._parameters;
    }

    /**
     * @return {@link List} of the parameter names.
     */
    public List<String> getParameterNames()
    {
        final List<String> results = Lists.newArrayList();
        for(final GenericParameter parm: _parameters)
        {
            results.add(parm.getName());

        }
        return results;
    }

    public int getParameterCount()
    {
        return this._parameters.size();
    }

    public void clearParameters()
    {
        _parameters.clear();
    }

    /**
     * Remove all parameters except for those passed in.
     * 
     * @param parametersToKeep List of parameter names.
     */
    public void clearParametersExceptForThese(final String[] parametersToKeep)
    {
        boolean keep = false;
        for(int i = _parameters.size() - 1; i >= 0; i--)
        {
            keep = false;
            final GenericParameter current = _parameters.get(i);
            for(int j = 0; j < parametersToKeep.length; j++)
            {
                if(current.isForParameter(parametersToKeep[j]))
                {
                    keep = true;
                }
            }
            if(!keep)
            {
                _parameters.remove(i);
            }
        }
    }

    /**
     * Make sure a GenericParameter exists for all parameters passed in, even if the value is empty. This makes sure
     * there is something to hold a parameter.
     * 
     * @param parametersToExist
     */
    public void ensureTheseParametersExist(final String[] parametersToExist)
    {
        for(int i = 0; i < parametersToExist.length; i++)
        {
            if(this.getParameterWithName(parametersToExist[i]) == null)
            {
                this.addParameter(parametersToExist[i], "");
            }
        }
    }

    public void addParameter(final GenericParameter param)
    {
        _parameters.add(param);
    }

    public void addParameter(final String name, final String value)
    {
        final GenericParameter param = new GenericParameter(name, value);
        param.setXMLTagName(this._parameterXMLTagName);
        addParameter(param);
    }

    public void addParameters(final GenericParameterList added)
    {
        for(int i = 0; i < added.getParameters().size(); i++)
        {
            this.addParameter(added.getParameters().get(i));
        }
    }

    public void addParameterIfItDoesNotExist(final String name, final String value)
    {
        if(!doesParameterExist(name))
        {
            addParameter(name, value);
        }
    }

    public void removeParameters(final String name)
    {
        for(int i = _parameters.size() - 1; i >= 0; i--)
        {
            if(_parameters.get(i).isForParameter(name))
            {
                _parameters.remove(i);
            }
        }
    }

    public void removeParameters(final String name, final String value)
    {
        final GenericParameter matcher = new GenericParameter(name, value);
        matcher.setXMLTagName(this._parameterXMLTagName);
        for(int i = _parameters.size() - 1; i >= 0; i--)
        {
            if(_parameters.get(i).equals(matcher))
            {
                _parameters.remove(i);
            }
        }
    }

    public void replaceAllMatchingParameters(final String name, final String newValue)
    {
        final List<GenericParameter> params = this.getParametersWithNameInOrder(name);
        for(int i = 0; i < params.size(); i++)
        {
            params.get(i).setValue(newValue);
        }
    }

    public void replaceFirstMatchingParameter(final String name, final String newValue)
    {
        this.replaceFirstMatchingParameter(name, newValue, false);
    }

    public void replaceFirstMatchingParameter(final String name, final String newValue, final boolean addIfNotFound)
    {
        GenericParameter parameter = this.getParameterWithName(name);
        if(parameter != null)
        {
            parameter.setValue(newValue);
        }
        else if(addIfNotFound)
        {
            parameter = new GenericParameter(name, newValue);
            parameter.setXMLTagName(this._parameterXMLTagName);
            _parameters.add(parameter);
        }
    }

    /**
     * @param parameterNamesToCheckFor List of parameter names.
     * @throws Exception If the parameters does not exist. This is not intended for use in if clauses. For if clauses
     *             call the getParameters* methods and check for list size.
     */
    public void checkForExistenceOfParameters(final List<String> parameterNamesToCheckFor) throws Exception
    {
        checkForExistenceOfParameters((String[])parameterNamesToCheckFor.toArray());
    }

    /**
     * @param parameterNamesToCheckFor Comma delimited list of parameter names in a String.
     * @param delimiter
     * @throws Exception If the parameters does not exist. This is not intended for use in if clauses. For if clauses
     *             call the getParameters* methods and check for list size.
     */
    public void checkForExistenceOfParameters(final String parameterNamesToCheckFor, final String delimiter) throws Exception
    {
        checkForExistenceOfParameters(HString.buildListFromString(parameterNamesToCheckFor, delimiter));
    }

    public boolean doesParameterExist(final String name)
    {
        if((name == null) || (name.length() == 0))
        {
            return false;
        }
        if(getParametersWithNameInOrder(name).size() == 0)
        {
            return false;
        }
        return true;
    }

    /**
     * @param parameterNamesToCheckFor Array of parameter names.
     * @throws Exception If the parameters does not exist. This is not intended for use in if clauses. For if clauses
     *             call the getParameters* methods and check for list size.
     */
    public void checkForExistenceOfParameters(final String[] parameterNamesToCheckFor) throws Exception
    {
        if((parameterNamesToCheckFor == null) || (parameterNamesToCheckFor.length == 0))
        {
            return;
        }
        for(int i = 0; i < parameterNamesToCheckFor.length; i++)
        {
            if(getParametersWithNameInOrder(parameterNamesToCheckFor[i]).size() == 0)
            {
                throw new Exception(parameterNamesToCheckFor[i] + " parameter is not defined.");
            }
        }
    }

    public List<GenericParameter> getParametersWithNameInOrder(final String name)
    {
        final List<GenericParameter> results = new ArrayList<GenericParameter>();
        for(int i = 0; i < _parameters.size(); i++)
        {
            if(_parameters.get(i).isForParameter(name))
            {
                results.add(_parameters.get(i));
            }
        }
        return results;
    }

    public GenericParameter getParameterWithName(final String name)
    {
        final List<GenericParameter> parms = getParametersWithNameInOrder(name);
        if(parms.size() == 0)
        {
            return null;
        }
        else
        {
            return parms.get(0);
        }
    }

    /**
     * @return The index of the first argument within {@link #_parameters} with the specified name, or -1 if not found.
     */
    public int indexOfArgumentWithName(final String name)
    {
        for(int i = 0; i < _parameters.size(); i++)
        {
            if(_parameters.get(i).isForParameter(name))
            {
                return i;
            }
        }
        return -1;
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
    {
        final GenericParameter param = new GenericParameter();
        param.setXMLTagName(this._parameterXMLTagName);
        if(elementName.equals(getXMLTagName()))
        {
            clearParameters();
        }
        else if(elementName.equals(param.getXMLTagName()))
        {
            _parameters.add(param);
            return param;
        }
        else
        {
            throw new XMLReaderException("Within " + this.getXMLTagName() + ", invalid element tag name '"
                + elementName + "'.");
        }

        return null;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        final Element mainElement = request.createElement(getXMLTagName());
        for(int i = 0; i < this._parameters.size(); i++)
        {
            mainElement.appendChild(_parameters.get(i).writePropertyToXMLElement(request));
        }
        return mainElement;
    }

    @Override
    public String toString()
    {
        String result = "GenericParameterList: ";
        result += "xmlTag = '" + this.getXMLTagName() + "'; ";
        for(int i = 0; i < this._parameters.size(); i++)
        {
            result += "[" + _parameters.get(i).toString() + "]";
        }
        result += ".";
        return result;
    }

    /**
     * Ignores ordering of list. However, every list item must be exactly matched.
     */
    @Override
    public boolean equals(final Object obj)
    {
        if(!(obj instanceof ohd.hseb.hefs.utils.plugins.GenericParameterList))
        {
            return false;
        }

        final GenericParameterList otherList = (GenericParameterList)obj;

        if(!ListTools.twoSidedListCheck(_parameters, otherList._parameters))
        {
            return false;
        }

        return true;
    }

    @Override
    public Object clone()
    {
        final GenericParameterList results = new GenericParameterList();
        results.copyFrom(this);
        return results;
    }

    @Override
    public String getShortGUIDisplayableParametersSummary()
    {
        String result = "";
        for(int i = 0; i < this._parameters.size(); i++)
        {
            result += this._parameters.get(i).getShortGUIDisplayableParametersSummary();
            if(i < this._parameters.size() - 1)
            {
                result += "; ";
            }
        }
        return result;
    }

    @Override
    public void copyFrom(final GeneralPlugInParameters parameters)
    {
        final GenericParameterList base = (GenericParameterList)parameters;
        this.clearParameters();
        this._parameterXMLTagName = base.getParameterXMLTagName();
        for(int i = 0; i < base.getParameters().size(); i++)
        {
            addParameter((GenericParameter)base.getParameters().get(i).clone());
        }
        setXMLTagName(base.getXMLTagName());
    }
}
