package ohd.hseb.hefs.utils.plugins;

import java.awt.FlowLayout;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Collection;
import java.util.Vector;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import ohd.hseb.hefs.utils.gui.tools.HSwingFactory;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.xml.vars.XMLTimeStep;
import ohd.hseb.util.misc.HString;

/**
 * Panel designed to edit a time step parameter. A parameter changed event will be fired if the user modifies the value.
 * It will only be fired when the modification results in an actual change.
 * 
 * @author hank.herr
 */
public class TimeStepGenericParameterPanel extends GeneralPlugInPanel implements ItemListener, ChangeListener
{
    private static final long serialVersionUID = 1L;

    private final JCheckBox _includeInListCheckBox = HSwingFactory.createJCheckBox("Override Parameter", null, false);
    private final JSpinner _quantitySpinner = HSwingFactory.createJSpinner(null, 3, 0, 0, null, 1);

    private final JComboBox<String> _unitComboBox;
    private String _nameOfParameterToManage;
    private final Collection<String> _specialUnits;

    /**
     * Subset of {@link XMLTimeStep#UNIT_CHOICES} that removes duplicates that exist to cover plural/singular.
     */
    private static final String[] NORMAL_UNIT_CHOICES = new String[]{"years", "months", "weeks", "days", "hours"};

    /**
     * Special version for if the panel is to be a stand-alone, it creates a parameter object to store the value. When
     * set, call {@link #getCurrentValueOfManagedParameter()}. Note that the {@link JCheckBox} returned by
     * {@link #getIncludeInListCheckBox()} is useless for this version and is assumed to always be checked.
     */
    public TimeStepGenericParameterPanel(final String initialTimeStep)
    {
        this(new GenericParameterList("filler"), "timeParm");
        getParameterList().addParameter("timeParm", initialTimeStep);
        makePanelReflectParameters();
    }

    public TimeStepGenericParameterPanel(final GeneralPlugInParameters parameters, final String nameOfParameterToManage)
    {
        this(parameters, nameOfParameterToManage, XMLTimeStep.PERIOD_SPECIAL_UNITS);
    }

    public TimeStepGenericParameterPanel(final GeneralPlugInParameters parameters,
                                         final String nameOfParameterToManage,
                                         final Collection<String> specialUnits)
    {
        this.setManagedParameters(parameters);
        _nameOfParameterToManage = nameOfParameterToManage;
        _specialUnits = specialUnits;
        _unitComboBox = new JComboBox<String>(new Vector<String>(ListTools.concat(_specialUnits.toArray(new String[1]),
                                                                                  NORMAL_UNIT_CHOICES)));
        createDisplay();
        makePanelReflectParameters();
    }

    public void createDisplay()
    {
        _includeInListCheckBox.setText("Override " + _nameOfParameterToManage);
    }

    protected void populateGUIItemsFromString(final String value)
    {
        final int pos = HString.findFirstNonNumericCharacter(value);
        if((pos < 0) || (pos >= value.length()))
        {
            //This should never happen, as long as I modify the validate method correctly.
            _includeInListCheckBox.setSelected(false);
            _quantitySpinner.setValue(0);
            _unitComboBox.setSelectedIndex(0);
        }
        else
        {
            final String quantity = value.substring(0, pos).trim();
            final String unit = value.substring(pos, value.length()).trim();
            _quantitySpinner.setValue(Integer.parseInt(quantity));
            for(int i = 0; i < _unitComboBox.getItemCount(); i++)
            {
                final String element = _unitComboBox.getItemAt(i);
                if(element.contains(unit))
                {
                    _unitComboBox.setSelectedItem(element);
                }
            }
        }
        _includeInListCheckBox.setSelected(true);
    }

    protected void removeListeners()
    {
        _includeInListCheckBox.removeItemListener(this);
        _quantitySpinner.removeChangeListener(this);
        _unitComboBox.removeItemListener(this);

    }

    protected void addListeners()
    {

        _includeInListCheckBox.addItemListener(this);
        _quantitySpinner.addChangeListener(this);
        _unitComboBox.addItemListener(this);
    }

    protected void makePanelReflectParameters()
    {
        removeListeners();

        final GenericParameter existingParameter = getEditedParameter();
        if(existingParameter != null)
        {
            this.populateGUIItemsFromString(existingParameter.getValue());
        }
        else
        {
            _includeInListCheckBox.setSelected(false);
            _quantitySpinner.setValue(0);
            _unitComboBox.setSelectedIndex(0);
        }

        addListeners();
    }

    protected void makeParametersReflectPanel()
    {
        if(!this._includeInListCheckBox.isSelected())
        {
            getParameterList().removeParameters(this._nameOfParameterToManage);
        }
        else
        {
            if(getEditedParameter() == null)
            {
                getParameterList().addParameter(this._nameOfParameterToManage, constructParameterValueFromPanel());
            }
            else
            {
                getEditedParameter().setValue("" + this._quantitySpinner.getValue() + " "
                    + this._unitComboBox.getSelectedItem());
            }
        }
    }

    protected String constructParameterValueFromPanel()
    {
        return "" + this._quantitySpinner.getValue() + " " + this._unitComboBox.getSelectedItem();
    }

    public JCheckBox getIncludeInListCheckBox()
    {
        return this._includeInListCheckBox;
    }

    public JPanel getEditingDisplayPanel()
    {
        final JPanel result = new JPanel(new FlowLayout(FlowLayout.LEFT));
        result.add(_quantitySpinner);
        result.add(_unitComboBox);
        return result;
    }

    public GenericParameterList getParameterList()
    {
        return (GenericParameterList)this.getManagedParameters();
    }

    public GenericParameter getEditedParameter()
    {
        return getParameterList().getParameterWithName(_nameOfParameterToManage);
    }

    public String getCurrentValueOfManagedParameter()
    {
        if(getEditedParameter() == null)
        {
            return null;
        }
        return getEditedParameter().getValue();
    }

    protected boolean hasParameterBeenChanged()
    {
        return !(constructParameterValueFromPanel().equals(getCurrentValueOfManagedParameter()));
    }

    protected void setNameOfParameterToManage(final String name)
    {
        this._nameOfParameterToManage = name;
    }

    protected String getNameOfParameterToManage()
    {
        return this._nameOfParameterToManage;
    }

    @Override
    public void disableSubPanel()
    {
        this._includeInListCheckBox.setEnabled(false);
        this._quantitySpinner.setEnabled(false);
        this._unitComboBox.setEnabled(false);
    }

    @Override
    public void enableSubPanel()
    {
        this._includeInListCheckBox.setEnabled(true);
        this._quantitySpinner.setEnabled(true);
        this._unitComboBox.setEnabled(true);
    }

    @Override
    public void itemStateChanged(final ItemEvent arg0)
    {
        if(_specialUnits.contains(_unitComboBox.getSelectedItem().toString()))
        {
            this._quantitySpinner.setValue(1);
            this._quantitySpinner.setEnabled(false);
        }
        else
        {
            this._quantitySpinner.setEnabled(true);
        }

        if(hasParameterBeenChanged())
        {
            this.makeParametersReflectPanel();
            this.fireParametersChanged();
        }
    }

    @Override
    public void stateChanged(final ChangeEvent arg0)
    {
        if(hasParameterBeenChanged())
        {
            this.makeParametersReflectPanel();
            this.fireParametersChanged();
        }
    }

}
