package ohd.hseb.hefs.utils.rc;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLDouble;
import ohd.hseb.hefs.utils.xml.vars.XMLFloat;
import ohd.hseb.hefs.utils.xml.vars.XMLString;

import com.google.common.collect.Lists;

/**
 * A {@link HRatingCurveTable} is a collection of {@link RatingCurveRow} instances (that class is defined here) combined
 * with an interpolation method, minimum stage, and maximum stage.
 * 
 * @author hankherr
 */
public class HRatingCurveTable implements XMLReadable, XMLWritable
{
    private final XMLString _interpolationMethod = new XMLString("interpolationMethod");
    private final XMLFloat _minStage = new XMLFloat("minStage");
    private final XMLFloat _maxStage = new XMLFloat("maxStage");
    private final List<RatingCurveRow> _rows = new ArrayList<>();

    public String getInterpolationMethod()
    {
        return _interpolationMethod.get();
    }

    public void setInterpolationMethod(final String method)
    {
        _interpolationMethod.set(method);
    }

    public Float getMinStage()
    {
        return _minStage.get();
    }

    public void setMinStage(final Float stage)
    {
        _minStage.set(stage);
    }

    public Float getMaxStage()
    {
        return _maxStage.get();
    }

    public void setMaxStage(final Float stage)
    {
        _maxStage.set(stage);
    }

    public List<RatingCurveRow> getRows()
    {
        return _rows;
    }

    @Override
    public XMLWriter getWriter()
    {
        final List<XMLWritable> elements = Lists.newArrayList((XMLWritable)_interpolationMethod, _minStage, _maxStage);
        elements.addAll(_rows);
        return new CompositeXMLWriter("table", elements);
    }

    @Override
    public XMLReader getReader()
    {
        return new CompositeXMLReader("table", _interpolationMethod, _minStage, _maxStage, new RatingCurveRow("row"));
    }

    /**
     * A {@link RatingCurveRow} specifies a stage, discharge, and log scale stage offset as attributes. The element is
     * an {@link XMLString} with an empty unchanging value: "".
     * 
     * @author hankherr
     */
    public class RatingCurveRow extends XMLString
    {
        public XMLFloat _stage = new XMLFloat("stage");
        public XMLFloat _discharge = new XMLFloat("discharge");
        public XMLDouble _logScaleStageOffset = new XMLDouble("logScaleStageOffset");

        public RatingCurveRow(final String tag)
        {
            setXMLTagName(tag);
            set(""); //Necessary 
            addAttribute(_stage, true);
            addAttribute(_discharge, true);
            addAttribute(_logScaleStageOffset, true);
        }

        private RatingCurveRow(final RatingCurveRow base)
        {
            this(base.getXMLTagName());
            _stage.set(base.getStage());
            _discharge.set(base.getDischarge());
            _logScaleStageOffset.set(base.getLogScaleStageOffset());
        }

        public Float getStage()
        {
            return _stage.get();
        }

        public void setStage(final Float stage)
        {
            _stage.set(stage);
        }

        public Float getDischarge()
        {
            return _discharge.get();
        }

        public void setDischarge(final Float discharge)
        {
            _discharge.set(discharge);
        }

        public Double getLogScaleStageOffset()
        {
            return _logScaleStageOffset.get();
        }

        public void setLogScaleStageOffset(final Double logScaleStageOffset)
        {
            _logScaleStageOffset.set(logScaleStageOffset);
        }

        @Override
        public void finalizeReading() throws XMLReaderException
        {
            _rows.add(new RatingCurveRow(this));
        }

    }

}
