package ohd.hseb.hefs.utils.rc;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import ohd.hseb.hefs.utils.xml.CollectionXMLReader;
import ohd.hseb.hefs.utils.xml.CollectionXMLWriter;
import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.GenericXMLReadingHandlerException;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLFloat;
import ohd.hseb.hefs.utils.xml.vars.XMLTimeZone;

import com.google.common.collect.Lists;

/**
 * Stores {@link HRatingCurve} instances in an {@link ArrayList} that also implements {@link XMLReadable} and
 * {@link XMLWritable} for use with {@link XMLTools} methods. See the static methods for reading and writing:
 * {@link #readRatingCurves(File)} and {@link #writeRatingCurves(File, Collection)}.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class HRatingCurves extends ArrayList<HRatingCurve> implements XMLReadable, XMLWritable
{

    /**
     * Time zones are specified as a shift from GMT within the Deltares XML. Hence, an {@link XMLFloat} is used instead
     * of {@link XMLTimeZone} which allows for specifying a time zone identifying string.
     */
    private final XMLFloat _timeZone = new XMLFloat("timeZone");

    @Override
    public XMLWriter getWriter()
    {
        final List<XMLWritable> writables = Lists.newArrayList((XMLWritable)_timeZone);
        writables.addAll(this);
        return new CollectionXMLWriter("RatingCurves", writables);
    }

    @Override
    public XMLReader getReader()
    {
        return new CompositeXMLReader("RatingCurves", _timeZone, new HRatingCurve()
        {
            @Override
            protected void finalizeReading()
            {
                add(this.moveRatingCurveInfoIntoNewInstance());
            }
        });
    }

    /**
     * @return A {@link HRatingCurves} object popuplated from the specified file.
     */
    public static HRatingCurves readRatingCurves(final File file) throws GenericXMLReadingHandlerException
    {
        final HRatingCurves curves = new HRatingCurves();

        final CollectionXMLReader xmlReader = new CollectionXMLReader<HRatingCurve>("RatingCurves",
                                                                                   curves,
                                                                                   new XMLReaderFactory<HRatingCurve>()
                                                                                   {

                                                                                       @Override
                                                                                       public HRatingCurve get()
                                                                                       {
                                                                                           return new HRatingCurve();
                                                                                       }
                                                                                   });

        XMLTools.readXMLFromFile(file, xmlReader);

        return curves;
    }

    /**
     * Writes any {@link Collection} of {@link HRatingCurve} objects to a rating curve XML file, making use of
     * {@link CollectionXMLWriter}.
     */
    public static void writeRatingCurves(final File file, final Collection<HRatingCurve> ratingCurves) throws Exception
    {
        final CollectionXMLWriter xmlWriter = new CollectionXMLWriter("RatingCurves", ratingCurves);
        XMLTools.writeXMLFileFromXMLWriter(file, xmlWriter, true);
    }
}
