package ohd.hseb.hefs.utils.status;

import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.SwingConstants;

import ohd.hseb.hefs.utils.gui.jtable.buttons.SelectUnreadyRowsButton;

/**
 * Label used to display a {@link StatusIndicator} icon.
 * 
 * @author hankherr
 */
@SuppressWarnings("serial")
public class StatusLabel extends JLabel implements StatusIndicator
{
    private final StatusIndicator _status;
    private String _descriptionOverride = null;
    private boolean _includeInRowReadyCheck = true;

    private StatusLabel(final StatusIndicator status)
    {
        _status = status;
        final Icon icon = _status.getIcon();
        this.setIcon(icon);
        this.setToolTipText(_status.getDescription());
        this.setHorizontalAlignment(SwingConstants.CENTER);
        this.setVerticalAlignment(SwingConstants.CENTER);
    }

    /**
     * Create a {@link StatusLabel} to show {@code status}. If {@code status} is already a {@link StatusLabel}, just
     * return it instead.
     * 
     * @param status the status to display in the label
     * @return a status label with the status {@code status}
     */
    public static StatusLabel make(final StatusIndicator status)
    {
        if(status instanceof StatusLabel)
        {
            return (StatusLabel)status;
        }
        else
        {
            return new StatusLabel(status);
        }
    }

    /**
     * Create a {@link StatusLabel} to show {@code status}, with the description {@code description}. If {@code status}
     * is already a {@link StatusLabel}, just return it instead.
     * 
     * @param status the status to display in the label
     * @param description the status description to use
     * @return a status label with the status {@code status}
     */
    public static StatusLabel make(final StatusIndicator status, final String description)
    {
        return make(status).setDescription(description);
    }

    /**
     * @param b The status.
     * @return A {@link StatusLabel} that wraps a {@link BooleanStatus}.
     */
    public static StatusLabel make(final Boolean b)
    {
        return new StatusLabel(BooleanStatus.get(b));
    }

    /**
     * If the {@link #_includeInRowReadyCheck} value is to be true, do not call this version: it defaults to true.
     * 
     * @param b The status.
     * @param includeInRowReadyCheck Passed to {@link #setIncludeInRowReadyCheck(boolean)}.
     * @return A {@link StatusLabel} that wraps a {@link BooleanStatus}.
     */
    public static StatusLabel make(final Boolean b, final boolean includeInRowReadyCheck)
    {
        return new StatusLabel(BooleanStatus.get(b)).setIncludeInRowReadyCheck(includeInRowReadyCheck);
    }

    /**
     * @param b The status.
     * @param description The description for the status.
     * @return A {@link StatusLabel} that wraps a {@link BooleanStatus}.
     */
    public static StatusLabel make(final Boolean b, final String description)
    {
        return make(b).setDescription(description);
    }

    /**
     * If the {@link #_includeInRowReadyCheck} value is to be true, do not call this version: it defaults to true.
     * 
     * @param b The status.
     * @param description The description for the status.
     * @param includeInRowReadyCheck Passed to {@link #setIncludeInRowReadyCheck(boolean)}.
     * @return A {@link StatusLabel} that wraps a {@link BooleanStatus}.
     */
    public static StatusLabel make(final Boolean b, final String description, final boolean includeInRowReadyCheck)
    {
        return make(b).setDescription(description).setIncludeInRowReadyCheck(includeInRowReadyCheck);
    }

    /**
     * Sets a flag indicating if the StatusLabel status should be used to determine if a row is ready or not. This is
     * used by the {@link SelectUnreadyRowsButton} which may be attached to the table.
     * 
     * @param b True to use this StatusLabel in the check row readiness.
     */
    public StatusLabel setIncludeInRowReadyCheck(final boolean b)
    {
        _includeInRowReadyCheck = b;
        return this;
    }

    @Override
    public boolean canBePrepared()
    {
        return _status.canBePrepared();
    }

    @Override
    public boolean isReady()
    {
        return _status.isReady();
    }

    @Override
    public boolean exists()
    {
        return _status.exists();
    }

    @Override
    public Boolean toBoolean()
    {
        return _status.toBoolean();
    }

    @Override
    public String getDescription()
    {
        return _descriptionOverride == null ? _status.getDescription() : _descriptionOverride;
    }

    /**
     * Sets the description. If null, use the underlying status's description instead.
     * 
     * @param description the description to use
     */
    public StatusLabel setDescription(final String description)
    {
        _descriptionOverride = description;
        return this;
    }

    public StatusIndicator getStatus()
    {
        return _status;
    }

    @Override
    public String getToolTipText()
    {
        return getDescription();
    }

    @Override
    public boolean includeInRowReadinessCheck()
    {
        return _includeInRowReadyCheck;
    }

}
