package ohd.hseb.hefs.utils.tools;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import ohd.hseb.hefs.utils.Dyad;

import com.google.common.collect.Lists;

public abstract class ArrayTools
{

    /**
     * Concatenate two arrays together.
     */
    public static <T> T[] concat(final T[] firstArray, final T[] secondArray)
    {
        @SuppressWarnings("unchecked")
        final T[] result = (T[])Array.newInstance(firstArray.getClass().getComponentType(), firstArray.length
            + secondArray.length);
        for(int i = 0; i < firstArray.length; i++)
        {
            result[i] = firstArray[i];
        }
        for(int i = 0; i < secondArray.length; i++)
        {
            result[i + firstArray.length] = secondArray[i];
        }
        return result;
    }

    /**
     * Creates a new array of type {@code T} which is a copy of {@code array} with {@code element} prepended.
     * 
     * @param <T> the type of the array
     * @param element the additional element to prepend
     * @param array the base array to copy
     * @return the new array
     */
    public static <T> T[] concat(final T element, final T[] array)
    {
        @SuppressWarnings("unchecked")
        final T[] result = (T[])Array.newInstance(array.getClass().getComponentType(), array.length + 1);
        result[0] = element;
        for(int i = 0; i < array.length; i++)
        {
            result[i + 1] = array[i];
        }
        return result;
    }

    /**
     * Creates a new array of type {@code T} which is a copy of {@code array} with {@code element} appended.
     * 
     * @param <T> the type of the array
     * @param array the base array to copy
     * @param element the additional element to append
     * @return the new array
     */
    public static <T> T[] append(final T[] array, final T element)
    {
        final T[] result = Arrays.copyOf(array, array.length + 1);
        result[array.length] = element;
        return result;
    }

    /**
     * Creates a new array composed of the values of {@code array} at each of the {@code indices}.
     * 
     * @param array the array to pull values from
     * @param indices each index to pull from the array
     * @return a new array
     */
    public static <T> T[] select(final T[] array, final int... indices)
    {
        @SuppressWarnings("unchecked")
        final T[] result = (T[])Array.newInstance(array.getClass().getComponentType(), indices.length);
        for(int i = 0; i < indices.length; i++)
        {
            result[i] = array[indices[i]];
        }
        return result;
    }

    /**
     * Sorts both arrays in place and together. The values are sorted by the first array values in ascending order.
     * 
     * @param array1 The first array; contains the values by which the two arrays are sorted.
     * @param array2 The second arrays, which is reordered relative to the sorted order of the first array.
     */
    public static <T extends Comparable> void sortArraysInTandem(final T[] array1, final T[] array2)
    {
        final List<Dyad<T, T>> dyads = Lists.newArrayList();

        for(int i = 0; i < array1.length; i++)
        {
            dyads.add(new Dyad<T, T>(array1[i], array2[i]));
        }

        Collections.sort(dyads, new Comparator<Dyad<T, T>>()
        {
            @SuppressWarnings("unchecked")
            @Override
            public int compare(final Dyad<T, T> o1, final Dyad<T, T> o2)
            {
                return o1.getFirst().compareTo(o2.getFirst());
            }
        });

        for(int i = 0; i < dyads.size(); i++)
        {
            array1[i] = dyads.get(i).getFirst();
            array2[i] = dyads.get(i).getSecond();
        }
    }

    /**
     * See {@link #sortArraysInTandem(Comparable[], Comparable[])}. This is the primitive double version of this, since
     * it may not be convenient to use {@link Double} all the time.
     */
    public static void sortArraysInTandem(final double[] array1, final double[] array2)
    {
        final List<Dyad<Double, Double>> dyads = Lists.newArrayList();

        for(int i = 0; i < array1.length; i++)
        {
            dyads.add(new Dyad<Double, Double>(array1[i], array2[i]));
        }

        Collections.sort(dyads, new Comparator<Dyad<Double, Double>>()
        {
            @Override
            public int compare(final Dyad<Double, Double> o1, final Dyad<Double, Double> o2)
            {
                return o1.getFirst().compareTo(o2.getFirst());
            }
        });

        for(int i = 0; i < dyads.size(); i++)
        {
            array1[i] = dyads.get(i).getFirst();
            array2[i] = dyads.get(i).getSecond();
        }
    }
}
