package ohd.hseb.hefs.utils.tools;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Enumeration;
import java.util.List;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import org.apache.commons.io.IOUtils;

import com.google.common.base.Function;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.io.Files;

import ohd.hseb.hefs.utils.AbstractFunction;
import ohd.hseb.hefs.utils.ZipEntryInputSupplier;

public abstract class CompressTools
{
    /**
     * Create a {@link File} with the same path as the given {@link ZipEntry}.
     */
    public static Function<ZipEntry, File> ENTRY_TO_FILE =
                                                         new Function<ZipEntry, File>()
                                                         {
                                                             @Override
                                                             public File apply(final ZipEntry input)
                                                             {
                                                                 return new File(input.getName());
                                                             }
                                                         };

    /**
     * Create a {@link File} with the same path as the given {@link ZipEntry}, but absolute.
     */
    public static Function<ZipEntry, File> ENTRY_TO_ABSOLUTE_FILE =
                                                                  new Function<ZipEntry, File>()
                                                                  {
                                                                      @Override
                                                                      public File apply(final ZipEntry input)
                                                                      {
                                                                          return new File(input.getName()).getAbsoluteFile();
                                                                      }
                                                                  };

    /**
     * @param inFile The file to gzip. This file will not be removed by this process.
     * @return The name of the file created, which will be the name of the input file (including the path) with ".gz"
     *         appended.
     * @throws Exception Typically I/O related when opening one of the streams or copying from the input to output.
     */
    public static File compressIntoGZIPFile(final File inFile) throws Exception
    {
        final String outFileName = inFile.getAbsolutePath() + ".gz";

        FileInputStream inStream = null;
        GZIPOutputStream outStream = null;
        try
        {
            inStream = new FileInputStream(inFile);
            outStream = new GZIPOutputStream(new FileOutputStream(outFileName));
            //StreamUtils.copy(inStream, outStream);
            IOUtils.copy(inStream, outStream);
            
        }
        finally
        {        
            StreamTools.closeStream(inStream);
            StreamTools.closeStream(outStream);
        }

        return new File(outFileName);
    }

    /**
     * Assumes the given file is a gzip file and gunzips it. The output file will be the same as the input without .gz
     * or with a .txt if no .gunzip exists.
     * 
     * @param gzipFile
     * @return The output file.
     * @throws Exception
     */
    public static File uncompressGZIPFile(final File gzipFile) throws Exception
    {
        String outFileName = "";
        if(gzipFile.getAbsolutePath().endsWith(".gz"))
        {
            outFileName = gzipFile.getAbsolutePath().substring(0,
                                                               gzipFile.getAbsolutePath()
                                                                       .length()
                                                                   - 3);
        }
        else
        {
            outFileName = gzipFile.getAbsolutePath() + ".gunzip";
        }

        GZIPInputStream inStream = null;
        FileOutputStream outStream = null;
        try
        {
            inStream = new GZIPInputStream(new FileInputStream(gzipFile));
            outStream = new FileOutputStream(new File(outFileName));
            //StreamUtils.copy(inStream, outStream);
            IOUtils.copy(inStream, outStream);
        }
        finally
        {
            StreamTools.closeStream(inStream);
            StreamTools.closeStream(outStream);
        }

        return new File(outFileName);
    }

    public static ZipOutputStream makeZipOutStream(final File outFile) throws FileNotFoundException
    {
        return new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(outFile)));
    }

    /**
     * Gets {@link ZipEntry ZipEntries} as a list, instead of the outdated {@link Enumeration}.
     * 
     * @param file the zip file to get the entries of
     * @return all entries in <code>file</code>
     */
    public static List<ZipEntry> getZipEntriesList(final ZipFile file)
    {
        return Lists.newArrayList(Iterators.forEnumeration(file.entries()));
    }

    private static final Function<ZipEntry, String> GET_ENTRY_NAME =
                                                                   new AbstractFunction<ZipEntry, String>()
                                                                   {
                                                                       @Override
                                                                       public String apply(final ZipEntry input)
                                                                       {
                                                                           return input.getName();
                                                                       }
                                                                   };

    /**
     * Gets the path of every file in a zip file.
     * 
     * @param file the zip file to read
     * @return every path contained in <code>file</code>
     */
    public static List<String> getZipPaths(final ZipFile file)
    {
        return ListTools.mapToArrayList(GET_ENTRY_NAME,
                                        getZipEntriesList(file));
    }

    public static void extractFile(final File zipFile,
                                   final File readFile,
                                   final File writeFile) throws IOException
    {
        System.out.println("EXTRACT FILE:");
        System.out.println("  " + readFile);
        System.out.println("  " + writeFile);
        Files.createParentDirs(writeFile);
        Files.copy(new ZipEntryInputSupplier(zipFile, readFile.getPath()),
                   writeFile);
    }

    public static void extractDirectory(final File zipFile,
                                        final File readDir,
                                        final File writeDir) throws IOException
    {
        final String baseDir = readDir.getPath();

        System.out.println("EXTRACT DIRECTORY:");
        System.out.println("-| " + zipFile);
        System.out.println("-| " + readDir);
        System.out.println("-| " + writeDir);
        System.out.println("-| " + baseDir);

        final ZipFile zip = new ZipFile(zipFile);
        final Enumeration<? extends ZipEntry> entries = zip.entries();

        while(entries.hasMoreElements())
        {
            final ZipEntry entry = entries.nextElement();

            if(entry.getName().startsWith(baseDir))
            {
                System.out.println("+++" + entry.getName());
                final File readFile = new File(entry.getName());
                final File writeFile =
                                     new File(writeDir,
                                              FileTools.getRelativeFile(baseDir,
                                                                        entry.getName()));
                extractFile(zipFile, readFile, writeFile);
            }
        }
        zip.close();
    }

    /**
     * @param gzipFile1 First file.
     * @param gzipFile2 Second file.
     * @return True if the contents of the two files are identical. The comparison is made via a {@link GZIPInputStream}
     *         passed to {@link IOUtils#contentEquals(java.io.InputStream, java.io.InputStream)}.
     * @throws IOException Standard reasons.
     */
    public static boolean areGZIPFileContentsTheSame(final File gzipFile1,
                                                     final File gzipFile2) throws IOException
    {
        GZIPInputStream gzip1 = null;
        GZIPInputStream gzip2 = null;

        try
        {
            final FileInputStream stream1 = new FileInputStream(gzipFile1);
            gzip1 = new GZIPInputStream(stream1);
            final FileInputStream stream2 = new FileInputStream(gzipFile2);
            gzip2 = new GZIPInputStream(stream2);
            return IOUtils.contentEquals(gzip1, gzip2);
        }
        finally
        {
            if(gzip1 != null)
            {
                gzip1.close();
            }
            if(gzip2 != null)
            {
                gzip2.close();
            }
        }
    }
}
