package ohd.hseb.hefs.utils.tools;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.URI;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.LineIterator;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.io.Files;

public abstract class FileTools
{
    /**
     * Returns true if the passed file exists.
     */
    public static Predicate<File> DOES_FILE_EXIST = new Predicate<File>()
    {
        @Override
        public boolean apply(final File input)
        {
            return input.exists();
        }
    };

    /**
     * @param baseDirectory Base directory.
     * @param file File beneath the directory.
     * @return Name of the file relative to the base directory. If the file is not beneath the base directory, the name
     *         will start with a '/', denoting a complete path name.
     */
    public static String determineFileNameRelativeToBaseDirectory(final File baseDirectory, final File file)
    {
        final String absPath = file.getAbsolutePath();
        final String relPath = baseDirectory.getAbsolutePath();
        if(absPath.startsWith(relPath))
        {
            final String result = absPath.substring(relPath.length());
            if(result.startsWith(File.separator))
            {
                return result.substring(1);
            }
            return result;
        }
        return file.getAbsolutePath();
    }

    /**
     * Wraps a call to {@link FileUtils#forceDelete(File)} inside a check to see if the file exists. If the file does
     * not exist and the force delete method is called, an exception is thrown, which can be an annoyance sometimes.
     * 
     * @param file The file to remove if the file exists.
     * @throws IOException Only if deletion cannot be done possibly due to permissions or other IO issues.
     */
    public static void deleteFileIfItExists(final File file) throws IOException
    {
        if(file.exists())
        {
            FileUtils.forceDelete(file);
        }
    }

    /**
     * @param files {@link List} of {@link File}s to delete.
     * @throws IOException
     */
    public static void deleteFiles(final List<File> files) throws IOException
    {
        for(final File file: files)
        {
            FileUtils.forceDelete(file);
        }
    }

    /**
     * Removes any file that can be removed and matches the filter.
     * 
     * @param dir Directory containing files.
     * @param filter FilenameFilter to use.
     */
    public static void deleteFiles(final File dir, final FilenameFilter filter) throws IOException
    {
        if(!dir.isDirectory())
        {
            throw new IOException("File " + dir.getAbsolutePath() + " is not a directory.");
        }
        final String[] fileNames = dir.list(filter);
        for(final String fileName: fileNames)
        {
            final File toRemove = new File(dir.getAbsolutePath() + File.separator + fileName);
            FileUtils.forceDelete(toRemove);
        }
    }

    /**
     * Calls the other version, building a FilenameFilter.
     * 
     * @param dir Directory containing files.
     * @param fileExtensionToRemove Extension of files to remove.
     * @throws IOException
     */
    public static void deleteFiles(final File dir, final String fileExtensionToRemove) throws IOException
    {
        FileTools.deleteFiles(dir, new FilenameFilter()
        {
            @Override
            public boolean accept(final File dir, final String name)
            {
                if(fileExtensionToRemove.isEmpty())
                {
                    return true;
                }
                return name.endsWith(fileExtensionToRemove);
            }
        });
    }

    /**
     * Deletes all files and directories under {@code directory}, but not the directory itself.
     * 
     * @param directory the directory to delete all files under
     */
    public static void deleteFilesIn(final File directory) throws IOException
    {
        Preconditions.checkArgument(directory.isDirectory(), "%s is not a directory", directory);
        for(final File file: directory.listFiles())
        {
            FileUtils.forceDelete(file);
        }
    }

    /**
     * Deletes all files and directories under {@code directory}, except those that start with a '.'.
     * 
     * @param directory the directory to delete all files under
     */
    public static void deleteVisibleFilesIn(final File directory) throws IOException
    {
        Preconditions.checkArgument(directory.isDirectory(), "%s is not a directory", directory);
        for(final File file: directory.listFiles())
        {
            if(!file.getName().startsWith("."))
            {
                FileUtils.forceDelete(file);
            }
        }
    }

    /**
     * @return A {@link Predicate} that uses the provided filter.
     */
    public static Predicate<File> makePredicate(final FileFilter filter)
    {
        return new Predicate<File>()
        {
            @Override
            public boolean apply(final File input)
            {
                return filter.accept(input);
            }
        };
    }

    /**
     * @return A {@link Predicate} that uses the provided filter.
     */
    public static Predicate<File> makePredicate(final FilenameFilter filter)
    {
        return new Predicate<File>()
        {
            @Override
            public boolean apply(final File input)
            {
                return filter.accept(input.getParentFile(), input.getName());
            }
        };
    }

    /**
     * @param extensions Acceptable extensions.
     * @return A {@link FileFilter} that looks for files with the provided extensions. The extension is the part of the
     *         file name after (not including) the last '.' in the name.
     */
    public static FileFilter makeExtensionFilter(final String... extensions)
    {
        return makeExtensionFilter(Arrays.asList(extensions));
    }

    /**
     * @param extensions Acceptable extensions.
     * @return A {@link FileFilter} that looks for files with the provided extensions. The extension is the part of the
     *         file name after (not including) the last '.' in the name.
     */
    public static FileFilter makeExtensionFilter(final Collection<String> extensions)
    {
        final Set<String> extSet = ImmutableSet.copyOf(extensions);
        return new FileFilter()
        {
            @Override
            public boolean accept(final File pathname)
            {
                final String name = pathname.getName();
                final String extension = name.substring(name.lastIndexOf('.') + 1);
                return extSet.contains(extension);
            }
        };
    }

    /**
     * @param suffix The allowable suffix for the file name.
     * @return A {@link FileFilter} that looks for files whose name ends with the provided suffix. The check performed
     *         is a straight {@link String#endsWith(String)} on the absolute pathname of the file.
     */
    public static FileFilter makeSuffixFilter(final String suffix)
    {
        return new FileFilter()
        {
            @Override
            public boolean accept(final File pathname)
            {
                return pathname.getAbsolutePath().endsWith(suffix);
            }
        };
    }

    /**
     * @param prefix The allowable prefix for the file name.
     * @return A {@link FileFilter} that looks for files whose name starts with the provided suffix. The check performed
     *         is a straight {@link String#startsWith(String)} on the absolute pathname of the file.
     */
    public static FileFilter makePrefixFilter(final String prefix)
    {
        return new FileFilter()
        {
            @Override
            public boolean accept(final File pathname)
            {
                return pathname.getName().startsWith(prefix);
            }
        };
    }

    /**
     * @return Returns a {@link Set} of files based on the return of {@link File#listFiles()} for the provided base
     *         directory.
     */
    public static Set<File> listFiles(final File base)
    {
        return Sets.newHashSet(base.listFiles());
    }

    /**
     * @return A {@link Set} of files based on the return of {@link File#listFiles(FileFilter)} for the provided base
     *         directory and filter.
     */
    public static Set<File> listFiles(final File base, final FileFilter filter)
    {
        return Sets.newHashSet(base.listFiles(filter));
    }

    /**
     * @return A {@link Set} of files based on the return of {@link File#listFiles(FilenameFilter)} for the provided
     *         base directory and filter.
     */
    public static Set<File> listFiles(final File base, final FilenameFilter filter)
    {
        return Sets.newHashSet(base.listFiles(filter));
    }

    /**
     * @return A {@link Set} of files with names that end with the provided suffix. The filter used is
     *         {@link #makeSuffixFilter(String)}.
     */
    public static Set<File> listFilesWithSuffix(final File base, final String fileNameSuffix)
    {
        return Sets.newHashSet(base.listFiles(makeSuffixFilter(fileNameSuffix)));
    }

    /**
     * @return A {@link Set} of files with names that start with the provided suffix. The filter used is
     *         {@link #makePrefixFilter(String)}.
     */
    public static Set<File> listFilesWithPrefix(final File base, final String fileNamePrefix)
    {
        return Sets.newHashSet(base.listFiles(makePrefixFilter(fileNamePrefix)));
    }

    /**
     * @param base Base directory
     * @return List of all files under the base directory. List will not include directories.
     */
    public static Set<File> listFilesRecursively(final File base)
    {
        final Set<File> set = new HashSet<File>();
        for(final File f: base.listFiles())
        {
            if(f.isDirectory())
            {
                set.addAll(listFilesRecursively(f));
            }
            else
            {
                set.add(f);
            }
        }
        return set;
    }

    /**
     * @return A {@link Set} of {@link File} instances specifying all files underneath of base that satisfy the method
     *         {@link FileFilter#accept(File)}.
     */
    public static Set<File> listFilesRecursively(final File base, final FileFilter filter)
    {
        return Sets.newHashSet(Sets.filter(listFilesRecursively(base), makePredicate(filter)));
    }

    /**
     * @return A {@link Set} of {@link File} instances specifying all files underneath of base that satisfy the method
     *         {@link FilenameFilter#accept(File, String)}.
     */
    public static Set<File> listFilesRecursively(final File base, final FilenameFilter filter)
    {
        return Sets.newHashSet(Sets.filter(listFilesRecursively(base), makePredicate(filter)));
    }

    /**
     * @return A {@link Set} of files with names that end with the provided suffix. The filter used is
     *         {@link #makeSuffixFilter(String)} and searching is done recursively starting with base.
     */
    public static Set<File> listFilesRecursivelyWithSuffix(final File base, final String fileNameSuffix)
    {
        return listFilesRecursively(base, makeSuffixFilter(fileNameSuffix));
    }

    /**
     * @return A {@link Set} of files with names that end with the provided suffix. The filter used is
     *         {@link #makePrefixFilter(String)} and searching is done recursively starting with base.
     */
    public static Set<File> listFilesRecursivelyWithPrefix(final File base, final String fileNamePrefix)
    {
        return listFilesRecursively(base, makePrefixFilter(fileNamePrefix));
    }

    /**
     * Removes all files under a directory, leaving the directory structure intact.
     * 
     * @param baseDir
     * @throws IOException
     */
    public static void deleteFilesRecursively(final File baseDir) throws IOException
    {
        if(!baseDir.exists())
        {
            return;
        }
        for(final File file: listFilesRecursively(baseDir))
        {
            FileUtils.forceDelete(file);
        }
    }

    /**
     * Creates the parent directory, {@link File#getParentFile()}, if it does not exist. Calls
     * {@link #mkdirIfItDoesNotExist(File)}.
     * 
     * @param file The file whose parent must exist.
     * @throws IOException
     */
    public static void ensureParentDirectoryExists(final File file) throws IOException
    {
        final File parent = file.getParentFile();
        if(parent != null)
        {
            mkdirIfItDoesNotExist(parent);
        }
    }

    /**
     * Creates the provided directory if it does not already exist.
     * 
     * @param dir Directory to create.
     * @throws IOException
     */
    public static void mkdirIfItDoesNotExist(final File dir) throws IOException
    {
        if(!dir.exists())
        {
            try
            {
                FileUtils.forceMkdir(dir);
            }
            catch(final Exception e)
            {
                throw new IOException("Unable to create the directory " + dir.getAbsolutePath() + ".");
            }
        }
        else if(!dir.isDirectory())
        {
            throw new IOException("File exists by the name " + dir.getAbsolutePath() + " and is not a directory.");
        }
    }

    /**
     * The parent directory must exist prior to calling this.
     * 
     * @param fromFile
     * @param toFile
     * @throws IOException
     */
    public static void renameFileReplaceExisting(final File fromFile, final File toFile) throws IOException
    {
        if(toFile.exists())
        {
            if(!toFile.delete())
            {
                throw new IOException("Failed to delete file " + toFile.getAbsolutePath());
            }
        }
        Files.move(fromFile, toFile);
    }

    /**
     * If {@code fromFile} and {@code toFile} exists and {@code swapFiles} is true, then the two files are swapped. If
     * both exist but swapFiles is false, then the fromFile is renamed and replaces toFile. If fromFile exists but not
     * toFile, then fromFile is renamed to toFile and the parent is created if needed. If toFile exists but not fromFile
     * and swapFiles is true, then toFile is renamed to fromFile and the parent is created if needed. If fromFile does
     * not exist and swapFiles is false, nothing happens.
     * 
     * @param fromFile The first file to move or swap.
     * @param toFile The second file to swap or target file if moving.
     * @param swapFiles True to swap files, false to move fromFile to toFile.
     * @throws IOException If any move fails for any reason. Note that a {@link FileNotFoundException} is never thrown.
     */
    public static void moveOrSwapFiles(final File fromFile, final File toFile, final boolean swapFiles) throws IOException
    {
        if(fromFile.exists())
        {
            Files.createParentDirs(toFile);
            if((!swapFiles) || (!toFile.exists()))
            {
                FileTools.renameFileReplaceExisting(fromFile, toFile);
            }
            else
            {
                final File tmpFile = new File(toFile.getAbsolutePath() + "." + Thread.currentThread().getName());
                FileTools.renameFileReplaceExisting(toFile, tmpFile);
                FileTools.renameFileReplaceExisting(fromFile, toFile);
                FileTools.renameFileReplaceExisting(tmpFile, fromFile);
            }
        }
        else if(toFile.exists() && swapFiles)
        {
            Files.createParentDirs(fromFile);
            FileTools.renameFileReplaceExisting(toFile, fromFile);
        }
    }

    /**
     * Wraps {@link #moveOrSwapFiles(File, File, boolean) moveOrSwapFiles} passing in false. Moves {@code fromFile} to
     * {@code toFile}, creating the parent of {@code toFile} if needed. Any existing file will be replaced. If fromFile
     * does not exist, nothing is done and no exception is thrown.
     * 
     * @param fromFile File to move.
     * @param toFile New file name.
     * @throws IOException If the move fails for any reason. Note that a {@link FileNotFoundException} is never thrown.
     */
    public static void moveFile(final File fromFile, final File toFile) throws IOException
    {
        moveOrSwapFiles(fromFile, toFile, false);
    }

    /**
     * Wraps {@link #moveOrSwapFiles(File, File, boolean) moveOrSwapFiles} passing in true. Swaps {@code file0} and
     * {@code file1}. If either file does not exist, this operates the same as moveFile.
     * 
     * @param file0 the first file to swap
     * @param file1 the second file to swap
     * @throws IOException If the swap fails for any reason. Note that a {@link FileNotFoundException} is never thrown.
     */
    public static void swapFiles(final File file0, final File file1) throws IOException
    {
        moveOrSwapFiles(file0, file1, true);
    }

    /**
     * Creates a {@link File} by applying {@link #toString()} to each of the {@code components}.<br>
     * 
     * @param components the file components
     * @return the specified file
     * @throws NullPointerException if any component but the first is null
     */
    public static File newFile(final Object... components) throws NullPointerException
    {
        Preconditions.checkElementIndex(0, components.length, "Must supply at least 1 path component.");

        if(components.length == 1)
        {
            return new File(components[0].toString());
        }

        File file = new File(StringTools.nullOrStringValue(components[0]), StringTools.nullOrStringValue(components[1]));
        for(int i = 2; i < components.length; i++)
        {
            file = new File(file, StringTools.nullOrStringValue(components[i]));
        }
        return file;
    }

    /**
     * Creates a {@link File} by applying {@link #toString()} to each of the {@code components}.<br>
     * 
     * @param components the file components
     * @return the specified file, in absolute format
     * @throws NullPointerException if any component but the first is null
     */
    public static File newAbsoluteFile(final Object... components) throws NullPointerException
    {
        return newFile(components).getAbsoluteFile();
    }

    /**
     * Gets the path of {@code file} with a trailing file separator
     * 
     * @param file
     * @return
     */
    public static String getDirPath(final File file)
    {
        return file.getPath() + File.separator;
    }

    /**
     * Returns the first {@link File} for which {@link File#exists() exists()} returns {@code true}.
     */
    public static File firstExisting(final File... files)
    {
        for(int i = 0; i < files.length; i++)
        {
            if(files[i].exists())
            {
                return files[i];
            }
        }
        return null;
    }

    /**
     * Basically, it gets the component of {@code file} which is descended from {@code directory}.<br/>
     * More specifically, it is as {@link java.net.URI#relativize(java.net.URI)}, but with {@link File}s directly
     * instead of {@link URI}s.
     * 
     * @param directory the base directory
     * @param file the file to get the relative path of
     * @return the original file relative to the given directory
     */
    public static File getRelativeFile(final File directory, final File file)
    {
        return new File(directory.toURI().relativize(file.toURI()).toString());
    }

    /**
     * @see FileTools#getRelativeFile(File, File).
     */
    public static String getRelativeFile(final String directory, final String file)
    {
        return new File(directory).toURI().relativize(new File(file).toURI()).toString();
    }

    /**
     * Partial Application of {@link FileTools#constructFileWithNewAncestor(File, File, File)}.
     * 
     * @return A {@link Function} that can be used to convert one file to another, exchanging the directory oldParentfor
     *         the directory newParent in its name.
     */
    public static Function<File, File> constructFileWithNewAncestor(final File oldParent, final File newParent)
    {
        return new Function<File, File>()
        {
            @Override
            public File apply(final File input)
            {
                return newFile(newParent, getRelativeFile(oldParent, input));
            }
        };
    }

    /**
     * @param oldParent an ancestor of {@code file}
     * @param newParent the path to replace {@code oldParent} with
     * @param file the base file
     * @return A new {@link File} which is identical to the provided file, but with the oldParent portion of its name
     *         replaced with newParent.
     */
    public static File constructFileWithNewAncestor(final File oldParent, final File newParent, final File file)
    {
        return constructFileWithNewAncestor(oldParent, newParent).apply(file);
    }

    /**
     * @return The name of a file excluding its extension and the '.' before the extension.
     */
    public static String getBaseName(final File file)
    {
        final String name = file.getName();
        return name.substring(0, name.lastIndexOf('.'));
    }

    /**
     * @return The extension of a file, which is the part of the name after (not including) the last '.'.
     */
    public static String getExtension(final File file)
    {
        final String name = file.getName();
        return name.substring(name.lastIndexOf('.') + 1);
    }

    /**
     * @return A new {@link File} with the same name as the file given, but with an extension equal to that provided,
     *         replacing any existing extension. If there is no existing extension, then the new extension is added to
     *         the file after a '.' is added.
     */
    public static File replaceExtension(final File file, final String newExtension)
    {
        final int index = file.getName().lastIndexOf('.');
        if(index < 0)
        {
            return new File(file.getAbsolutePath() + "." + newExtension);
        }
        else
        {
            return FileTools.newFile(file.getParent(), file.getName().substring(0, index) + "." + newExtension);
        }
    }

    /**
     * @param file The {@link File} whose name may or may not contain a valid extension.
     * @param defaultExtension The default extension to set if the current filename's extension is not valid.
     * @param allowableExtensions Array of valid extensions (not including any '.').
     * @return A new {@link File} instance that contains a valid, allowable extension. Note that the default extension
     *         will be added to the existing file name if the current extension is not recognized; the current extension
     *         will not be removed.
     */
    public static File ensureValidExtension(final File file,
                                            final String defaultExtension,
                                            final String... allowableExtensions)
    {
        String fileName = file.getAbsolutePath();
        final int indexOfLastDot = fileName.lastIndexOf('.');

        //The file has a dot, so it has an extension.
        if(indexOfLastDot >= 0)
        {
            boolean validExt = false;

            //Check the list of file extensions.
            for(final String extension: allowableExtensions)
            {
                if(fileName.endsWith("." + extension))
                {
                    validExt = true;
                }
            }
            if(!validExt)
            {
                fileName += "." + defaultExtension;
            }
        }
        //File does not have an extension.
        else
        {
            fileName += "." + defaultExtension;
        }
        return new File(fileName);
    }

    /**
     * @param file File to check.
     * @param text Text to look for.
     * @return True if any line contains the text to search for (text cannot span lines). False otherwise.
     * @throws IOException For standard file I/O reasons.
     */
    public static boolean doesFileContainText(final File file, final String text) throws IOException
    {
        final LineIterator iter = FileUtils.lineIterator(file, "UTF-8");
        try
        {
            while(iter.hasNext())
            {
                final String line = iter.nextLine();
                if(line.contains(text))
                {
                    return true;
                }
            }
        }
        finally
        {
            LineIterator.closeQuietly(iter);
        }
        return false;
    }

    /**
     * Writes an {@link Object} (provided) to a file. Writing is done using {@link ObjectOutputStream} which creates a
     * binary file.
     * 
     * @param file The file to which to write.
     * @param obj The object to write.
     * @throws IOException
     */
    public static void writeObjectToFile(final File file, final Object obj) throws IOException
    {
        final FileOutputStream fileStream = new FileOutputStream(file);
        final ObjectOutputStream objStream = new ObjectOutputStream(fileStream);
        try
        {
            objStream.writeObject(obj);
        }
        finally
        {
            objStream.close();
        }
    }

    /**
     * Reads an {@link Object} from a file.
     * 
     * @param file The file from which to read.
     * @param type Specifies the return type of this method; the object itself should be null. The return of
     *            {@link ObjectInputStream#readObject()} will be cast to this.
     * @return An object of type specified by type.
     * @throws IOException Standard reasons.
     * @throws ClassNotFoundException The class specified by the file is not found.
     */
    @SuppressWarnings("unchecked")
    public static <T> T readObjectFromFile(final File file, final T type) throws IOException, ClassNotFoundException
    {
        final FileInputStream fileStream = new FileInputStream(file);
        final ObjectInputStream objStream = new ObjectInputStream(fileStream);
        try
        {
            return (T)objStream.readObject();
        }
        finally
        {
            objStream.close();
        }
    }

    /**
     * Copies all files (not directories) from baseDir to targetDir that match the filter.
     * 
     * @param baseDir Directory containing files to copy.
     * @param filter Filter dictating which files to copy.
     * @param targetDir Directory to which to copy the files. The directory will be created if it does not exist, but
     *            file creation will fail if its parent does not exist.
     * @throws IOException
     */
    public static void copyFiles(final File baseDir, final FileFilter filter, final File targetDir) throws IOException
    {
        if(!baseDir.exists() && !baseDir.canRead())
        {
            throw new IOException("Copy base directory " + baseDir.getAbsolutePath()
                + " does not exist or cannot be read.");
        }
        if(!targetDir.exists())
        {
            if(!targetDir.mkdir())
            {
                throw new IOException("Unable to create target directory, " + targetDir.getAbsolutePath());
            }
        }

        final File[] fileList = baseDir.listFiles();
        for(final File file: fileList)
        {
            if(file.isDirectory())
            {
                continue;
            }
            if(filter.accept(file))
            {
                FileUtils.copyFile(file, FileTools.newFile(targetDir, file.getName()));
            }
        }
    }

}
