package ohd.hseb.hefs.utils.tools;

import java.lang.reflect.Method;
import java.util.Calendar;

import org.apache.logging.log4j.Logger;

import ohd.hseb.hefs.utils.arguments.ArgumentsProcessor;
import ohd.hseb.hefs.utils.arguments.DefaultPredefinedArguments;
import ohd.hseb.hefs.utils.datetime.HEFSDateTools;
import ohd.hseb.hefs.utils.log4j.LoggingTools;
import ohd.hseb.hefs.utils.plugins.GenericParameter;


import com.google.common.base.Predicate;

/**
 * General tools that may be of use.
 * 
 * @author hank.herr
 */
public abstract class GeneralTools
{

    /**
     * @param first Object to check.
     * @param second Object to check.
     * @return True if the two objects are either both null or first.equals(second) returns true.
     */
    public static boolean checkForFullEqualityOfObjects(final Object first, final Object second)
    {
        if((first != null) && (second == null))
        {
            return false;
        }
        if((first == null) && (second != null))
        {
            return false;
        }
        if(first == null)
        {
            return true; //we've checked all we need to.
        }
        return first.equals(second);
    }

    public static void dumpStackTrace()
    {
        try
        {
            throw new Exception();
        }
        catch(final Exception e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Uses {@link LoggingTools#outputStackTraceAsDebug(Logger, Throwable)} to output the current stack trace to the
     * provided {@link Logger}.
     */
    public static void dumpStackTraceToLoggerAsDebug(final Logger log)
    {
        try
        {
            throw new Exception();
        }
        catch(final Exception e)
        {
            LoggingTools.outputStackTraceAsDebug(log, e);
        }
    }

    /**
     * Uses {@link LoggingTools#outputStackTraceAsDebug(Logger, Throwable)} to output the current stack trace to the
     * provided {@link Logger}. Only the specified number of rows are output.
     */
    public static void dumpStackTraceToLoggerAsDebug(final Logger log, final int rows)
    {
        try
        {
            throw new Exception();
        }
        catch(final Exception e)
        {
            LoggingTools.outputStackTraceAsDebug(log, e, rows);
        }
    }

    public static void dumpStackTrace(final int rows)
    {
        final Throwable t = new Throwable();
        System.err.println("####>> ---------------- DEBUG DUMP STACKTRACE");
        for(int i = 0; i < rows; i++)
        {
            if(i >= t.getStackTrace().length)
            {
                break;
            }
            System.err.println("    " + t.getStackTrace()[i].toString());
        }
    }

    /**
     * Returns the user-defined system time. If the passed-in ArgumentProcessor is null, then it throws
     * NullPointException. This assumes that argProc contains the argument DefaultPredefinedArguments.SYSTEM_TIME.<br>
     * <br>
     * This tool may belong in some kind of ArgumentProcessor toolset, but I'll leave it here for now.
     * 
     * @param argProc Arguments processor that must contain the argument DefaultPredefinedArguments.SYSTEM_TIME.
     * @return The arguments value. A null pointer exception will be thrown if it is not found.
     */
    public static Calendar getSystemTime(final ArgumentsProcessor argProc)
    {
        final GenericParameter systemTimeArg = argProc.getArgument(DefaultPredefinedArguments.SYSTEM_TIME);

        return HEFSDateTools.computeFixedDate(systemTimeArg.getValue());
    }

    /**
     * @param <T> The klass type.
     * @param klass The class containing the method to call.
     * @param methodName The name of the method to call, which must have a boolean return type!
     * @return A {@link Predicate} whose {@link Predicate#apply(Object)} method invokes the method specified for the
     *         given klass and methodName.
     */
    public static <T> Predicate<T> getMethodAsPredicate(final Class<T> klass, final String methodName)
    {
        try
        {
            final Method method = klass.getMethod(methodName);
            if(!method.getReturnType().equals(boolean.class))
            {
                throw new RuntimeException("Not boolean.");
            }
            return new Predicate<T>()
            {
                @Override
                public boolean apply(final T input)
                {
                    try
                    {
                        return (Boolean)method.invoke(input);
                    }
                    catch(final Exception e)
                    {
                        throw new RuntimeException(e);
                    }
                }
            };
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * @param <T> Class of the arguments.
     * @param arguments Variable list of arguments.
     * @return The first non-null argument in the variable list of arguments.
     */
    public static <T> T firstNonNull(final T... arguments)
    {
        for(final T arg: arguments)
        {
            if(arg != null)
            {
                return arg;
            }
        }
        return null;
    }
}
