package ohd.hseb.hefs.utils.tools;

import java.net.URL;

import javax.swing.ImageIcon;
import javax.swing.JLabel;

import com.google.common.base.Optional;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

/**
 * Loads and caches icons by name underneath of hefsIcons (see nonsrc).
 * 
 * @author alexander.garbarino
 */
public abstract class IconTools
{
    public static final String ICON_FOLDER = "hefsIcons";
    public static final String[] ICON_ORDERED_TYPES = {"png", "gif"};

    /**
     * Cache of icon names to icons.
     */
    private static final LoadingCache<String, Optional<ImageIcon>> _icons;
    static
    {
        final CacheLoader<String, Optional<ImageIcon>> loader = new CacheLoader<String, Optional<ImageIcon>>()
        {
            @Override
            public Optional<ImageIcon> load(final String key) throws Exception
            {
                final URL url = getResourceForIcon(key);
                if(url == null)
                {
                    return Optional.absent();
                }
                else
                {
                    return Optional.of(new ImageIcon(url));
                }
            }
        };
        _icons = CacheBuilder.newBuilder().build(loader);
    }

    /**
     * Used in two places to acquire the URL for a system resource icon obeying standard rules for location (
     * {@link #ICON_FOLDER} and name. It allows for either png or gif images, with png having preference.
     * 
     * @param iconName Name of image file withing {@link #ICON_FOLDER} without extension.
     * @return URL or null if not found.
     */
    private static URL getResourceForIcon(final String iconName)
    {
        URL url = null;
        for(final String type: ICON_ORDERED_TYPES)
        {
            url = ClassLoader.getSystemResource(String.format("%s/%s.%s", ICON_FOLDER, iconName, type));
            if(url != null)
            {
                break;
            }
        }
        return url;
    }

    /**
     * Attempts to find the icon using the standard naming convention assumed by the loading cache and
     * {@link #getResourceForIcon(String)}.
     * 
     * @return A newly constructed, non-shared {@link ImageIcon}.
     */
    public static ImageIcon loadNewConstructedIcon(final String iconName)
    {
        final URL url = getResourceForIcon(iconName);
        return new ImageIcon(url);
    }

    /**
     * @param iconName The name of the icon without .png. It will look for a resource under hefsIcons with this name
     *            with a .png extension.
     * @return An {@link ImageIcon} generated from the image file corresponding to the icon name.
     */
    public static ImageIcon getHSEBIcon(final String iconName)
    {
        return _icons.getUnchecked(iconName).orNull();
    }

    public static JLabel makeIconLabel(final String iconName)
    {
        final ImageIcon icon = getHSEBIcon(iconName);
        if(icon == null)
        {
            return new JLabel(iconName);
        }
        else
        {
            return new JLabel(icon);
        }
    }
}
