package ohd.hseb.hefs.utils.tools;

import static com.google.common.collect.Lists.newArrayList;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import ohd.hseb.hefs.utils.AbstractFunction;
import ohd.hseb.hefs.utils.Dyad;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

/**
 * Tools that can be used with general lists of objects.
 * 
 * @author hank.herr
 */
public abstract class ListTools
{

    /**
     * Add the items from list toAdd to list base, but only if they are not already present. Based on List contains
     * method.
     * 
     * @param base List to which to add stuff.
     * @param toAdd Stuff to add.
     */
    public static <E> void addAllUniqueItemsToList(final List<E> base, final Collection<E> toAdd)
    {
        if(toAdd == null)
        {
            return;
        }
        for(final E item: toAdd)
        {
            addItemIfNotAlreadyInList(base, item);
        }
    }

    /**
     * Add one items to the base list if it is not already present. Based on List contains method.
     * 
     * @param base List to which to add item.
     * @param item Item to add.
     */
    public static <E> void addItemIfNotAlreadyInList(final List<E> base, final E item)
    {
        if(!base.contains(item))
        {
            base.add(item);
        }
    }

    /**
     * A variation on {@link #addItemIfNotAlreadyInList(List, Object)} that allows for a customized comparison Checker
     * to be passed in.
     * 
     * @param base The list to which to add.
     * @param item The item to add.
     * @param comparisonChecker A {@link Comparator} whose {@link Comparator#compare(Object, Object)} returns 0 if the
     *            two items should be considered equal.
     */
    public static <E> void addItemIfNotAlreadyInList(final List<E> base,
                                                     final E item,
                                                     final Comparator<E> comparisonChecker)
    {
        for(final E existingItem: base)
        {
            if(comparisonChecker.compare(existingItem, item) == 0)
            {
                return;
            }
        }
        base.add(item);
    }

    /**
     * Add one items to the base list if it is not already present. Based on List contains method.
     * 
     * @param base List to which to add item.
     * @param index Index at which to add it
     * @param item Item to add.
     */
    public static <E> void insertItemIfNotAlreadyInList(final List<E> base, final int index, final E item)
    {
        if(!base.contains(item))
        {
            base.add(index, item);
        }
    }

    /**
     * Add any non null items to the list.
     * 
     * @param base List to which to add stuff.
     * @param toAdd Stuff to add if not null.
     */
    public static <E> void allAllItemsToListIfNotNull(final List<E> base, final Collection<E> toAdd)
    {
        if(toAdd != null)
        {
            base.addAll(toAdd);
        }
    }

    /**
     * Returns the index of the object in the provided list. This does NOT use the equals method, which is what indexOf
     * uses. Rather, it uses ==.
     * 
     * @param list List of stuff.
     * @param object Object to find.
     * @return Index of object in list or -1 if not found.
     */
    public static <E> int indexOfObjectInList(final List<E> list, final E object)
    {
        for(int i = 0; i < list.size(); i++)
        {
            if(list.get(i) == object)
            {
                return i;
            }
        }
        return -1;
    }

//    This is List.equals(Object).
//    
//    /**
//     * Passes through the first List, checking each item to see if the the items as the same index in the second List
//     * equals it (via equals method). Result is that both lists must be exactly identical, including order.
//     * 
//     * @param first First list to check.
//     * @param second Second list to check.
//     * @return Returns false if the two lists are not exactly identical, according to the equals method.
//     */
//    public static <E> boolean oneSidedListCheckWhereOrderMatters(List<E> first, List<E> second)
//    {
//        int i;
//        if(first.size() != second.size())
//        {
//            return false;
//        }
//        for(i = 0; i < first.size(); i++)
//        {
//            if(!first.get(i).equals(second.get(i)))
//            {
//                return false;
//            }
//        }
//        return true;
//    }

    /**
     * For strings. Two-sided check allows for the two lists to be in different order.
     * 
     * @param first First list of strings to check.
     * @param second Second list of strings to check.
     * @return True if both lists contain the same strings, possibly not in the same order. False if either list is
     *         larger than the other or if a string exists in one and not the other.
     */
    public static boolean twoSidedListCheckStringsIgnoreCase(final List<String> first, final List<String> second)
    {

        int i, j;
        if(first.size() != second.size())
        {
            return false;
        }
        for(i = 0; i < first.size(); i++)
        {
            for(j = 0; j < second.size(); j++)
            {
                if(first.get(i).equalsIgnoreCase(second.get(j)))
                {
                    break;
                }
            }
            if(j == second.size())
            {
                return false;
            }
        }
        for(j = 0; j < second.size(); j++)
        {
            for(i = 0; i < first.size(); i++)
            {
                if(first.get(i).equalsIgnoreCase(second.get(j)))
                {
                    break;
                }
            }
            if(i == first.size())
            {
                return false;
            }
        }
        return true;
    }

    /**
     * Two-sided refers allows for the two lists to be in different order.
     * 
     * @param first First list
     * @param second Second list
     * @return True if the two lists are equal, but possibly out of order.
     */
    public static <E> boolean twoSidedListCheck(final List<E> first, final List<E> second)
    {
        int i, j;
        if(first.size() != second.size())
        {
            return false;
        }
        for(i = 0; i < first.size(); i++)
        {
            for(j = 0; j < second.size(); j++)
            {
                if(first.get(i).equals(second.get(j)))
                {
                    break;
                }
            }
            if(j == second.size())
            {
                return false;
            }
        }
        for(j = 0; j < second.size(); j++)
        {
            for(i = 0; i < first.size(); i++)
            {
                if(first.get(i).equals(second.get(j)))
                {
                    break;
                }
            }
            if(i == first.size())
            {
                return false;
            }
        }
        return true;
    }

    /**
     * Return a new list composed of all the given elements.
     * 
     * @param colls all the collections whose elements you want in this list
     * @return a new list containing the elements of all the supplied collections
     */
    public static <E> List<E> concat(final Iterable<? extends E>... colls)
    {
        final List<E> list = new ArrayList<E>();
        for(int i = 0; i < colls.length; i++)
        {
            Iterables.addAll(list, colls[i]);
        }
        return list;
    }

    /**
     * Constructs a list composed of the elements of the given arrays.
     * 
     * @param eltArrays arrays of elements to includes
     * @return a new list containing the given elements in order
     */
    public static <E> List<E> concat(final E[]... eltArrays)
    {
        final List<E> list = new ArrayList<E>();
        for(final E[] eltArray: eltArrays)
        {
            for(final E elt: eltArray)
            {
                list.add(elt);
            }
        }
        return list;
    }

    /**
     * Takes a collection of collections, and returns a single list of all their elements.
     * 
     * @param coll
     * @return
     */
    public static <E> List<E> flatten(final Collection<? extends Collection<E>> coll)
    {
        final List<E> list = new ArrayList<E>();
        for(final Collection<E> innerColl: coll)
        {
            for(final E elt: innerColl)
            {
                list.add(elt);
            }
        }
        return list;
    }

    @SuppressWarnings("unchecked")
    public static void setAll(final List list, final Object value)
    {
        for(int i = 0; i < list.size(); i++)
        {
            list.set(i, value);
        }
    }

    /**
     * Returns the given list, or a new empty list if given null.
     * 
     * @param list the list to return
     * @return the given list, or a new empty one
     */
    public static <E> List<E> nullToEmptyList(final List<E> list)
    {
        if(list == null)
        {
            return newArrayList();
        }
        else
        {
            return list;
        }
    }

    /**
     * Adds {@code element} to {@code list}, and returns {@code list}.
     */
    public static <E> List<E> addTo(final List<E> list, final E element)
    {
        list.add(element);
        return list;
    }

    /**
     * Adds {@code element} to {@code list} at position {@code index}, and returns {@code list}.
     */
    public static <E> List<E> addTo(final List<E> list, final int index, final E element)
    {
        list.add(index, element);
        return list;
    }

    /**
     * Partial application of {@link #mapToArrayList(Function, Iterable)}.<br/>
     * Takes a function from {@code F} to {@code T} and makes a function that applies the original to every element in a
     * list.
     * 
     * @param <F> the from type
     * @param <T> the to type
     * @param function the function to apply to each list element
     * @return a new function that works on lists
     */
    public static <F, T> AbstractFunction<Iterable<? extends F>, ArrayList<T>> mapToArrayList(final Function<F, T> function)
    {
        return new AbstractFunction<Iterable<? extends F>, ArrayList<T>>()
        {
            @Override
            public ArrayList<T> apply(final Iterable<? extends F> input)
            {
                final ArrayList<T> output = newArrayList();
                for(final F f: input)
                {
                    output.add(function.apply(f));
                }
                return output;
            }
        };
    }

    /**
     * Creates a new list composed of {@code fun} applied to each element of {@code elements} in sequence.
     */
    public static <F, T> ArrayList<T> mapToArrayList(final Function<F, T> fun, final Iterable<? extends F> elements)
    {
        return mapToArrayList(fun).apply(elements);
    }

    /**
     * Partial application of {@link #mapArrayToArrayList(Function, Object[])}. Proper use:
     * ListTools.mapArrayToArrayList([function to apply to list elements]).apply([list of items]).
     * 
     * @param <F> the input element type
     * @param <T> the output element type
     * @param function the function used to create the list
     * @return a function which applies {@code function} to every element of the passed array
     */
    public static <F, T> AbstractFunction<F[], ArrayList<T>> mapArrayToArrayList(final Function<F, T> function)
    {
        return new AbstractFunction<F[], ArrayList<T>>()
        {
            @Override
            public ArrayList<T> apply(final F[] input)
            {
                final ArrayList<T> output = newArrayList();
                for(final F f: input)
                {
                    output.add(function.apply(f));
                }
                return output;
            }
        };
    }

    /**
     * Creates a new array list which is the result of {@code function} applied to every element in {@code elements}.
     * 
     * @param <F> the input element type
     * @param <T> the output element type
     * @param function the function used to create the list
     * @param elements the elements given to {@code function}
     * @return the new, mapped list
     */
    public static <F, T> ArrayList<T> mapArrayToArrayList(final Function<F, T> function, final F[] elements)
    {
        return mapArrayToArrayList(function).apply(elements);
    }

    /**
     * Creates an array list of {@link Dyad} objects.
     * 
     * @param <A> Left element type for {@link Dyad}
     * @param <B> Right element type for {@link Dyad}
     * @param lefts List of left elements
     * @param rights List of right elements
     * @return List of {@link Dyad} instances containing the paired elements for lefts and rights.
     */
    public static <A, B> ArrayList<Dyad<A, B>> createArrayList(final Iterable<? extends A> lefts,
                                                               final Iterable<? extends B> rights)
    {
        final ArrayList<Dyad<A, B>> list = newArrayList();
        final Iterator<? extends A> leftIter = lefts.iterator();
        final Iterator<? extends B> rightIter = rights.iterator();
        while(leftIter.hasNext() && rightIter.hasNext())
        {
            list.add(new Dyad<A, B>(leftIter.next(), rightIter.next()));
        }
        return list;
    }

    /**
     * @param <F> Base type
     * @param <T> Target type
     * @param base Base collection.
     * @param targetUnit An instance of the target type, which can be null.
     * @return An {@link ArrayList} of type T containing elements from base cast to type T.
     */
    @SuppressWarnings("unchecked")
    public static <F, T> ArrayList<T> convertCollection(final Collection<F> base, final T targetUnit)
    {
        final ArrayList<T> result = new ArrayList<T>();
        for(final F item: base)
        {
            result.add((T)item);
        }
        return result;
    }

    /**
     * @param <E> The list type.
     * @param baseList Base list containing elements that are to be returned.
     * @param indices List of indices within baseList of elements to be returned.
     * @return A sublist containing only those elements specified by indices.
     */
    public static <E> List<E> retrieveAll(final List<E> baseList, final int[] indices)
    {
        final List<E> results = new ArrayList<E>();
        for(final int i: indices)
        {
            results.add(baseList.get(i));
        }
        return results;
    }

    /**
     * @param <E> The type of the list.
     * @param baseList List containing items to get (they will not be copied).
     * @param numberOfElements The number of elements to extract starting with first.
     * @return A sublist containing the first numberOfElements in the baseList. The whole list is returned if
     *         numberOfElements is larger than the size of baseList.
     */
    public static <E> List<E> createSubList(final Collection<E> baseList, final int numberOfElements)
    {
        return createSubList(baseList, 0, true, numberOfElements);
    }

    /**
     * @param <E>
     * @param baseList The list from which to grab elements.
     * @param startIndex The starting index. If a negative value or too large of a value is provided, and
     *            {@link IllegalArgumentException} will be thrown.
     * @param after True to pick up values starting from and after the specified startIndex, false for before.
     * @param numberOfElements The number of elements to put in the sublist. If negative, all elements before/after the
     *            startIndex are grabbed. If 0, the returned list is empty. If it is more than the number of elements
     *            before/after the startIndex, all elements before/after are grabbed.
     * @return A {@link ArrayList} of consecutive elements from the provided base list.
     */
    public static <E> List<E> createSubList(final Collection<E> baseList,
                                            final int startIndex,
                                            final boolean after,
                                            int numberOfElements)
    {
        if((startIndex < 0) || (startIndex >= baseList.size()))
        {
            throw new IllegalArgumentException("in createSubListAfter, firstIndex is invalid relative to baseList.");
        }

        //This will allow for all elements to be grabbed before/after startIndex.
        if(numberOfElements < 0)
        {
            numberOfElements = Integer.MAX_VALUE;
        }

        //Determine the maximum possible number of elements based on baseList size and startIndex.
        int possibleNumberOfElements = numberOfElements;
        if(after)
        {
            possibleNumberOfElements = baseList.size() - startIndex;
        }
        else
        {
            possibleNumberOfElements = startIndex;
        }

        //Initialize the results based on the maximum possible number of elements and the requested number of elements.
        final List<E> results = new ArrayList<E>(Math.min(possibleNumberOfElements, numberOfElements));

        //Loop through all elements starting from the beginning.  Since I have a collection, I have no get method,
        //so iteration is the only choice without mapping the collection to something else.
        int index = 0;
        for(final E item: baseList)
        {
            if((after) && (index >= startIndex))
            {
                results.add(item);
            }
            if((!after) && (index < startIndex))
            {
                results.add(item);
            }
            if(results.size() >= numberOfElements)
            {
                break;
            }
            index++;
        }
        return results;
    }

    /**
     * Wraps baseList.iterator().next().
     * 
     * @param <E>
     * @param baseList List to search.
     * @return The first item in the baseList. Null is returned if the list is empty.
     */
    public static <E> E first(final Collection<E> baseList)
    {
        if(baseList.isEmpty())
        {
            return null;
        }
        return baseList.iterator().next();
    }

    /**
     * Provides one-liner access to the last element in the provided list. If you know the type of {@link Collection}
     * passed in, you might want to use methods available for that type for getting the last element. This method needs
     * to iterate through the list and is, therefore, not very efficient.
     * 
     * @param <E>
     * @param baseList List to search.
     * @return The last item in the baseList. Null is returned if the list is empty.
     */
    public static <E> E last(final Collection<E> baseList)
    {
        E last = null;
        for(final E item: baseList)
        {
            last = item;
        }
        return last;
    }

    /**
     * An sorter that allows for one-line sorting.
     * 
     * @param items Items to sort
     * @return A sorted list of the passed in items.
     */
    @SuppressWarnings("unchecked")
    public static Collection sort(final Collection items)
    {
        final List sorted = Lists.newArrayList(items);
        Collections.sort(sorted);
        return sorted;
    }

    /**
     * @param strToCompare {@link String} to look for ignoring case.
     * @param list {@link List} to search.
     * @return Index of strToCompare in the provided list, ignoring case. -1 is returned if the item is not found.
     */
    public static int indexOfCaseInsensitive(final String strToCompare, final List<String> list)
    {
        for(int i = 0; i < list.size(); i++)
        {
            if(list.get(i).equalsIgnoreCase(strToCompare))
            {
                return i;
            }
        }
        return -1;
    }
}
