package ohd.hseb.hefs.utils.tools;

import static com.google.common.collect.Lists.newArrayList;
import static ohd.hseb.hefs.utils.tools.ListTools.mapToArrayList;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import ohd.hseb.hefs.utils.AbstractFunction;

import com.google.common.base.Function;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

public abstract class MapTools
{
    /**
     * For every {@code key} in {@code keys}, adds {@code key -> keyToValueFunction(key)} to {@code map}.
     * 
     * @param map the map to modify
     * @param keys the keys to place into the map
     * @param keyToValueFunction the function to generate values with
     */
    public static <K, V> void mapKeysIntoMap(final Map<K, V> map,
                                             final Iterable<? extends K> keys,
                                             final Function<? super K, ? extends V> keyToValueFunction)
    {
        for(final K key: keys)
        {
            map.put(key, keyToValueFunction.apply(key));
        }
    }

    /**
     * Partial Application of {@link #mapToHashMap(Function, Iterable)}.
     */
    public static <K, V> Function<Iterable<? extends K>, HashMap<K, V>> mapToHashMap(final Function<K, V> keyToValueFunction)
    {
        return new AbstractFunction<Iterable<? extends K>, HashMap<K, V>>()
        {
            @Override
            public HashMap<K, V> apply(final Iterable<? extends K> input)
            {
                return createHashMap(input, mapToArrayList(keyToValueFunction, input));
            }
        };
    }

    /**
     * Creates a hash map by pairing each key in {@code keys} with the value from applying {@code keyToValueFunction} to
     * that key.
     * 
     * @param keyToValueFunction the function used to generate values from keys
     * @param keys the keys in the map
     */
    public static <K, V> HashMap<K, V> mapToHashMap(final Function<K, V> keyToValueFunction,
                                                    final Iterable<? extends K> keys)
    {
        return mapToHashMap(keyToValueFunction).apply(keys);
    }

    /**
     * As {@link Map#get(Object)}, but allows a default value to be specified if the map does not contain the specified
     * key. This properly returns a null value if it has been added to the map.
     * 
     * @param <K> the key class
     * @param <V> the value class
     * @param map the map to get from
     * @param key the key to retrieve
     * @param defaultValue the default value to use if the specified key is not found
     * @return the value for the given key in the given map, or the default value if not present
     */
    public static <K, V> V get(final Map<K, V> map, final K key, final V defaultValue)
    {
        if(map.containsKey(key))
        {
            return map.get(key);
        }
        else
        {
            return defaultValue;
        }
    }

    /**
     * Creates a new {@link HashMap}, pairing each successive key in {@code keys} to each successive value in
     * {@code values}.
     * 
     * @param <K> the key type
     * @param <V> the value type
     * @param keys every key to be placed in the map
     * @param values every value to be placed in the map
     * @return the new map
     */
    public static <K, V> HashMap<K, V> createHashMap(final Iterable<? extends K> keys,
                                                     final Iterable<? extends V> values)
    {
        final HashMap<K, V> map = Maps.newHashMap();
        addToMap(map, keys, values);
        return map;
    }

    /**
     * Creates a new {@link LinkedHashMap}, pairing each successive key in {@code keys} to each successive value in
     * {@code values}.
     * 
     * @param <K> the key type
     * @param <V> the value type
     * @param keys every key to be placed in the map
     * @param values every value to be placed in the map
     * @return the new map
     */
    public static <K, V> LinkedHashMap<K, V> createLinkedHashMap(final Iterable<? extends K> keys,
                                                                 final Iterable<? extends V> values)
    {
        final LinkedHashMap<K, V> map = Maps.newLinkedHashMap();
        addToMap(map, keys, values);
        return map;
    }

    /**
     * Creates a new {@link LinkedHashMap}, pairing each successive key in {@code keys} to each successive value in
     * {@code values}.
     * 
     * @param <K> the key type
     * @param <V> the value type
     * @param keys every key to be placed in the map
     * @param values every value to be placed in the map
     * @return the new map
     */
    public static <K, V> LinkedHashMap<K, V> createLinkedHashMap(final K[] keys, final Iterable<? extends V> values)
    {
        final LinkedHashMap<K, V> map = Maps.newLinkedHashMap();
        addToMap(map, Lists.newArrayList(keys), values);
        return map;
    }

    /**
     * Creates a new {@link HashMap}, pairing each successive key in {@code keys} to each successive value in
     * {@code values}.
     * 
     * @param <K> the key type
     * @param <V> the value type
     * @param keys every key to be placed in the map
     * @param values every value to be placed in the map
     * @return the new map
     */
    public static <K, V> HashMap<K, V> createHashMap(final K[] keys, final Iterable<? extends V> values)
    {
        final HashMap<K, V> map = Maps.newHashMap();
        addToMap(map, Lists.newArrayList(keys), values);
        return map;
    }

    /**
     * Creates a new {@link HashMap}, pairing each successive key in {@code keys} to each successive value in
     * {@code values}.
     * 
     * @param <K> the key type
     * @param <V> the value type
     * @param keys every key to be placed in the map
     * @param values every value to be placed in the map
     * @return the new map
     */
    public static <K, V> HashMap<K, V> createHashMap(final K[] keys, final V[] values)
    {
        final HashMap<K, V> map = Maps.newHashMap();
        addToMap(map, keys, values);
        return map;
    }

    /**
     * Adds each successive key in {@code keys} paired to each successive value in {@code values} to {@code map}.
     * 
     * @param <K> the key type
     * @param <V> the value type
     * @param map the map to add to
     * @param keysToAdd each key to add
     * @param valuesToAdd each value to add
     */
    public static <K, V> void addToMap(final Map<K, V> map,
                                       final Iterable<? extends K> keysToAdd,
                                       final Iterable<? extends V> valuesToAdd)
    {
        final Iterator<? extends K> keyIter = keysToAdd.iterator();
        final Iterator<? extends V> valIter = valuesToAdd.iterator();
        while(keyIter.hasNext() && valIter.hasNext())
        {
            map.put(keyIter.next(), valIter.next());
        }
        if(keyIter.hasNext() || valIter.hasNext())
        {
            throw new IllegalArgumentException("Unequal number of keys and values.");
        }
    }

    /**
     * Adds each successive key in {@code keys} paired to each successive value in {@code values} to {@code map}.
     * 
     * @param <K> the key type
     * @param <V> the value type
     * @param map the map to add to
     * @param keysToAdd each key to add
     * @param valuesToAdd each value to add
     */
    public static <K, V> void addToMap(final Map<K, V> map, final K[] keysToAdd, final V[] valuesToAdd)
    {
        addToMap(map, newArrayList(keysToAdd), newArrayList(valuesToAdd));
    }

    /**
     * Takes an {@link Iterable} and creates a {@link BiMap} with each element having its own index as the key.
     * 
     * @param items the items to index
     * @return a map of indices to values
     */
    public static <V> BiMap<Integer, V> createIndexMap(final Iterable<? extends V> items)
    {
        final BiMap<Integer, V> map = HashBiMap.create();
        int i = 0;
        for(final V item: items)
        {
            map.put(i, item);
            i++;
        }
        return map;
    }

    /**
     * Concatenates the contents of map2 into map1. It will check as it goes to make sure that if one key has a value in
     * both maps, then the class of those results must be identical (i.e., you are not trying to concatenate two
     * different data types). If the value for a key is a {@link Collection}, then it calls the
     * {@link Collection#addAll(Collection)} method to add the contents of the {@link Collection} returned from map2 to
     * the collection in map1. If the value for a key is a {@link Map}, it calls this same
     * {@link #concatenateMaps(Map, Map)} method recursively to combine the map value from map2 onto the value from map1<br>
     * <br>
     * If a map2 key is not found in map1, then it simply puts the map2 value into map1. It does NOT check in any way if
     * the type of data being given is valid. PLEASE MAKE SURE TO PASS IN TWO MAPS OF THE SAME TYPE OR YOU MAY GET WEIRD
     * RESULTS.
     * 
     * @param map1 First map which will be modified.
     * @param map2 Second map whose contents are added to the first map.
     */
    @SuppressWarnings("unchecked")
    public static void concatenateMaps(final Map map1, final Map map2)
    {
        for(final Object key: map2.keySet())
        {
            if(map1.containsKey(key))
            {
                if(!map1.get(key).getClass().equals(map2.get(key).getClass()))
                {
                    throw new IllegalArgumentException("Map values do not appear to be of the same type.");
                }

                if(map1.get(key) instanceof Map)
                {
                    concatenateMaps((Map)map1.get(key), (Map)map2.get(key));
                }
                else if(map1.get(key) instanceof Collection)
                {
                    ((Collection)map1.get(key)).addAll((Collection)map2.get(key));
                }
                else
                {
                    map1.put(key, map2.get(key));
                }
            }
            else
            {
                map1.put(key, map2.get(key));
            }
        }
    }

    /**
     * @param map Map from which to remove entries based on keys.
     * @param keysToRemove The keys to remove.
     */
    public static void removeAll(final Map map, final Collection keysToRemove)
    {
        for(final Object key: keysToRemove)
        {
            map.remove(key);
        }
    }
}
