package ohd.hseb.hefs.utils.tools;

import static java.lang.Math.abs;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.math.BigInteger;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import ohd.hseb.hefs.utils.BinaryPredicate;
import ohd.hseb.util.misc.HNumber;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.LineIterator;
import org.apache.commons.lang.ArrayUtils;

import com.google.common.collect.Lists;

public abstract class NumberTools
{
    /**
     * @return A float[] cast from the given input.
     */
    public static float[] convertToFloatArray(final double[] input)
    {
        final float[] results = new float[input.length];
        for(int i = 0; i < input.length; i++)
        {
            results[i] = (float)input[i];
        }
        return results;
    }

    /**
     * @return A float[] cast from the given input.
     */
    public static float[] convertToFloatArray(final int[] input)
    {
        final float[] results = new float[input.length];
        for(int i = 0; i < input.length; i++)
        {
            results[i] = input[i];
        }
        return results;
    }

    /**
     * @return A double[] cast from the given input.
     */
    public static double[] convertToDoubleArray(final float[] input)
    {
        final double[] results = new double[input.length];
        for(int i = 0; i < input.length; i++)
        {
            results[i] = input[i];
        }
        return results;
    }

    /**
     * @return A double[] cast from the given input.
     */
    public static double[] convertToDoubleArray(final int[] input)
    {
        final double[] results = new double[input.length];
        for(int i = 0; i < input.length; i++)
        {
            results[i] = input[i];
        }
        return results;
    }

    /**
     * Creates a list of integers from {@code 0} to {@code size - 1}.
     * 
     * @param size the size of the list
     * @return an list of integers from {@code 0} to {@code size - 1}
     */
    public static List<Integer> iota(final int size)
    {
        final List<Integer> list = Lists.newArrayList();
        for(int i = 0; i < size; i++)
        {
            list.add(i);
        }
        return list;
    }

    /**
     * @return A list of numbers from first to last (inclusive) incrementing by 1.
     */
    public static List<Integer> createList(final int first, final int last)
    {
        final List<Integer> list = Lists.newArrayList();
        for(int i = first; i <= last; i++)
        {
            list.add(i);
        }
        return list;
    }

    /**
     * Note: uses {@link Number#doubleValue()} on all inputs.
     * 
     * @param tolerance
     * @return Returns a {@link BinaryPredicate} that can be used to compare a large number of floats via the
     *         {@link BinaryPredicate#apply(Object, Object)} method and
     *         {@link BinaryPredicate#areAllTrue(Iterable, Iterable)} method.
     */
    public static BinaryPredicate<? extends Number, ? extends Number> nearEqual(final Number tolerance)
    {
        return new BinaryPredicate<Number, Number>()
        {
            @Override
            public boolean apply(final Number input0, final Number input1)
            {
                return abs(input0.doubleValue() - input1.doubleValue()) < tolerance.doubleValue();
            }
        };
    }

    /**
     * Uses the {@link BinaryPredicate} returned by {@link #nearEqual(Number)}.
     * 
     * @param one First number.
     * @param two Second number.
     * @param delta Tolerance allowed; always a double.
     * @return True if equal within tolerance.
     */
    @SuppressWarnings("unchecked")
    public static boolean nearEquals(final Number one, final Number two, final double tolerance)
    {
        return ((BinaryPredicate<Number, Number>)NumberTools.nearEqual(tolerance)).apply(one, two);
    }

    /**
     * Uses the {@link BinaryPredicate} returned by {@link #nearEqual(Number)}.
     * 
     * @param first First {@link Iterable} set of {@link Number}s.
     * @param second Second {@link Iterable} set of {@link Number}s.
     * @param tolerance Tolerance allowed; always a double.
     * @return True if equal within tolerance.
     */
    @SuppressWarnings("unchecked")
    public static boolean nearEquals(final Iterable<? extends Number> first,
                                     final Iterable<? extends Number> second,
                                     final double tolerance)
    {
        return ((BinaryPredicate<Number, Number>)NumberTools.nearEqual(tolerance)).areAllTrue(first, second);
    }

    /**
     * Wrapper on {@link #nearEquals(Iterable, Iterable, double)}, converting the arrays to lists.
     */
    public static boolean nearEquals(final double[] first, final double[] second, final double tolerance)
    {
        final List<Double> firstList = Arrays.asList(ArrayUtils.toObject(first));
        final List<Double> secondList = Arrays.asList(ArrayUtils.toObject(second));
        return nearEquals(firstList, secondList, tolerance);
    }

    /**
     * @param numbers List of numbers of any kind.
     * @return Array of double values.
     */
    public static double[] convertNumbersToDoublesArray(final Collection<? extends Number> numbers)
    {
        final double[] results = new double[numbers.size()];
        int i = 0;
        for(final Number num: numbers)
        {
            results[i] = num.doubleValue();
            i++;
        }
        return results;
    }

    /**
     * @param numbers List of numbers of any kind.
     * @return Array of int values.
     */
    public static int[] convertNumbersToIntsArray(final Collection<? extends Number> numbers)
    {
        final int[] results = new int[numbers.size()];
        int i = 0;
        for(final Number num: numbers)
        {
            results[i] = num.intValue();
            i++;
        }
        return results;
    }

    /**
     * Calls {@link #writeNumbersToFile(File, List)}, converting the double[] to a list. Each number is written in a
     * separate line.
     * 
     * @param file File to which to write.
     * @param numbers Numbers to write.
     * @throws Exception Standard reasons.
     */
    public static void writeNumbersToFile(final File file, final double[] numbers) throws Exception
    {
        writeNumbersToFile(file, Arrays.asList(ArrayUtils.toObject(numbers)));
    }

    /**
     * Writes a {@link List} of {@link Number}s to a file. Each number is written on a separate line.
     * 
     * @param file The file to which to write.
     * @param numbers The numbers to write.
     * @throws IOException Standard reasons.
     */
    public static void writeNumbersToFile(final File file, final List<? extends Number> numbers) throws IOException
    {
        final FileWriter writer = new FileWriter(file);
        try
        {
            for(final Number number: numbers)
            {
                writer.write("" + number.doubleValue() + "\n");
            }
        }
        finally
        {
            writer.close();
        }
    }

    /**
     * All lines of the file are read except lines that are empty after trimming.
     * 
     * @param file The file to read
     * @return List of {@link Number}s from a file. Each number is an instance of {@link Double}.
     * @throws IOException Standard reasons
     * @throws NubmerFormatException If any line is not a number after trimming.
     */
    public static List<Number> readNumbersFromFile(final File file) throws IOException, NumberFormatException
    {
        final List<Number> results = new ArrayList<Number>();
        final LineIterator iter = FileUtils.lineIterator(file);
        try
        {
            while(iter.hasNext())
            {
                final String line = iter.nextLine();
                if(!line.trim().isEmpty())
                {
                    final double value = Double.parseDouble(line.trim());
                    results.add(value);
                }
            }
        }
        finally
        {
            iter.close();
        }
        return results;
    }

    /**
     * Calls {@link #readNumbersFromFile(File)} and converts the results to an array of doubles.
     * 
     * @param file File to read.
     * @return Array specifying numbers read.
     * @throws IOException Standard reasons.
     * @throws NumberFormatException If any line is not a number after trimming.
     */
    public static double[] readNumbersFromFileAsArray(final File file) throws IOException, NumberFormatException
    {
        final List<Number> numbers = readNumbersFromFile(file);
        return convertNumbersToDoublesArray(numbers);
    }

    /**
     * All lines of the file are read except lines that are empty after trimming.
     * 
     * @param file The file to read
     * @return List of {@link Number}s from a file. Each number is an instance of {@link Double}.
     * @throws IOException Standard reasons
     * @throws NubmerFormatException If any line is not a number after trimming.
     */
    public static List<Float> readNumbersFromFileAsFloat(final File file) throws IOException, NumberFormatException
    {
        final List<Float> results = new ArrayList<Float>();
        final LineIterator iter = FileUtils.lineIterator(file);
        try
        {
            while(iter.hasNext())
            {
                final String line = iter.nextLine();
                if(!line.trim().isEmpty())
                {
                    final float value = Float.parseFloat(line.trim());
                    results.add(value);
                }
            }
        }
        finally
        {
            iter.close();
        }
        return results;
    }

    /**
     * Wrapper on {@link NumberFormat} also handles {@link Double#NaN}, {@link Double#POSITIVE_INFINITY}, and
     * {@link Double#NEGATIVE_INFINITY} (or float versions), ensuring that the returned strings use standard ASCII
     * characters.
     * 
     * @param nf The {@link NumberFormat} instance being wrapped.
     * @param number The number to format.
     * @return A string: either the return of {@link NumberFormat#format(double)} or 'NaN', '+INF', or '-INF', as
     *         appropriate.
     */
    public static String formatNumber(final NumberFormat nf, final Number number)
    {
        if(Double.isNaN(number.doubleValue()))
        {
            return "NaN";
        }
        if(Double.isInfinite(number.doubleValue()))
        {
            if(Double.valueOf(number.doubleValue()).equals(Double.POSITIVE_INFINITY))
            {
                return "+INF";
            }
            else
            {
                return "-INF";
            }
        }
        return nf.format(number);
    }

    /**
     * Generates a sequence of integers.
     * 
     * @param start The first value in the sequence.
     * @param step The step size between values.
     * @param upperLimit The upperLimit, inclusive. No value beyond this amount will be added.
     * @return A {@link List} of {@link Integer}s.
     */
    public static List<Integer> generateNumberSequence(final int start, final int step, final int upperLimit)
    {
        final List<Integer> results = Lists.newArrayList();
        for(int i = start; i <= upperLimit; i += step)
        {
            results.add(i);
        }
        return results;
    }

    /**
     * Generic function to determine if a number is between (inclusive) two bounds. The relative size of the bounds does
     * not matter.
     * 
     * @param checkNumber Number to check.
     * @param bound1 First bound to use; can be smaller or larger than the other.
     * @param bound2 Second bound to use; can be smaller or larger than the other.
     * @return True if the number is within the bounds, inclusive.
     */
    public static boolean isBetween(final Number checkNumber, final Number bound1, final Number bound2)
    {
        final double bound1D = bound1.doubleValue();
        final double bound2D = bound2.doubleValue();
        final double valueD = checkNumber.doubleValue();
        if(bound1D < bound2D)
        {
            return (bound1D <= valueD) && (valueD <= bound2D);
        }
        return (bound2D <= valueD) && (valueD <= bound1D);
    }

    /**
     * Generates a sequence of doubles.
     * 
     * @param start The starting value in the sequence.
     * @param step The step size between values. Negative is allowed.
     * @param limit The ending limit for the sequence, inclusive. The limit can be larger or smaller than start, but if
     *            it is smaller, then the step size should be negative or you will get a one element list with start.
     * @return A {@link List} of {@link Double}s. The method {@link #isBetween(Number, Number, Number)} is used.
     */
    public static List<Double> generateNumberSequence(final double start, final double step, final double limit)
    {
        final List<Double> results = Lists.newArrayList();
        double value = start;
        while(isBetween(value, start, limit))
        {
            results.add(value);
            value += step;
        }
        return results;
    }

    /**
     * @param fullListSortedInAscendingOrder Full list containing all numbers.
     * @param value The desired number.
     * @return Returns the lower bound and upper bounds from the provided list of numbers that bound the provided
     *         number. The lower bound is the first number in the two item returned list, and the upper bound is the
     *         second. If the lower bound is null, then the value is smaller than all numbers in the list and the upper
     *         bound will be set to the smallest value in the list. If the upper bound is null, then the value is larger
     *         than all in the list and the upper bound will be set to the largest value in the list.
     */
    public static Integer[] determineBoundingIntegers(final List<Integer> fullListSortedInAscendingOrder,
                                                      final int value)
    {
        Integer lb = null;
        Integer ub = null;
        int i;

        //In this loop, if all values are larger than value, then ub will be set to the first item in the list
        //and lb will be unchanged.  If no items are larger than value, then the if clause below is triggered.  If the
        //number is in the middle somewhere, then the lb and ub are set appropriately from the list contents.  If the 
        //value is one of the items in the list exactly, then the lb will be set to it.  
        for(i = 0; i < fullListSortedInAscendingOrder.size(); i++)
        {
            if(fullListSortedInAscendingOrder.get(i) > value)
            {
                ub = fullListSortedInAscendingOrder.get(i);
                if(i > 0)
                {
                    lb = fullListSortedInAscendingOrder.get(i - 1);
                }
                break;
            }
        }

        //In this case, the desired value is larger than all values in the list.
        if(i == fullListSortedInAscendingOrder.size())
        {
            lb = ListTools.last(fullListSortedInAscendingOrder);
            ub = null;
        }

        return new Integer[]{lb, ub};
    }

    /**
     * Performs basic linear interpolation of y-values given realizations along the x-axis.
     * 
     * @param x1 The value along the x-axis, which can be any {@link Number}.
     * @param y1 Its corresponding y-value, which must be a double.
     * @param x2 Second value along the x-axis.
     * @param y2 Its corresponding y-value.
     * @param x The value for which we want to compute a y-value.
     * @return The y-value.
     */
    public static double linearlyInterpolate(final Number x1,
                                             final double y1,
                                             final Number x2,
                                             final double y2,
                                             final Number x)
    {
        return y1 + (x.doubleValue() - x1.doubleValue()) / (x2.doubleValue() - x1.doubleValue()) * (y2 - y1);
    }

    /**
     * Modifies the given array, rounding all numbers via {@link HNumber#roundDouble(double, int)} and putting them back
     * in place.
     * 
     * @param numbers Numbers to round IN PLACE!
     * @param decimalPlaces The number of places to round it to.
     * @return The provided array is returned by this method after the rounding is performed.
     */
    public static double[] roundAllNumbersInPlace(final double[] numbers, final int decimalPlaces)
    {
        for(int i = 0; i < numbers.length; i++)
        {
            numbers[i] = HNumber.roundDouble(numbers[i], decimalPlaces);
        }
        return numbers;
    }

    /**
     * @param integers The values for which to compute the overall GCD.
     * @return The greatest common divisor over many integers. This is not an efficient algorithm and should only be
     *         used for small data sets.
     */
    public static int computeGCD(final Integer[] integers)
    {
        BigInteger workingGCD = BigInteger.valueOf(integers[0]);
        for(int i = 1; i < integers.length; i++)
        {
            final BigInteger big1 = BigInteger.valueOf(integers[i]);
            workingGCD = workingGCD.gcd(big1);
        }
        return workingGCD.intValue();
    }

    /**
     * @return The smallest {@link Number} in the provided list of {@link Number}s. Relies on
     *         {@link Number#doubleValue()} for the comparison.
     */
    public static Number findSmallest(final Collection<? extends Number> numbers)
    {
        Number smallest = Double.POSITIVE_INFINITY;
        for(final Number num: numbers)
        {
            if(num.doubleValue() < smallest.doubleValue())
            {
                smallest = num;
            }
        }
        return smallest;
    }

    /**
     * Binary search the provided numbers, which must be sorted in ascending order, for the insertion point of the
     * searchNumber provided. The insertion point is the index of the first value equal or larger than the provided
     * searchNumber.
     * 
     * @param numbers Numbers to search.
     * @param searchNumber Number to search for.
     * @return The insertion point into the list if the number were to be put into it in sorted order. This is either
     *         the index of the first number found EQUAL to the searchNumber, or the first number found LARGER than the
     *         searchNumber.
     */
    public static int binarySearchSortedNumbers(final double[] numbers, final double searchNumber)
    {
        //All numbers are 
        if(searchNumber < numbers[0])
        {
            return 0;
        }
        if(searchNumber >= numbers[numbers.length - 1])
        {
            return numbers.length;
        }
        int regionLB = 0;
        int regionUB = numbers.length;
        while(regionUB - regionLB > 1)
        {
            final int halfIndex = (int)((double)(regionUB - regionLB) / 2) + regionLB;
            if(numbers[halfIndex] >= searchNumber)
            {
                regionUB = halfIndex;
            }
            else
            {
                regionLB = halfIndex;
            }
        }
        return regionUB;
    }

//TESTING PURPOSES ONLY!!!
//    public static void main(final String[] args)
//    {
//        final DecimalFormat nf = new DecimalFormat();
//        nf.setMaximumFractionDigits(4);
//        final double posInf = 1d / 0d;
//        final double negInf = -1d / 0d;
//        System.out.println("####>> here: " + nf.format(Double.NaN) + ", " + nf.format(posInf) + ", "
//            + nf.format(negInf));
//        System.out.println("####>>     via formatNumber: " + formatNumber(nf, Double.NaN) + ", "
//            + formatNumber(nf, posInf) + ", " + formatNumber(nf, negInf));
//
//        System.out.println("####>> here floats: " + nf.format(Float.NaN) + ", " + nf.format(Float.POSITIVE_INFINITY));
//        System.out.println("####>>     via formatNumber: " + formatNumber(nf, Float.NaN) + ", "
//            + formatNumber(nf, Float.POSITIVE_INFINITY));
//    }
}
