package ohd.hseb.hefs.utils.tools;

import java.awt.Container;

import javax.swing.JFrame;

import org.jpedal.PdfDecoder;
import org.jpedal.examples.viewer.Viewer;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Tools that wrap org.jpedal LPGL for PDF viewing. See http://lgpl.jpedal.org/pdf-viewer/. The interface to use the
 * free software is quite difficult to work with, so I created these tools as well as the {@link PDFViewerWrapper}
 * class.
 * 
 * @author hankherr
 */
public abstract class PDFTools
{
    public final static String PDF_DEFAULT_PROPERTIES_FILENAME = "jar:/help/pdfHelpProperties.txt";

    /**
     * Views the pdf file at the provided page number using the given properties.
     * 
     * @param fileName Name of file to read. If it starts with 'jar:', then it is assumed to be a system resource. Note
     *            that accessing it via jar file may not work properly.
     * @param pageNumber Page number at which to open the file.
     * @param pdfPropertiesFileName The name of the properties file dictating features of the viewer made available.
     */
    public static void viewPDF(final String fileName, final int pageNumber, final String pdfPropertiesFileName)
    {
        final JFrame frame = new JFrame();
        final Viewer viewer = new Viewer(frame, pdfPropertiesFileName);
        Viewer.exitOnClose = false;
        viewer.setupViewer();
        viewer.openDefaultFileAtPage(fileName, pageNumber);
    }

    /**
     * Calls {@link #viewPDF(String, int, String)} using {@link #PDF_DEFAULT_PROPERTIES_FILENAME} for properties.
     */
    public static void viewPDF(final String fileName, final int pageNumber)
    {
        viewPDF(fileName, pageNumber, PDF_DEFAULT_PROPERTIES_FILENAME);
    }

    /**
     * Views the pdf at the provided bookmark or title.
     * 
     * @param fileName Name of file to read. If it starts with 'jar:', then it is assumed to be a system resource. Note
     *            that accessing it via jar file may not work properly.
     * @param bookmarkOrTitle Bookmark or title to look for. This is first treated as a bookmark. However, the 3rd party
     *            software appears to see MSWord bookmarks as title elements in its outline. So, if it is not found as a
     *            bookmark, I added a search to look at the outline XML to find the page number via the title elements.
     *            If it is not found there, then the viewer will not have its page changed.
     * @param pdfPropertiesFileName The name of the properties file dictating features of the viewer made available.
     */
    public static void viewPDF(final String fileName, final String bookmarkOrTitle, final String pdfPropertiesFileName)
    {
        final JFrame frame = new JFrame();
        final PDFViewerWrapper viewer = new PDFViewerWrapper(frame, pdfPropertiesFileName);
        Viewer.exitOnClose = false;
        viewer.setupViewer();

        //Reads the file the first time in order to get reference information.
        viewer.openDefaultFile(fileName);

        //if a page number was found, reopen the file.
        final int pageNumber = viewer.getReferencePageNumber(bookmarkOrTitle);
        if(pageNumber > 0)
        {
            //Page number was found, so set the page.  
            viewer.openDefaultFileAtPage(fileName, pageNumber);
        }
    }

    /**
     * Calls {@link #viewPDF(String, String, String)} using {@link #PDF_DEFAULT_PROPERTIES_FILENAME} for properties.
     */
    public static void viewPDF(final String fileName, final String bookmark)
    {
        viewPDF(fileName, bookmark, PDF_DEFAULT_PROPERTIES_FILENAME);
    }

    /**
     * Created to expose the {@link PdfDecoder} to external applications and added
     * {@link #getReferencePageNumber(String)}. This should only be needed for the static methods herein.
     * 
     * @author hankherr
     */
    private static class PDFViewerWrapper extends Viewer
    {

        public PDFViewerWrapper(final Container rootContainer, final String preferencesPath)
        {
            super(rootContainer, preferencesPath);
        }

        public PdfDecoder getDecoder()
        {
            return decode_pdf;
        }

        /**
         * Looks for either a bookmark with the given name or a title within the PDF outline with the given name.
         * 
         * @param bookmarkOrTitle Name to look for.
         * @return Page number of the reference or -1 if not found.
         */
        public int getReferencePageNumber(final String bookmarkOrTitle)
        {
            if((getDecoder() == null) || (getDecoder().currentPdfFile == null))
            {
                return -1;
            }

            //Theoretically, bookmarks are found via this call.  However, I can never get it to work.
            final String ref = getDecoder().currentPdfFile.convertNameToRef(bookmarkOrTitle);

            //If the bookmark reference is found, then set the page number.
            if(ref != null)
            {
                return getDecoder().currentPdfFile.convertObjectToPageNumber(ref);
            }
            //Otherwise, look through the outline XML and search the titles.  When the title is found, returned its page number. 
            else
            {
                final Document outline = getDecoder().getOutlineAsXML();
                final NodeList nodeList = outline.getElementsByTagName("title");

                for(int nodeIndex = 0; nodeIndex < nodeList.getLength(); nodeIndex++)
                {
                    final Node titleNode = nodeList.item(nodeIndex).getAttributes().getNamedItem("title");
                    if((titleNode != null) && (titleNode.getNodeValue().equals(bookmarkOrTitle)))
                    {
                        final Node pageNode = nodeList.item(nodeIndex).getAttributes().getNamedItem("page");
                        if(pageNode != null)
                        {
                            try
                            {
                                return Integer.parseInt(pageNode.getNodeValue());
                            }
                            catch(final Exception e)
                            {
                                //If this happens, then the file is corrupted somehow???
                                e.printStackTrace();
                            }
                        }
                    }
                }
            }

            return -1;
        }
    }
}
