package ohd.hseb.hefs.utils.tools;

import java.util.Collection;

import nl.wldelft.util.timeseries.TimeSeriesHeader;
import ohd.hseb.hefs.utils.xml.vars.XMLEnum;
import ohd.hseb.measurement.MeasuringUnit;

import com.google.common.base.Supplier;
import com.google.common.collect.ImmutableList;

public enum ParameterId
{
    MAP(false, false, Type.PRECIPITATION, Extremum.NONE, Accumulation.ACCUMULATIVE, MeasuringUnit.mm), /* \n */
    MAPX(false, false, Type.PRECIPITATION, Extremum.NONE, Accumulation.ACCUMULATIVE, MeasuringUnit.mm), /* \n */
    MAT(false, false, Type.TEMPERATURE, Extremum.NONE, Accumulation.INSTANTANEOUS, MeasuringUnit.degreesCelsius), /* \n */
    TMAX(false, false, Type.TEMPERATURE, Extremum.MAX, Accumulation.ACCUMULATIVE, MeasuringUnit.degreesCelsius), /* \n */
    TAMX(false, false, Type.TEMPERATURE, Extremum.MAX, Accumulation.ACCUMULATIVE, MeasuringUnit.degreesCelsius), /* \n */
    TMIN(false, false, Type.TEMPERATURE, Extremum.MIN, Accumulation.ACCUMULATIVE, MeasuringUnit.degreesCelsius), /* \n */
    TAMN(false, false, Type.TEMPERATURE, Extremum.MIN, Accumulation.ACCUMULATIVE, MeasuringUnit.degreesCelsius), /* \n */
    QIN(false, false, Type.STREAMFLOW, Extremum.NONE, Accumulation.INSTANTANEOUS, MeasuringUnit.cms), /* \n */
    QME(false, false, Type.STREAMFLOW, Extremum.NONE, Accumulation.MEAN, MeasuringUnit.cmsd), /* \n */
    AQME(false, false, Type.STREAMFLOW, Extremum.NONE, Accumulation.MEAN, MeasuringUnit.cmsd), /* \n */
    /* \n */
    FMAP(true, false, Type.PRECIPITATION, Extremum.NONE, Accumulation.ACCUMULATIVE, MeasuringUnit.mm), /* \n */
    FMAT(true, false, Type.TEMPERATURE, Extremum.NONE, Accumulation.INSTANTANEOUS, MeasuringUnit.degreesCelsius), /* \n */
    TFMX(true, false, Type.TEMPERATURE, Extremum.MAX, Accumulation.ACCUMULATIVE, MeasuringUnit.degreesCelsius), /* \n */
    TFMN(true, false, Type.TEMPERATURE, Extremum.MIN, Accumulation.ACCUMULATIVE, MeasuringUnit.degreesCelsius), /* \n */
    QINE(true, false, Type.STREAMFLOW, Extremum.NONE, Accumulation.INSTANTANEOUS, MeasuringUnit.cms), /* \n */
    SQIN(true, true, Type.STREAMFLOW, Extremum.NONE, Accumulation.INSTANTANEOUS, MeasuringUnit.cms), /* \n */
    SQME(true, true, Type.STREAMFLOW, Extremum.NONE, Accumulation.MEAN, MeasuringUnit.cmsd), /* \n */
    STG(false, false, Type.RIVERSTAGE, Extremum.NONE, Accumulation.INSTANTANEOUS, MeasuringUnit.meters), /* \n */
    SSTG(true, false, Type.RIVERSTAGE, Extremum.NONE, Accumulation.INSTANTANEOUS, MeasuringUnit.meters);

    private final boolean _forecast;
    private final boolean _simulated;
    public final Type _type;
    public final Extremum _extremum;
    public final Accumulation _accumulation;
    public final MeasuringUnit _defaultUnit;

    private ParameterId(final boolean forecast,
                        final boolean simulated,
                        final Type type,
                        final Extremum extremum,
                        final Accumulation accumulation,
                        final MeasuringUnit defaultUnit)
    {
        _forecast = forecast;
        _simulated = simulated;
        _type = type;
        _extremum = extremum;
        _accumulation = accumulation;
        _defaultUnit = defaultUnit;
    }

    public static ParameterId get(final Type type, final boolean forecast)
    {
        return get(type, forecast, Extremum.NONE);
    }

    public static ParameterId get(final Type type, final boolean forecast, Extremum extremum)
    {
        if(extremum == null)
        {
            extremum = Extremum.NONE;
        }
        for(final ParameterId parameter: values())
        {
            if(parameter._forecast == forecast && parameter._type.equals(type) && parameter._extremum.equals(extremum))
            {
                return parameter;
            }
        }
        throw new IllegalArgumentException("No parameter with specified values.");
    }

    public static ParameterId of(final TimeSeriesHeader header)
    {
        return valueOf(header.getParameterId());
    }

    public ParameterId dropExtremum()
    {
        return get(_type, _forecast, Extremum.NONE);
    }

    public ParameterId dropForecast()
    {
        return get(_type, false, _extremum);
    }

    public boolean isForecast()
    {
        return _forecast;
    }

    public boolean isObserved()
    {
        return !_forecast;
    }

    public boolean isSimulated()
    {
        return _simulated;
    }

    public boolean isAccumulativeOrMean()
    {
        return _accumulation.equals(Accumulation.ACCUMULATIVE) || _accumulation.equals(Accumulation.MEAN);
    }

    public boolean isAccumulative()
    {
        return _accumulation.equals(Accumulation.ACCUMULATIVE);
    }

    public boolean isMean()
    {
        return _accumulation.equals(Accumulation.MEAN);
    }

    public boolean isInstantaenous()
    {
        return _accumulation.equals(Accumulation.INSTANTANEOUS);
    }

    public boolean isPrecipitation()
    {
        return _type.equals(Type.PRECIPITATION);
    }

    public boolean isTemperature()
    {
        return _type.equals(Type.TEMPERATURE);
    }

    public boolean isStreamflow()
    {
        return _type.equals(Type.STREAMFLOW);
    }

    public boolean isRiverStage()
    {
        return _type.equals(Type.RIVERSTAGE);
    }

    public boolean isMax()
    {
        return _extremum.equals(Extremum.MAX);
    }

    public boolean isMin()
    {
        return _extremum.equals(Extremum.MIN);
    }

    public static enum Type
    {
        NONE, PRECIPITATION, TEMPERATURE, STREAMFLOW, RIVERSTAGE;

        public String printName()
        {
            final String name = toString();
            return name.substring(0, 1) + name.substring(1).toLowerCase();
        }

        public static ImmutableList<String> makePrintNameList(final Collection<Type> types)
        {
            final ImmutableList.Builder<String> builder = ImmutableList.builder();
            for(final Type type: types)
            {
                builder.add(type.printName());
            }
            return builder.build();
        }

        public static class Variable extends XMLEnum<Type>
        {
            public static String XML_TAG = "dataType";

            public Variable(final Type value)
            {
                super(Type.class, XML_TAG, value);
            }

            public Variable(final String tag)
            {
                super(Type.class, tag);
            }

            public Variable(final String tag, final Type value)
            {
                super(Type.class, tag, value);
            }

            public static final Supplier<Variable> DEFAULT_FACTORY = new Supplier<Variable>()
            {
                @Override
                public Variable get()
                {
                    return new Variable(XML_TAG);
                }
            };
        }
    }

    public static enum Extremum
    {
        NONE, MAX, MIN;
    }

    public static enum Accumulation
    {
        ACCUMULATIVE, INSTANTANEOUS, MEAN;
    }
}
