package ohd.hseb.hefs.utils.tools;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;

import com.google.common.base.Function;
import com.google.common.base.Predicate;

public class PrimitiveDoubleTools
{
    /**
     * Creates a primitive double array from a collection
     * 
     * @param base the base collection to draw from
     * @return a primitive double array
     */
    public static double[] newDoubleArray(final Collection<Double> base)
    {
        final double[] array = new double[base.size()];
        final Iterator<Double> iter = base.iterator();
        for(int i = 0; i < array.length; i++)
        {
            array[i] = iter.next();
        }
        return array;
    }

    /**
     * Maps a collection to a primitive double array.
     * 
     * @param function the mapping function
     * @param base the original collection
     * @return a double array consisting of {@code function} applied to every element of {@code collection} in order
     */
    public static <A> double[] mapToDoubleArray(final Function<A, Double> function, final Collection<? extends A> base)
    {
        final double[] array = new double[base.size()];
        final Iterator<? extends A> iter = base.iterator();
        for(int i = 0; i < array.length; i++)
        {
            array[i] = function.apply(iter.next());
        }
        return array;
    }

    public static <A> ArrayList<A> mapToArrayList(final Function<Double, A> function, final double[] inputs)
    {
        final ArrayList<A> list = newArrayList();
        for(final double input: inputs)
        {
            list.add(function.apply(input));
        }
        return list;
    }

    /**
     * Splits an array into two based on a predicate.
     * 
     * @param predicate the predicate to partition on
     * @param input the array to split
     * @return An array of two arrays. The first contains all values in {@code input} that failed {@code predicate}, and
     *         the second all values that passed
     */
    public static double[][] partition(final Predicate<Double> predicate, final double[] input)
    {
        // TODO Does it in 2 passes, not sure if the best implementation.

        // Count successes to size array.
        int pass = 0;
        final int[] targetArray = new int[input.length]; // Which array to place in.
        for(int i = 0; i < input.length; i++)
        {
            if(predicate.apply(input[i]))
            {
                pass++;
                targetArray[i] = 1;
            }
        }

        // Place in proper arrays.
        final double[][] result = new double[2][];
        result[0] = new double[input.length - pass];
        result[1] = new double[pass];
        final int[] index = new int[2];
        for(int i = 0; i < input.length; i++)
        {
            result[targetArray[i]][index[targetArray[i]]] = input[i];
            index[targetArray[i]]++;
        }

        return result;
    }

    /**
     * Splits an array by {@code boundary}. All values {@code >= boundary} in {@code input} are placed in the first
     * array, and the rest in the second.
     * 
     * @param boundary the boundary to partition on
     * @param input the array to partition
     * @return the partitioned array
     */
    public static double[][] partition(final double boundary, final double[] input)
    {
        return partition(lessThan(boundary), input);
    }

    /**
     * A predicate testing for doubles less than {@code boundary}.
     * 
     * @param boundary the boundary value
     */
    public static Predicate<Double> lessThan(final double boundary)
    {
        return new Predicate<Double>()
        {
            @Override
            public boolean apply(final Double input)
            {
                return input < boundary;
            }
        };
    }

    /**
     * @return A {@link HashSet} containing the unique values within the provided input.
     */
    public static HashSet<Double> newHashSetFromArray(final double[] input)
    {
        final HashSet<Double> result = newHashSet();
        for(final double value: input)
        {
            result.add(value);
        }
        return result;
    }
}
