package ohd.hseb.hefs.utils.tools;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Properties;
import java.util.regex.Pattern;

import ohd.hseb.hefs.utils.adapter.PropertyVariable;

import com.google.common.collect.Lists;

/**
 * Tools associated with {@link Properties}.
 * 
 * @author hankherr
 */
public abstract class PropertiesTools
{

    /**
     * @param allProperties The {@link Properties} to search.
     * @param suffix Suffix to look for.
     * @return All property keys for which the key, assumed to be a String, ends in the provided suffix. This will throw
     *         a {@link ClassCastException} if a key is not a String.
     */
    public static List<String> retrieveAllKeysEndingWithSuffix(final Properties allProperties, final String suffix)
    {
        final List<String> results = Lists.newArrayList();
        for(final Object keyObj: allProperties.keySet())
        {
            final String key = (String)keyObj;
            if(key.endsWith(suffix))
            {
                results.add(key);
            }
        }
        return results;
    }

    /**
     * See the main method here for examples.
     * 
     * @param allProperties {@link Properties} to search.
     * @param regExPattern A regular expression String pattern.
     * @return All property keys that match the given pattern.
     */
    public static List<String> retrieveallKeysMatchingPattern(final Properties allProperties, final String regExPattern)
    {
        final List<String> results = Lists.newArrayList();
        for(final Object keyObj: allProperties.keySet())
        {
            final String key = (String)keyObj;
            if(Pattern.matches(regExPattern, key))
            {
                results.add(key);
            }
        }
        return results;
    }

    /**
     * Works with instances of {@link PropertyVariable}. This calls {@link PropertyVariable#read(Properties)} for each
     * property variable and then, if no exception was thrown when reading, it calls the
     * {@link PropertyVariable#processAfterSuccessfulRead()} for each property variable. Properties are read in their
     * order within the collection.
     * 
     * @param vars The property variable instances to read in.
     * @param props The properties from which to read the variable values.
     * @throws Exception From the {@link PropertyVariable#read(Properties)} and
     *             {@link PropertyVariable#processAfterSuccessfulRead()} methods.
     */
    public static void readPropertyVariables(final Collection<PropertyVariable> vars, final Properties props) throws Exception
    {
        //Read
        for(final PropertyVariable var: new ArrayList<PropertyVariable>(vars))
        {
            var.read(props);
        }

        //Post-process
        for(final PropertyVariable var: new ArrayList<PropertyVariable>(vars))
        {
            var.processAfterSuccessfulRead();
        }
    }

    public static void main(final String[] args)
    {
        final Properties props = new Properties();
        props.setProperty("aaaa", "1");
        props.setProperty("aaaaSUFFIX", "1");
        props.setProperty("aaaaSUFFIX.part2", "1");
        props.setProperty("aaaaSUFFIXpart2", "1");
        props.setProperty("bbbb", "1");

        retrieveallKeysMatchingPattern(props, "^.*SUFFIX|^.*SUFFIX\\..*");

    }
}
