package ohd.hseb.hefs.utils.tools;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.utils.dist.types.ContinuousDist;

public class SamplingTools {
	
	public static enum SamplingTechnique {RANDOM, STRATIFIED_RANDOM, QUANTILE}

	public SamplingTools() {
		// TODO Auto-generated constructor stub
	}

	/**
	 * Generate a sample point from the uniform distribution U(0,1) with the use of the Stratified Random Sampling
	 * technique. Assume a sample of size "sampleSize" needs to be generated. For a given integer "index" between 0 and
	 * sampleSize-1, inclusive, the method returns a value r1 in [a,b) using equation r1 = a+(b-a)*r0, where a =
	 * index/sampleSize, b = (index+1)/sampleSize, and r0 is a random number drawn from U(0,1).
	 * 
	 * @param sampleSize The size of the random sample to be generated.
	 * @param index The position of a sub-interval of (0,1), for which a random sample point will be generated. Can
	 *            range from 0 to sampleSize - 1.
	 */
	public static double generateStratifiedRandomSampleProb(final int sampleSize, final int index)
	{
	    return (double)index / (double)sampleSize + ThreadedRandomTools.getRandomNumGen().nextDouble() / sampleSize;
	}

	/**
	 * Pass through to {@link #generateStratifiedRandomSample(ContinuousDist, int, int, boolean)} providing false for
	 * the log message flag.
	 */
	public static double generateStratifiedRandomSample(final ContinuousDist dist,
	                                                    final int sampleSize,
	                                                    final int index)
	{
	    return generateStratifiedRandomSample(dist, sampleSize, index, null, null);
	}

	/**
	 * Calls {@link #generateStratifiedRandomSampleProb(int, int)} and samples a quantile from the provided
	 * {@link ContinuousDist} based on that probability.
	 * 
	 * @param dist The distribution from which to sample.
	 * @param sampleSize The total number of stratified samples to be generated
	 * @param index The position of a sub-interval of (0,1), for which a random sample point will be generated. Can
	 *            range from 0 to sampleSize - 1.
	 * @param thresholdToIncludeLogMessage The threshold such that if the probability is greater than it a log message
	 *            is produced. Null implies no messages. This is for #58561.
	 * @param messagePrefix The prefix for that log message.
	 * @return The quantile
	 */
	public static double generateStratifiedRandomSample(final ContinuousDist dist,
	                                                    final int sampleSize,
	                                                    final int index,
	                                                    final Double thresholdToIncludeLogMessage,
	                                                    final String messagePrefix)
	{
	    final double prob = generateStratifiedRandomSampleProb(sampleSize, index);
	    final double quan = dist.functionInverseCDF(prob);
	    if((thresholdToIncludeLogMessage != null) && (prob >= thresholdToIncludeLogMessage))
	    {
	        ThreadedRandomTools.LOG.info(messagePrefix + "In generating a stratified random sample, the randomly generated probability is "
	            + prob + " yielding quantile " + quan);
	    }
	    return quan;
	}

	/**
	 * Pass through to {@link #sampleFromValues(List, int, boolean, boolean)} passing in false for the logging flag.
	 */
	public static List<Double> sampleFromValues(final List<Float> valuesToSample,
	                                            final int numberOfValues,
	                                            final SamplingTools.SamplingTechnique samplingTechnique)
	{
	    return sampleFromValues(valuesToSample, numberOfValues, samplingTechnique, null, null);
	}

	/**
	 * @param valuesToSample {@link List} of {@link Float} instances from which to sample.
	 * @param numberOfValues Number of values to sample.
	 * @param stratifiedRandomSampling True for stratified random sampling, false for simple random sampling.
	 * @param thresholdToIncludeLogMessage The threshold such that if the probability is greater than it a log message
	 *            is produced. Null implies no messages. This is for #58561.
	 * @param messagePrefix The prefix for that log message.
	 * @return A {@link List} of {@link Double} resampled from the provided values.
	 */
	public static List<Double> sampleFromValues(final List<Float> valuesToSample,
	                                            final int numberOfValues,
	                                            final SamplingTools.SamplingTechnique samplingTechnique,
	                                            final Double thresholdToIncludeLogMessage,
	                                            final String messagePrefix)
	{
		if (numberOfValues < 0) 
		{
        	throw new IllegalArgumentException("numberOfValues - The Number of values to sample could not be Zero or less");
		}
	    final List<Double> quaValues = new ArrayList<Double>(numberOfValues);
	
	    for(int i = 0; i < numberOfValues; i++)
	    {
	        double prob = 0.0d;
	        switch(samplingTechnique) 
	        { 
	            case STRATIFIED_RANDOM: 
	                prob = generateStratifiedRandomSampleProb(numberOfValues, i);
	                break; 
	            case RANDOM:
	            	prob = ThreadedRandomTools.getRandomNumGen().nextDouble();
	                break; 
	            case QUANTILE: 
	                prob = generateQuantileSampleProb(numberOfValues, i);
	                break; 
	        }
	
	        //This affects the ensemble generation test output.  For some reason, it does not affect the adapter output,
	        //but maybe I'm just not testing EPT yet (???).
	        if (valuesToSample !=null && !valuesToSample.isEmpty() ) {
	        
		        final int indexToUse = (int)(prob * valuesToSample.size());
		
		        final double quan = valuesToSample.get(indexToUse).doubleValue();
		        quaValues.add(quan);
		        if((thresholdToIncludeLogMessage != null) && (prob >= thresholdToIncludeLogMessage))
		        {
		            ThreadedRandomTools.LOG.info(messagePrefix
		                + "In picking a random sample from set of values, the randomly generated probability is " + prob
		                + " yielding quantile " + quan);
		        }
	        } 
	        else
	        {
	        	throw new IllegalArgumentException("valuesToSample {@link List} of {@link Float} instances from which to sample"
                        + " could not be emtpy ");
	        }
	    }
	
	    return quaValues;
	}

	
    public static double generateQuantileSampleProb(int sampleSize, int index)
    {
        return (double)(index + 1)/ (double)(sampleSize + 1);
    }
   
    /**
     * 
     * @param dist The distribution from which to sample
     * @param sampleSize The total number of Quantile samples to be generated
     * @param index The position of a sub-interval of (0,1), for which a random sample point will be generated. Can
	 *            range from 0 to sampleSize - 1.

     * @param thresholdToIncludeLogMessage The threshold such that if the probability is greater than it a log message
	 *            is produced. Null implies no messages. This is for #58561.
	 * @param messagePrefix The prefix for that log message.
	 * @return A Double resampled from the provided values.
     */
    public static double generateQuantileSample(final ContinuousDist dist,
                                                final int sampleSize,
                                                final int index,
                                                final Double thresholdToIncludeLogMessage,
                                                final String messagePrefix)
    {
    	
		if (sampleSize <= 0) 
		{
        	throw new IllegalArgumentException("sampleSize - The Number of values to sample could not be Zero or less");
		}
		if (index < 0) 
		{
        	throw new IllegalArgumentException("index - The position of the sub-interval to sample could not be less than Zero");
		}
        final double prob = generateQuantileSampleProb(sampleSize, index);
        final double quan = dist.functionInverseCDF(prob);
        if((thresholdToIncludeLogMessage != null) && (prob >= thresholdToIncludeLogMessage))
        {
        	ThreadedRandomTools.LOG.info(messagePrefix + "In generating a Quantile sample, the randomly generated probability is "
                + prob + " yielding quantile " + quan);
        }
        return quan;
    }

	/**
	 * Pass through to {@link #generateRandomSample(ContinuousDist, boolean)} passing in false for the logging flag.
	 */
	public static double generateRandomSample(final ContinuousDist dist)
	{
	    return generateRandomSample(dist, null, null);
	}

	/**
	 * @param dist The distribution to sample.
	 * @param thresholdToIncludeLogMessage The threshold such that if the probability is greater than it a log message
	 *            is produced. Null implies no messages. This is for #58561.
	 * @param messagePrefix The prefix for that log message.
	 * @return A single random sample from the provided {@link ContinuousDist} making use of the
	 *         {@link ThreadedRandomTools#getRandomNumGen()}.
	 */
	public static double generateRandomSample(final ContinuousDist dist,
	                                          final Double thresholdToIncludeLogMessage,
	                                          final String messagePrefix)
	{
	    final double prob = ThreadedRandomTools.getRandomNumGen().nextDouble();
	    final double quan = dist.functionInverseCDF(prob);
	    if((thresholdToIncludeLogMessage != null) && (prob >= thresholdToIncludeLogMessage))
	    {
	        ThreadedRandomTools.LOG.info(messagePrefix + "In generating a random sample, the randomly generated probability is " + prob
	            + " yielding quantile " + quan);
	    }
	    return quan;
	}

	/**
	 * wraps the three different types of generate*Sample methods, taking a SamplingTechnique and calling the appropriate method for the value to return
	 * 
	 * @param dist The distribution to sample.
	 * @param sampleSize The total number of stratified samples to be generated
	 * @param index The position of a sub-interval of (0,1), for which a random sample point will be generated. Can
	 *            range from 0 to sampleSize - 1.
	 * @param samplingTechnique
	 * @return
	 */
	public static double generateSample(final ContinuousDist dist,
	        							final int sampleSize,
	        							final int index,
	        							final Double thresholdToIncludeLogMessage,
	                                    final String messagePrefix,
	        							final SamplingTechnique samplingTechnique)
	{
		double result = 0.0d;
	    switch(samplingTechnique) 
	    { 
	        case STRATIFIED_RANDOM: 
	            result = generateStratifiedRandomSample(dist, sampleSize, index, thresholdToIncludeLogMessage, messagePrefix);
	            break; 
	        case RANDOM: 
	            result = generateRandomSample(dist);
	            break; 
	        case QUANTILE: 
	            result = generateQuantileSample(dist, sampleSize, index, null, null);
	            break; 
	    }
		return result;
	}

}
