package ohd.hseb.hefs.utils.tools;

import java.text.BreakIterator;
import java.util.ArrayList;
import java.util.List;

/**
 * Tools that can be used with Strings, lists of strings, or arrays of strings.
 * 
 * @author hank.herr
 */
public abstract class StringTools
{

    /**
     * @param str String to truncate.
     * @param maxAllowedLength The maximum allowed length in characters.
     * @return Results of {@link String#substring(int, int)} with the end being either the length of the provided string
     *         or the provided maxAllowedLength, whichever is shorter.
     */
    public static String truncateString(final String str, final int maxAllowedLength)
    {
        return str.substring(0, Math.min(str.length(), maxAllowedLength));
    }

    /**
     * @param first String to check
     * @param second String to check
     * @param caseSensitive Case sensitive flag
     * @return True if the two strings are equal. A null check is also performed, so that this will return true if both
     *         are null or if both strings equal (based on caseSensitive flag).
     */
    public static boolean checkForFullEqualityOfStrings(final String first,
                                                        final String second,
                                                        final boolean caseSensitive)
    {
        if((first != null) && (second == null))
        {
            return false;
        }
        if((first == null) && (second != null))
        {
            return false;
        }
        if(first == null)
        {
            return true; //we've checked all we need to.
        }
        if(caseSensitive)
        {
            return first.equals(second);
        }
        return first.equalsIgnoreCase(second);
    }

    /**
     * @param desiredStrings Choices you wish to have.
     * @param acceptableStrings List of all default choices.
     * @return Intersection of the two lists.
     */
    public static String[] buildArrayOfStringsFromAcceptableArray(final String[] desiredStrings,
                                                                  final String[] acceptableStrings)
    {
        if(desiredStrings == null || desiredStrings.length <= 0)
        {
            return null;
        }

        if(acceptableStrings == null || acceptableStrings.length <= 0)
        {
            return null;
        }

        //Loop through the input choices.
        final List<String> intersection = new ArrayList<String>();
        for(final String type: desiredStrings)
        {
            //Then loop through the default choice.
            for(final String element: acceptableStrings)
            {
                //If the input choice equals the default choice, add it to validOptions.
                if(type.equals(element))
                {
                    intersection.add(type);
                    break;
                }
            }
        }

        return intersection.toArray(new String[intersection.size()]);
    }

    /**
     * Reformats a string where lines that are longer than <tt>width</tt> are split apart at the earliest wordbreak or
     * at maxLength, whichever is sooner. If the width specified is less than 5 or greater than the input Strings length
     * the string will be returned as is.
     * <p/>
     * Please note that this method can be lossy - trailing spaces on wrapped lines may be trimmed.<br>
     * <br>
     * Copied from http://www.java2s.com/Code/Java/Data-Type/WordWrap.htm.
     * 
     * @param input the String to reformat.
     * @param width the maximum length of any one line.
     * @return a new String with reformatted as needed.
     */
    public static String wordWrap(final String input, final int width)
    {
        // protect ourselves
        if(input == null)
        {
            return "";
        }
        else if(width < 5)
        {
            return input;
        }
        else if(width >= input.length())
        {
            return input;
        }

        final StringBuilder buf = new StringBuilder(input);
        boolean endOfLine = false;
        int lineStart = 0;

        for(int i = 0; i < buf.length(); i++)
        {
            if(buf.charAt(i) == '\n')
            {
                lineStart = i + 1;
                endOfLine = true;
            }

            // handle splitting at width character
            if(i > lineStart + width - 1)
            {
                if(!endOfLine)
                {
                    final int limit = i - lineStart - 1;
                    final BreakIterator breaks = BreakIterator.getLineInstance();
                    breaks.setText(buf.substring(lineStart, i));
                    int end = breaks.last();

                    // if the last character in the search string isn't a space,
                    // we can't split on it (looks bad). Search for a previous
                    // break character
                    if(end == limit + 1)
                    {
                        if(!Character.isWhitespace(buf.charAt(lineStart + end)))
                        {
                            end = breaks.preceding(end - 1);
                        }
                    }

                    // if the last character is a space, replace it with a \n
                    if(end != BreakIterator.DONE && end == limit + 1)
                    {
                        buf.replace(lineStart + end, lineStart + end + 1, "\n");
                        lineStart = lineStart + end;
                    }
                    // otherwise, just insert a \n
                    else if(end != BreakIterator.DONE && end != 0)
                    {
                        buf.insert(lineStart + end, '\n');
                        lineStart = lineStart + end + 1;
                    }
                    else
                    {
                        buf.insert(i, '\n');
                        lineStart = i + 1;
                    }
                }
                else
                {
                    buf.insert(i, '\n');
                    lineStart = i + 1;
                    endOfLine = false;
                }
            }
        }

        return buf.toString();
    }

    /**
     * This is a slow implementation that does not use {@link StringBuffer} -- use sparingly.
     * 
     * @return Returns the given string with the first letter in uppercase and the rest in lowercase.
     */
    public static String capitalize(final String input)
    {
        return input.substring(0, 1).toUpperCase() + input.substring(1).toLowerCase();
    }

    /**
     * @param text Text to convert.
     * @return An html version of the string. This will replace all '\n' with line breaks and return the string inside
     *         an html element.
     */
    public static String htmlizeString(final String text)
    {
        final String htmlString = text.replaceAll("\n", "<br>");
        return "<html>" + htmlString + "</html>";
    }

    /**
     * @param obj Object to return as a String.
     * @return Either the object's toString or null if the object is null.
     */
    public static String nullOrStringValue(final Object obj)
    {
        if(obj != null)
        {
            return obj.toString();
        }
        return null;
    }

    /**
     * As {@link String#replace(char, char)}, but using a buffer.
     */
    public static void replace(final StringBuffer buffer, final String toBeReplaced, final String replaceValue)
    {
        final int index = buffer.indexOf(toBeReplaced);
        if(index >= 0)
        {
            buffer.replace(index, index + toBeReplaced.length(), replaceValue);
        }
    }

//XXX At one point, I thought about using the tools below in HCalendar to make StringBuffer replacements for date components.
//When I discovered SimpleDateFormat was MUCH faster, I decided to just change HCalendar to use SimpleDateFormat.
//These tools replace strings with numbers in a StringBuffer.
//    /**
//     * As {@link String#replace(char, char)}, but using a buffer and the replacement value is determined by a
//     * {@link Number}.
//     * 
//     * @param buffer
//     * @param toBeReplaced
//     * @param value
//     * @param integerDigits The number of integer digits
//     */
//    public static void replace(final StringBuffer buffer,
//                               final String toBeReplaced,
//                               final Number value,
//                               final int integerDigits)
//    {
//        int number = value.intValue();
//        final char[] replaceValue = new char[integerDigits];
//        for(int workingDigit = integerDigits - 1; workingDigit >= 0; workingDigit--)
//        {
//            replaceValue[workingDigit] = (char)('0' + number % 10);
//            number = (int)(number / 10d);
//        }
//
//        final int index = buffer.indexOf(toBeReplaced);
//        if(index >= 0)
//        {
//            buffer.replace(index, index + toBeReplaced.length(), String.valueOf(replaceValue));
//        }
//    }
//
//    /**
//     * As {@link String#replace(char, char)}, but using a buffer and the replacement value is determined by a
//     * {@link Number}. The formatting done here is slightly faster than {@link NumberFormat}. For example, on the mac,
//     * converting 1,000,000 millis via {@link HCalendar#buildDateStr(long, String)} took about 4-4.5 seconds using the
//     * method {@link #replace(StringBuffer, String, String)} with a {@link NumberFormat} generating replacement value,
//     * but took only about 3-3.5 seconds using this method. Note that this method is dealing with fraction digits, which
//     * are not needed for date string replacement. See {@link #replace(StringBuffer, String, Number, int)}.
//     * 
//     * @param buffer
//     * @param toBeReplaced
//     * @param value
//     * @param integerDigits The number of integer digits
//     * @param fractionDigits The number of fraction digits
//     */
//    public static void replace(final StringBuffer buffer,
//                               final String toBeReplaced,
//                               final Number value,
//                               final int integerDigits,
//                               final int fractionDigits)
//    {
//        //All of this is to avoid number formatting, which is slow.  Can I make it faster?
//
//        //Number to put in as a string
//        final String numberStr = Double.toString(HNumber.roundDouble(value.doubleValue(), fractionDigits));
//        final int decimalIndex = Math.max(numberStr.indexOf('.'), numberStr.length());
//
//        //Replacement string length
//        int replacementStrLength = integerDigits;
//        if(fractionDigits > 0)
//        {
//            replacementStrLength += 1 + fractionDigits;
//        }
//
//        //Replacement
//        final char[] replaceValue = new char[replacementStrLength];
//        for(int i = 0; i < replaceValue.length; i++)
//        {
//            if(i < integerDigits)
//            {
//                if(integerDigits - i > decimalIndex)
//                {
//                    replaceValue[i] = '0';
//                }
//                else
//                {
//                    replaceValue[i] = numberStr.charAt(decimalIndex - (integerDigits - i));
//                }
//            }
//            else if(i == integerDigits)
//            {
//                replaceValue[i] = '.';
//            }
//            else
//            {
//                replaceValue[i] = numberStr.charAt(decimalIndex + (i - integerDigits));
//            }
//        }
//
//        final int index = buffer.indexOf(toBeReplaced);
//        if(index >= 0)
//        {
//            buffer.replace(index, index + toBeReplaced.length(), String.valueOf(replaceValue));
//        }
//    }
}
