package ohd.hseb.hefs.utils.tools;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.TimeZone;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.xml.parsers.SAXParser;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;

import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.io.IOUtils;

/**
 * Package of tools related to commons.compress Tar Archive library. Only use this tool with tar streams returned by its
 * {@link #openTarArchiveForReading(File, boolean)}, {@link #openTarArchiveForReading(File, boolean, boolean)}, or
 * {@link #openTarArchiveForWriting(File, boolean)}. These methods return an object of type
 * {@link OHDWrapperTarArchiveInputStream}, which is the only kind of stream these methods are designed to work with.<br>
 * <br>
 * Note that {@link TarArchiveInputStream} instances are meant to be read in order, meaning that the entries are
 * traversed in the order in which they are packaged. If you need to reset the archive and start from scratch, you must
 * close and reopen the stream. The static methods herein mean to allow navigating the entries by going to next,
 * {@link #gotoNextEntry(TarArchiveInputStream)} and
 * {@link #readXMLFromTarStream(TarArchiveInputStream, boolean, String, XMLReadable, boolean)} in particular, always
 * check the current entry before going to the next in order to find the entry to read.
 * 
 * @author hankherr
 */
public abstract class TarTools
{

    /**
     * @param tarStream
     * @return The next entry in the stream. This simply wraps {@link TarArchiveInputStream#getNextTarEntry()} as a
     *         convenience to allow for consistency without needing to directly interact with
     *         {@link TarArchiveInputStream}.
     */
    public static TarArchiveEntry gotoNextEntry(final TarArchiveInputStream tarStream) throws IOException
    {
        return tarStream.getNextTarEntry();
    }

    /**
     * Goes to the archive entry within the provided {@link TarArchiveInputStream} that has the provided entryName
     * starting with the current entry and moving forward; i.e., if the desired entry is the current entry, nothing is
     * done to the stream. Note that the tar stream cannot be reset, so if you want to start from the beginning, you
     * must reopen the stream.
     * 
     * @param tarStream Stream to scan.
     * @param entryName Name of entry to file.
     * @return {@link TarArchiveEntry} found or null if not found.
     * @throws IOException
     */
    public static TarArchiveEntry gotoTarArchiveEntry(final TarArchiveInputStream tarStream, final String entryName) throws IOException
    {
        //Check the current entry first.  Go nowhere if we are already there.
        if(isCurrentEntry(tarStream, entryName))
        {
            return ((OHDWrapperTarArchiveInputStream)tarStream).getOHDCurrentEntry();
        }

        //Move forward until we find it.
        TarArchiveEntry entry = tarStream.getNextTarEntry();
        while(entry != null)
        {
            if(entry.getName().equals(entryName))
            {
                return entry;
            }
            entry = tarStream.getNextTarEntry();
        }
        return null;
    }

    /**
     * Calls {@link #openTarArchiveForReading(File, boolean, boolean)} passing in false for preventClosing. This means
     * that if XML is parsed within the given stream, it will be closed by the XML parser.
     */
    public static TarArchiveInputStream openTarArchiveForReading(final File tarFile, final boolean gzipped) throws IOException
    {
        return openTarArchiveForReading(tarFile, gzipped, false);
    }

    /**
     * Opens up a tar file and returns a {@link TarArchiveInputStream} for it, ready for reading.<br>
     * <br>
     * IMPORTANT NOTE: If preventClosing is true, then the returned stream {@link NonClosingTarArchiveInputStream} will
     * NOT close when its {@link NonClosingTarArchiveInputStream#close()} is called. This is to avoid XML parsers
     * closing the stream. To close it, call {@link OHDWrapperTarArchiveInputStream#reallyClose()}.
     * 
     * @param tarFile The tar file to open.
     * @param gzipped True if the tar file is also gzipped; false if not.
     * @param preventClosing If true, then a {@link NonClosingTarArchiveInputStream} object is returned which will not
     *            close when {@link NonClosingTarArchiveInputStream#close()} is called; only when
     *            {@link NonClosingTarArchiveInputStream#reallyClose()} is called. This can then be used for called
     *            {@link #readXMLFromTarStream(TarArchiveInputStream, boolean, String, XMLReadable, boolean)} when will
     *            close a {@link TarArchiveInputStream} upon completion parsing XML.
     * @return {@link TarArchiveInputStream} for the file.
     * @throws IOException Standard {@link IOException} returned with file I/O.
     */
    public static TarArchiveInputStream openTarArchiveForReading(final File tarFile,
                                                                 final boolean gzipped,
                                                                 final boolean preventClosing) throws IOException
    {
        InputStream inputStream = new FileInputStream(tarFile);
        if(gzipped)
        {
            inputStream = new GZIPInputStream(inputStream);
        }
        return new OHDWrapperTarArchiveInputStream(inputStream, preventClosing);
    }

    /**
     * Opens up a tar file and returns a {@link TarArchiveOutputStream} for it, ready for writing.
     * 
     * @param tarFile The tar file to open.
     * @param gzipped True if the tar file is also gzipped; false if not.
     * @return {@link TarArchiveOutputStream} for the file. Note that it must be flushed and closed upon completion of
     *         writing.
     * @throws IOException
     */
    public static TarArchiveOutputStream openTarArchiveForWriting(final File tarFile, final boolean gzipped) throws IOException
    {
        OutputStream outputStream = new FileOutputStream(tarFile);
        if(gzipped)
        {
            outputStream = new GZIPOutputStream(outputStream);
        }
        final TarArchiveOutputStream tarStream = new TarArchiveOutputStream(outputStream);
        return tarStream;
    }

    /**
     * Assumes the XML is NOT fastInfoset.
     * 
     * @param tarFile Gzipped tar file to read.
     * @param entryName Name of entry.
     * @param readable {@link XMLReadable} which provides a reader to do the reading.
     * @throws Exception
     */
    public static void readXMLFromGzippedTarFile(final File gzippedTarFile,
                                                 final String entryName,
                                                 final XMLReadable readable) throws Exception
    {
        readXMLFromTarFile(gzippedTarFile, true, entryName, readable, false);
    }

    /**
     * Assumes the XML is in fastInfoset.
     * 
     * @param tarFile Gzipped tar file to read.
     * @param entryName Name of entry.
     * @param readable {@link XMLReadable} which provides a reader to do the reading.
     * @throws Exception
     */
    public static void readFastInfosetXMLFromGzippedTarFile(final File gzippedTarFile,
                                                            final String entryName,
                                                            final XMLReadable readable) throws Exception
    {
        readXMLFromTarFile(gzippedTarFile, true, entryName, readable, true);
    }

    /**
     * The tar file stream is closed when done XML parsing.
     * 
     * @param tarFile Tar file to read.
     * @param gzipped Indicates if the file is also gzipped.
     * @param entryName Name of the entry to read, which must be XML or fastInfoset XML.
     * @param readable The {@link XMLReadable} to use to read the XML.
     * @param fastInfoset Indicates if the XML should be read in fastInfoset.
     * @throws Exception
     */
    public static void readXMLFromTarFile(final File tarFile,
                                          final boolean gzipped,
                                          final String entryName,
                                          final XMLReadable readable,
                                          final boolean fastInfoset) throws Exception
    {
        final TarArchiveInputStream tarStream = TarTools.openTarArchiveForReading(tarFile, gzipped);

        //Goto the entry.
        final TarArchiveEntry inputEntry = TarTools.gotoTarArchiveEntry(tarStream, entryName);
        if(inputEntry == null)
        {
            throw new Exception("Expected entry " + entryName + "  was not found.");
        }

        //Read it.
        try
        {
            //This method handles the exception messaging.  It will first goto the entry that matches the name and then read it via
            //XMLTools.
            XMLTools.readXMLFromStream(tarStream, fastInfoset, readable);
        }
        finally
        {
            //The XML parsing should close the tar file, but just in case...
            tarStream.close();
        }
    }

    /**
     * Reads XML from the provided tar stream. It calls
     * {@link XMLTools#readXMLFromStream(InputStream, boolean, XMLReadable)}. This will close the stream provided upon
     * completing reading the XML, unless an {@link OHDWrapperTarArchiveInputStream} is used with prevent closing made
     * true.
     * 
     * @param tarStream Stream from which to read.
     * @param gzipped Indicates if the file is also gzipped.
     * @param entryName Name of the entry to read, which must be XML or fastInfoset XML.
     * @param readable The {@link XMLReadable} to use to read the XML.
     * @param fastInfoset Indicates if the XML should be read in fastInfoset.
     * @throws Exception
     */
    public static void readXMLFromTarStream(final TarArchiveInputStream tarStream,
                                            final boolean gzipped,
                                            final String entryName,
                                            final XMLReadable readable,
                                            final boolean fastInfoset) throws Exception
    {
        try
        {
            final TarArchiveEntry inputEntry = TarTools.gotoTarArchiveEntry(tarStream, entryName);
            if(inputEntry == null)
            {
                throw new Exception("Expected entry " + entryName + " was not found.");
            }
            XMLTools.readXMLFromStream(tarStream, fastInfoset, readable);
        }
        catch(final Exception e)
        {
            final Exception relayed = new Exception("Unable to parse entry " + entryName + ": " + e.getMessage());
            relayed.setStackTrace(e.getStackTrace());
            throw relayed;
        }
    }

    /**
     * FastInfoset is not yet supported for time series reading in this manner. This method will record the entire XML
     * string for the time series in memory before parsing; memory limitations may be an issue.
     * 
     * @param tarFile Tar file to read.
     * @param gzipped Indicates if the tar file is gzipped.
     * @param entryName Name of entry to read which specifies time series XML.
     * @return {@link TimeSeriesArrays} read from that xml.
     * @throws Exception
     */
    public static TimeSeriesArrays readTimeSeriesFromTarFile(final File tarFile,
                                                             final boolean gzipped,
                                                             final String entryName) throws Exception
    {
        final TarArchiveInputStream tarStream = TarTools.openTarArchiveForReading(tarFile, gzipped);
        try
        {
            final TarArchiveEntry inputEntry = TarTools.gotoTarArchiveEntry(tarStream, entryName);
            if(inputEntry == null)
            {
                throw new Exception("Expected time series entry " + entryName + "  was not found.");
            }

            final String xmlTxt = IOUtils.toString(tarStream);
            return TimeSeriesArraysTools.createTimeSeriesArraysFromXml(xmlTxt, TimeZone.getTimeZone("GMT"));
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            final Exception relayed = new Exception("Unable to parse time series entry " + entryName + ": "
                + e.getMessage());
            relayed.setStackTrace(e.getStackTrace());
            throw relayed;
        }
        finally
        {
            //The XML parsing may close the tar file, but just in case...
            tarStream.close();
        }
    }

    /**
     * Places a time series in a tar archive as an XML (ASCII) file. FastInfoset is not supported, yet. This writes the
     * entire times series XML to a string in memory before writing; memory limitations may be an issue.
     * 
     * @param tarStream The stream to which to add an entry.
     * @param entryName The name of the entry.
     * @param ts The time series to write to that entry as XML.
     * @throws IOException Standard reasons.
     */
    public static void addTimeSeriesEntryToTarArchive(final TarArchiveOutputStream tarStream,
                                                      final String entryName,
                                                      final TimeSeriesArrays ts) throws IOException
    {
        final byte[] bytes = TimeSeriesArraysTools.writeToString(ts).getBytes();
        final TarArchiveEntry entry = new TarArchiveEntry(new File(entryName));
        entry.setSize(bytes.length);
        tarStream.putArchiveEntry(entry);
        tarStream.write(bytes);
        tarStream.closeArchiveEntry();
    }

    /**
     * Places a time series in a tar archive as an XML (ASCII) file. FastInfoset is not supported, yet.
     * 
     * @param tarStream The stream to which to add an entry.
     * @param entryName The name of the entry.
     * @param ts The time series to write to that entry as XML.
     * @throws IOException Standard reasons.
     */
    public static void addTimeSeriesEntryToTarArchive(final TarArchiveOutputStream tarStream,
                                                      final String entryName,
                                                      final TimeSeriesArray ts) throws IOException
    {
        addTimeSeriesEntryToTarArchive(tarStream, entryName, new TimeSeriesArrays(ts));
    }

    /**
     * Adds an XML entry to the provided {@link TarArchiveOutputStream}.
     * 
     * @param tarStream The stream to which to add the entry.
     * @param entryName Name of the entry to add (as a {@link File}).
     * @param writable The provider of the {@link XMLWriter} to use for writing XML.
     * @param fastInfoset True if the file should be written in fastInfoset.
     * @throws Exception
     */
    public static void addXMLEntryToTarArchive(final TarArchiveOutputStream tarStream,
                                               final String entryName,
                                               final XMLWritable writable,
                                               final boolean fastInfoset) throws Exception
    {
        byte[] bytes = null;
        if(!fastInfoset)
        {
            bytes = XMLTools.writeXMLStringFromXMLWriterWithIndentation(writable).getBytes();
        }
        else
        {
            bytes = XMLTools.writeFISByteArrayFromXMLWriter(writable);
        }
        final TarArchiveEntry entry = new TarArchiveEntry(new File(entryName));
        entry.setSize(bytes.length);
        tarStream.putArchiveEntry(entry);
        tarStream.write(bytes);
        tarStream.closeArchiveEntry();
    }

    /**
     * Adds a {@link File} entry to the tar file.
     * 
     * @param tOutputStream - The stream to which to add the entry.
     * @param fileToTar - The file to add to the tar
     * @param base - The basepath to remove from the passed in file. For example, if the file being added is
     *            /fs/users/dsa/test.txt and you want test.txt to appear in the root of the tarball, you would set this
     *            string to /fs/users/dsa
     * @throws IOException
     */
    private static void addFileToTar(final TarArchiveOutputStream tOutputStream, final File fileToTar, final String base) throws IOException
    {
        String entryName;

        if(base.isEmpty())
        {
            entryName = fileToTar.getName();
        }
        else if(base.startsWith("\\"))
        {
            entryName = base.substring(1) + "/" + fileToTar.getName();
        }
        else
        {
            entryName = base + "/" + fileToTar.getName();
        }
        final TarArchiveEntry tarEntry = new TarArchiveEntry(fileToTar, entryName);

        tOutputStream.setLongFileMode(TarArchiveOutputStream.LONGFILE_GNU);
        tOutputStream.putArchiveEntry(tarEntry);

        if(fileToTar.isFile())
        {
            IOUtils.copy(new FileInputStream(fileToTar), tOutputStream);

            tOutputStream.closeArchiveEntry();
        }
        else
        {
            tOutputStream.closeArchiveEntry();

            final File[] children = fileToTar.listFiles();

            if(children != null)
            {
                for(final File child: children)
                {
                    addFileToTar(tOutputStream, child, entryName + "/");
                }
            }
        }
    }

    /**
     * Adds a Collection of {@link File} entries to the tar file.
     * 
     * @param tarStream - The stream to which to add the entry.
     * @param fileList - The list of Files to add to the tar
     * @param basePathToRemove - The basepath to remove from each file in the List of files. For example, if the file
     *            being added is /fs/users/dsa/test.txt and you want test.txt to appear in the root of the tarball, you
     *            would set this string to /fs/users/dsa
     * @throws Exception
     */
    public static void addFilesToTarArchive(final TarArchiveOutputStream tarStream,
                                            final List<File> fileList,
                                            final String basePathToRemove) throws Exception
    {
        for(final File fileToTar: fileList)
        {
            final String fileToTarDir = fileToTar.getParent() + "/";
            if(!fileToTarDir.startsWith(basePathToRemove))
            {
                throw new Exception(fileToTar.getName() + " does not reside in the base path passed in.  Aborting!");
            }
        }

        for(final File fileToTar: fileList)
        {

            final String fileBase = fileToTar.getParent().replace(basePathToRemove, "");
            addFileToTar(tarStream, fileToTar, fileBase);
        }
        tarStream.finish();
        tarStream.close();
    }

    /**
     * Adds a {@link String} entry to the tar file. The string entry is converted to bytes
     * 
     * @param tarStream The stream to which to add the entry.
     * @param entryName Name of the entry to add (as a {@link File}).
     * @param text Text to write to the tar file. It must comprise a complete entry (or file).
     * @throws Exception
     */
    public static void addTextEntryToTarArchive(final TarArchiveOutputStream tarStream,
                                                final String entryName,
                                                final String text) throws Exception
    {
        final byte[] bytes = text.getBytes();
        addByteEntryToTarArchive(tarStream, entryName, bytes);
    }

    /**
     * Adds an entry to the file containing the provided bytes.
     * 
     * @param tarStream The stream to which to add the entry.
     * @param entryName Name of the entry to add (as a {@link File}).
     * @param bytes The bytes to write to the file. It must be a complete entry.
     * @throws Exception
     */
    public static void addByteEntryToTarArchive(final TarArchiveOutputStream tarStream,
                                                final String entryName,
                                                final byte[] bytes) throws Exception
    {
        final TarArchiveEntry entry = new TarArchiveEntry(new File(entryName));
        entry.setSize(bytes.length);
        tarStream.putArchiveEntry(entry);
        tarStream.write(bytes);
        tarStream.closeArchiveEntry();
    }

    /**
     * Returns a list of the archive entries for the input stream. This will call {@link TarArchiveInputStream#reset()}
     * before and, most importantly for the caller, after obtaining the list.
     * 
     * @param tarStream Stream to check.
     * @return List of {@link TarArchiveEntry} objects.
     * @throws IOException
     */
    public static List<TarArchiveEntry> getListOfEntries(final TarArchiveInputStream tarStream) throws IOException
    {
        //Get list of entries for stream1.
        final List<TarArchiveEntry> entries = new ArrayList<TarArchiveEntry>();
        tarStream.reset();
        boolean hasMore = true;
        while(hasMore)
        {
            final TarArchiveEntry entry = tarStream.getNextTarEntry();
            if(entry == null)
            {
                hasMore = false;
            }
            else
            {
                entries.add(entry);
            }
        }
        tarStream.reset();
        return entries;
    }

    /**
     * @param tarStream Must be an {@link OHDWrapperTarArchiveInputStream}.
     * @return The name of the current entry in the tar stream.
     */
    public static String getCurrentEntryName(final TarArchiveInputStream tarStream)
    {
        return ((OHDWrapperTarArchiveInputStream)tarStream).getOHDCurrentEntry().getName();
    }

    /**
     * @param tarStream Must be an {@link OHDWrapperTarArchiveInputStream}.
     * @return True if the current entry matches the provided name.
     */
    public static boolean isCurrentEntry(final TarArchiveInputStream tarStream, final String entryName)
    {
        if(((OHDWrapperTarArchiveInputStream)tarStream).getOHDCurrentEntry() == null)
        {
            return false;
        }
        return (((OHDWrapperTarArchiveInputStream)tarStream).getOHDCurrentEntry().getName().equals(entryName));
    }

    /**
     * The tar stream provided is reset before and after checking.
     * 
     * @param tarStream {@link TarArchiveInputStream} to check.
     * @param entryName Name of entry to look for.
     * @return True if an entry by that name is present, false otherwise.
     * @throws IOException
     */
    public static boolean isEntryPresent(final TarArchiveInputStream tarStream, final String entryName) throws IOException
    {
        final TarArchiveEntry entry = gotoTarArchiveEntry(tarStream, entryName);
        return (entry != null);
    }

    /**
     * @param tarFile Tar file to check.
     * @param gzipped True if the file is gzipped.
     * @param entryName Name of entry to look for.
     * @return True if the entry is found, false otherwise, calling
     *         {@link #isEntryPresent(TarArchiveInputStream, String)}.
     * @throws Exception
     */
    public static boolean isEntryPresent(final File tarFile, final boolean gzipped, final String entryName) throws Exception
    {
        final TarArchiveInputStream tarStream = TarTools.openTarArchiveForReading(tarFile, gzipped);
        try
        {
            return isEntryPresent(tarStream, entryName);
        }
        finally
        {
            //The XML parsing may close the tar file, but just in case...
            tarStream.close();
        }
    }

    /**
     * This checks only the contents of the tar file, each individually. The meta-data about a tar file is not checked.
     * 
     * @param tarFile1 First file to check.
     * @param tarFile2 Second file to check.
     * @param gzipped True if the files are gzipped tar files.
     * @return True if the content is identical for each entry in the two files -and- the entries in the two files are
     *         identical (i.e., the same numbef of entries in the same order).
     * @throws IOException
     */
    public static boolean areTarFileContentsTheSame(final File tarFile1, final File tarFile2, final boolean gzipped) throws IOException
    {
        final TarArchiveInputStream stream1 = openTarArchiveForReading(tarFile1, gzipped);
        final TarArchiveInputStream stream2 = openTarArchiveForReading(tarFile2, gzipped);

        boolean done = false;
        while(!done)
        {
            final TarArchiveEntry entry1 = stream1.getNextTarEntry();
            final TarArchiveEntry entry2 = stream2.getNextTarEntry();

            //This should pickup on out of order or different quantity entries.
            if(!GeneralTools.checkForFullEqualityOfObjects(entry1, entry2))
            {
                System.err.println("Comparing " + tarFile1.getAbsolutePath() + " and " + tarFile2.getAbsolutePath()
                    + ": Found entry " + entry1 + " in first file, which does not match entry " + entry2 + ".");
                return false;
            }
            if(entry1 == null)
            {
                done = true;
            }
            else
            {
                if(!IOUtils.contentEquals(stream1, stream2))
                {
                    System.err.println("Comparing " + tarFile1.getAbsolutePath() + " and " + tarFile2.getAbsolutePath()
                        + ": Entry with name " + entry1.getName() + " contents do not match.");
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * A subclass of {@link TarArchiveInputStream} that does nothing when its {@link TarArchiveInputStream#close()}
     * method is called. To close it call {@link #reallyClose()}. This is returned by
     * {@link TarTools#openTarArchiveForReading(File, boolean)}, because some readers, such as {@link SAXParser}, will
     * close streams when done parsing. However, with a tar ball, you don't want the stream closed because you may need
     * to read multiple XMLs from the same tar ball and reopening it can take significant time.
     * 
     * @author hankherr
     */
    public static class OHDWrapperTarArchiveInputStream extends TarArchiveInputStream
    {
        private final boolean _preventClosing;

        public OHDWrapperTarArchiveInputStream(final InputStream is,
                                               final int blockSize,
                                               final int recordSize,
                                               final String encoding,
                                               final boolean preventClosing)
        {
            super(is, blockSize, recordSize, encoding);
            _preventClosing = preventClosing;
        }

        public OHDWrapperTarArchiveInputStream(final InputStream is,
                                               final int blockSize,
                                               final int recordSize,
                                               final boolean preventClosing)
        {
            super(is, blockSize, recordSize);
            _preventClosing = preventClosing;
        }

        public OHDWrapperTarArchiveInputStream(final InputStream is,
                                               final int blockSize,
                                               final String encoding,
                                               final boolean preventClosing)
        {
            super(is, blockSize, encoding);
            _preventClosing = preventClosing;
        }

        public OHDWrapperTarArchiveInputStream(final InputStream is, final int blockSize, final boolean preventClosing)
        {
            super(is, blockSize);
            _preventClosing = preventClosing;
        }

        public OHDWrapperTarArchiveInputStream(final InputStream is, final String encoding, final boolean preventClosing)
        {
            super(is, encoding);
            _preventClosing = preventClosing;
        }

        public OHDWrapperTarArchiveInputStream(final InputStream is, final boolean preventClosing)
        {
            super(is);
            _preventClosing = preventClosing;
        }

        public TarArchiveEntry getOHDCurrentEntry()
        {
            return super.getCurrentEntry();
        }

        @Override
        public void close() throws IOException
        {
            if(!_preventClosing)
            {
                super.close();
            }
        }

        public void reallyClose() throws IOException
        {
            super.close();
        }
    }
}
