package ohd.hseb.hefs.utils.tools;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Random;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * Abstract class of method that allow for generating random numbers using a {@link ThreadLocal} instance of
 * {@link Random}. This is useful in CHPS adapters in which a single static {@link Random} won't work because they are
 * not run in truly separate virtual machines; hence the needed for a thread specific version.
 * 
 * @author Hank.Herr
 */
public abstract class ThreadedRandomTools
{
    public static final Logger LOG = LogManager.getLogger(ThreadedRandomTools.class);

    /**
     * Global random number generator {@link ThreadLocal} contains an instance of {@link Random} that is accessed by all
     * HEFS components and is associated with the current {@link Thread}, allowing for models executing on different
     * threads to access different random number generators. The variable is initialized to be empty.
     */
    private static ThreadLocal<Random> RANDOM_NUM_GEN_LOCAL = new ThreadLocal<Random>();

    /**
     * Initializes {@link #RANDOM_NUM_GEN_LOCAL} and sets the {@link Random} object within it initialized using the
     * provided seed.
     * 
     * @param fixedSeed Seed to use to initialize the random number generator.
     */
    public static void initializeRandomNumberGeneratorForTesting(final long fixedSeed)
    {
        //Create a random for this thread with the appropriate seed.
        RANDOM_NUM_GEN_LOCAL.set(new Random(fixedSeed));
        LOG.info("Random number generator for thread " + Thread.currentThread().getId() + " initialized with the seed "
            + fixedSeed);
    }

    /**
     * General call to initialize the random number generator based on the current system time. If you need to track the
     * seed for each run of a model, then you'll want to call this method so that the seed used is set and output to the
     * log when this calls {@link #initializeRandomNumberGeneratorForTesting(long)}.
     */
    public static void initializeRandomNumberGenerator()
    {
        final long l = new Date().getTime();
        initializeRandomNumberGeneratorForTesting(l);
    }

    /**
     * @return {@link Random} to use for generating random numbers in MEFP models associated with the currently
     *         executing {@link Thread} via {@link ThreadLocal}. If {@link #RANDOM_NUM_GEN_LOCAL} is not initializes
     *         yet, then this will call {@link #initializeRandomNumberGenerator()}.
     */
    public static Random getRandomNumGen()
    {
        if(RANDOM_NUM_GEN_LOCAL.get() == null)
        {
            initializeRandomNumberGenerator();
        }
        return RANDOM_NUM_GEN_LOCAL.get();
    }

    /**
     * @return A value randomly selected with a given probability from two provided values, making use of the
     *         {@link #getRandomNumGen()}.
     */
    public static int selectBetweenTwoValues(final int value1, final int value2, final double probabilityOfValue1)
    {
        final double rnd = getRandomNumGen().nextDouble();
        if(rnd < probabilityOfValue1)
        {
            return value1;
        }
        else
        {
            return value2;
        }
    }

    /**
     * @return A shuffled list of integers including all numbers between the provided start and end values, inclusive.
     */
    public static List<Integer> generateListOfInts(final int startNum, final int lastNum)
    {
        final List<Integer> numbers = new ArrayList<>();
        for(int i = startNum; i <= lastNum; i++)
        {
            numbers.add(i);
        }
        shuffleNumbers(numbers);
        return numbers;
    }

    /**
     * Calls {@link Collections#shuffle(List, Random)} passing in {@link #getRandomNumGen()} as the number generator.
     * 
     * @param values The values to shuffle.
     */
    public static void shuffleNumbers(final List<? extends Number> values)
    {
        Collections.shuffle(values, getRandomNumGen());
    }
}
