package ohd.hseb.hefs.utils.tsarrays;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import nl.wldelft.util.Period;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;

/**
 * An {@link ArrayList} implementation of a lagged ensemble. Note that {@link TimeSeriesEnsemble} cannot be used because
 * lagged ensembles, by definition, will fail its member compatibility checks.
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
public class LaggedEnsemble extends ArrayList<TimeSeriesArray>
{
    private long _forecastTime;

    /**
     * Constructs a lagged ensemble from the given time series. The forecastTime dictates the first member, and the rest
     * of the members move backward in time, from that index.
     * 
     * @param forecastTime The targeted forecast time. If no series in the list has the forecast time, an
     *            {@link IllegalArgumentException} is thrown.
     * @param sourceList Sorted list of time series from which to extract the lagged ensemble.
     * @param ensembleSize The number of members to extract. If not enough members can be extracted because the starting
     *            index is too early in the list, an {@link IllegalArgumentException} is thrown.
     */
    public LaggedEnsemble(final long forecastTime, final List<TimeSeriesArray> sourceList, final int ensembleSize)
    {
        super(ensembleSize);
        _forecastTime = forecastTime;

        //Find index of the time series with the forecastTime provided.
        final int index = TimeSeriesArraysTools.searchByForecastTime(sourceList, forecastTime);
        if(index < ensembleSize)
        {
            throw new IllegalArgumentException("Cannot construct lagged ensemble of size " + ensembleSize
                + " for forecast time " + new Date(forecastTime).toString());
        }

        //The lagged ensemble starts from the found index and contains time series leading up to and including
        //it such that numberOfMembers many time series are put in the list.
        for(int i = index; size() < ensembleSize; i--)
        {
            add(sourceList.get(i));
        }
    }

    /**
     * @param members {@link List} of {@link TimeSeriesArray} instances to add to the lagged ensemble. It is assumed
     *            that the index 0 member has the forecast time for the lagged ensemble. It is also assumed that the
     *            members will be lagged backwards in time, although all members have the same forecast time is
     *            acceptable, and no checks are performed in any case.
     */
    public LaggedEnsemble(final Collection<TimeSeriesArray> members)
    {
        super(members.size());
        addAll(members);
        _forecastTime = members.iterator().next().getHeader().getForecastTime();
    }

    /**
     * Builds a lagged ensemble by cutting out time windows from the provided base members. All members will have the
     * same forecast time, start time, and end time. Empty time series are put into the lagged ensemble without change,
     * except for the forecast time.<br>
     * <br>
     * Time series are added via {@link TimeSeriesArray#subArray(Period)}, which maintains the same class for the
     * headers as the original time series. Header classes are not changed.
     * 
     * @param baseMembers The base members from which to cut out the lagged ensemble. Empty time series are allowed.
     * @param forecastTime Lagged ensemble forecast time.
     * @param startTime Desired data start time.
     * @param endTime Desired data end time.
     */
    public LaggedEnsemble(final Collection<TimeSeriesArray> baseMembers,
                          final long forecastTime,
                          final long startTime,
                          final long endTime)
    {
        _forecastTime = forecastTime;
        for(final TimeSeriesArray member: baseMembers)
        {
            final long stepMillis = member.getHeader().getTimeStep().getStepMillis();

            //If empty, keep the member without subarraying it.  Otherwise, determine the subarray.
            TimeSeriesArray modifiedMember = member;
            if(!member.isEmpty())
            {
                modifiedMember = member.subArray(new Period(startTime, endTime));

                //Make sure modified member includes missings for any times at the end that were beyond the limits
                //of the CFSv2 archived forecast (if any).
                for(long time = modifiedMember.getEndTime() + stepMillis; time <= endTime; time += stepMillis)
                {
                    modifiedMember.putValue(time, Float.NaN);
                }
            }

            //Set the forecast time.
            if(modifiedMember.getHeader() instanceof DefaultTimeSeriesHeader)
            {
                ((DefaultTimeSeriesHeader)modifiedMember.getHeader()).setForecastTime(_forecastTime);
            }
            else
            {
                throw new IllegalArgumentException("The header of at least one member of the provided time series is "
                    + "not a DefaultTimeSeriesHeader (or PiTimeSeriesHeader, which extends the default).");
            }

            this.add(modifiedMember);
        }
    }

    /**
     * @param ensembleId The id to be assigned to all members of this.
     */
    public void assignEnsembleId(final String ensembleId)
    {
        for(final TimeSeriesArray member: this)
        {
            //Set the forecast time.
            if(member.getHeader() instanceof DefaultTimeSeriesHeader)
            {
                ((DefaultTimeSeriesHeader)member.getHeader()).setEnsembleId(ensembleId);
            }
            else
            {
                throw new IllegalArgumentException("The header of at least one member of the provided time series is "
                    + "not a DefaultTimeSeriesHeader (or PiTimeSeriesHeader, which extends the default).");
            }
        }
    }

    /**
     * Assigns member indices to all members starting with firstIndex for the member in slot 0 and increasing it by one
     * thereafter.
     * 
     * @param firstIndex
     */
    public void assignEnsembleMemberIndices(final int firstIndex)
    {
        int index = firstIndex;
        for(final TimeSeriesArray member: this)
        {
            //Set the forecast time.
            if(member.getHeader() instanceof DefaultTimeSeriesHeader)
            {
                ((DefaultTimeSeriesHeader)member.getHeader()).setEnsembleMemberIndex(index);
            }
            else
            {
                throw new IllegalArgumentException("The header of at least one member of the provided time series is "
                    + "not a DefaultTimeSeriesHeader (or PiTimeSeriesHeader, which extends the default).");
            }
            index++;
        }
    }

    public long getForecastTime()
    {
        return _forecastTime;
    }

    public void setForecastTime(final long forecastTime)
    {
        _forecastTime = forecastTime;
    }
}
