package ohd.hseb.hefs.utils.tsarrays;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

import nl.wldelft.fews.pi.PiTimeSeriesHeader;
import nl.wldelft.util.Period;
import nl.wldelft.util.timeseries.ComplexEquidistantTimeStep;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader.DefaultThreshold;
import nl.wldelft.util.timeseries.SimpleEquidistantTimeStep;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import nl.wldelft.util.timeseries.TimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesHeader.Threshold;
import nl.wldelft.util.timeseries.TimeStep;
import ohd.hseb.hefs.utils.tools.NumberTools;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.util.misc.HCalendar;

import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

public abstract class TimeSeriesArrayTools
{
    public static final Comparator<TimeSeriesArray> FORECAST_TIME_COMPARATOR = new ForecastTimeComparator();

    /**
     * Comparator used to sort by forecast time (T0 of which time series was created).
     * 
     * @author hank.herr
     */
    private static class ForecastTimeComparator implements Comparator<TimeSeriesArray>
    {
        private ForecastTimeComparator()
        {
        }

        @Override
        public int compare(final TimeSeriesArray t0, final TimeSeriesArray t1)
        {
            return Long.valueOf(t0.getHeader().getForecastTime()).compareTo(t1.getHeader().getForecastTime());
        }
    }

    /**
     * Creates an empty time series array which is forecast on the given year. Set to January 1, 12:00 for the given
     * year. Mainly for use with Comparator TimeSeriesArrayTools.FORECAST_COMPARATOR.
     * 
     * @param year the year to be forecast on
     * @return empty time series with the given forecast year
     */
    public static TimeSeriesArray forecastOnYear(final int year)
    {
        final DefaultTimeSeriesHeader header = new DefaultTimeSeriesHeader();
        final Calendar date = HCalendar.convertStringToCalendar(String.valueOf(year) + "0101 12:00:00",
                                                                "CCYYMMDD hh:mm:ss");
        header.setForecastTime(date.getTimeInMillis());
        return new TimeSeriesArray(header);
    }

    /**
     * Returns a list of all time series that were forecast on the given year, sorted by forecast date. Will also take
     * null as a year, in which case it will consider any array with a forecast date <= 0L, which we are assuming means
     * it was observed.
     * 
     * @param arrays all time series to consider
     * @param year the year to be forecast on
     * @return A sorted list of time series forecast on the given years
     */
    public static List<TimeSeriesArray> filterByForecastYear(final Collection<TimeSeriesArray> arrays,
                                                             final Integer year)
    {
        return filterByForecastYear(arrays, Lists.newArrayList(year));
    }

    /**
     * Returns a list of all time series that were forecast on the given years, sorted by forecast date. Will also take
     * null as a year, in which case it will add any series with a negative forecast date, which is assumed to mean it
     * is an observed time series.
     * 
     * @param arrays all time series to consider
     * @param year the set of years to be forecast on
     * @return A sorted list of time series forecast on the given years
     */
    public static List<TimeSeriesArray> filterByForecastYear(final Collection<TimeSeriesArray> arrays,
                                                             final Collection<Integer> years)
    {
        final List<TimeSeriesArray> seriesList = Lists.newArrayList();
        for(final TimeSeriesArray series: arrays)
        {
            final long time = series.getHeader().getForecastTime();
            final Calendar date = HCalendar.computeCalendarFromMilliseconds(time);
            if((time <= 0L && years.contains(null)) || years.contains(date.get(Calendar.YEAR)))
            {
                seriesList.add(series);
            }
        }
        Collections.sort(seriesList, FORECAST_TIME_COMPARATOR);
        return seriesList;
    }

    /**
     * Returns a time series trimmed to the given year.
     * 
     * @param tsa time series to trim
     * @param year year to limit to
     * @return a new, trimmed time series
     */
    public static TimeSeriesArray trimToYear(final TimeSeriesArray tsa, final int year)
    {
        final Calendar start = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        start.set(year, 0, 1, 0, 0, 0);
        final Calendar end = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        end.set(year + 1, 0, 1, 0, 0, 0);
        return tsa.subArray(new Period(start.getTimeInMillis(), end.getTimeInMillis()));
    }

    /**
     * Splits a collection of time series into a map from start time to the series.
     * 
     * @param arrays a collection of TimeSeriesArray objects.
     * @return
     */
    public static Map<Long, List<TimeSeriesArray>> splitByStartTime(final Collection<TimeSeriesArray> arrays)
    {
        final Map<Long, List<TimeSeriesArray>> map = Maps.newHashMap();
        for(final TimeSeriesArray tsa: arrays)
        {
            final long time = tsa.getStartTime();
            List<TimeSeriesArray> seriesList = map.get(time);
            if(seriesList == null)
            {
                seriesList = new ArrayList<TimeSeriesArray>();
                map.put(time, seriesList);
            }
            seriesList.add(tsa);
        }
        return map;
    }

    /**
     * Splits a collection of time series into a map from forecast time to the series.
     * 
     * @param arrays a collection of TimeSeriesArray objects.
     * @return
     */
    public static Map<Long, List<TimeSeriesArray>> splitByForecastTime(final Collection<TimeSeriesArray> arrays)
    {
        final Map<Long, List<TimeSeriesArray>> map = Maps.newHashMap();
        for(final TimeSeriesArray tsa: arrays)
        {
            final long time = tsa.getHeader().getForecastTime();
            List<TimeSeriesArray> seriesList = map.get(time);
            if(seriesList == null)
            {
                seriesList = new ArrayList<TimeSeriesArray>();
                map.put(time, seriesList);
            }
            seriesList.add(tsa);
        }
        return map;
    }

    /**
     * Splits a collection of time series into a map from parameter id to the series.
     * 
     * @param arrays a collection of TimeSeriesArray objects.
     * @return
     */
    public static Map<String, List<TimeSeriesArray>> splitByParameterId(final Collection<TimeSeriesArray> arrays)
    {
        final Map<String, List<TimeSeriesArray>> map = Maps.newHashMap();
        for(final TimeSeriesArray tsa: arrays)
        {
            final String paramId = tsa.getHeader().getParameterId();
            List<TimeSeriesArray> seriesList = map.get(paramId);
            if(seriesList == null)
            {
                seriesList = new ArrayList<TimeSeriesArray>();
                map.put(paramId, seriesList);
            }
            seriesList.add(tsa);
        }
        return map;
    }

    /**
     * Fills the missing values, i.e. gaps, in a time series from its start to stop times with NaNs. It only fills in
     * values for times for which {@link TimeSeriesArray#containsTime(long)} returns false. If the time series is empty,
     * it does nothing.
     * 
     * @param series The time series to fill, which must already have data in place in order to identify the start time
     *            and end time.
     */
    public static void fillNaNs(final TimeSeriesArray series)
    {
        if(series.isEmpty())
        {
            return;
        }
        //Note that the looping does not need to include start or end time, since both are known to exist, as they are defining the
        //time series range.  But I'll leave them included anyway.
        for(long time = series.getStartTime(); time <= series.getEndTime(); time += series.getHeader()
                                                                                          .getTimeStep()
                                                                                          .getStepMillis())
        {
            if(!series.containsTime(time))
            {
                series.put(time, Float.NaN);
            }
        }
    }

    /**
     * Sets all values between start and end (inclusive) to be missing in the given series. The time series should
     * cleared before calling this, or at least have no values defined between start and end (inclusive).
     * 
     * @param series The time series to fill.
     * @param start The time of the first value to be {@link Float#NaN}.
     * @param end The time of the last value.
     */
    public static void fillNaNs(final TimeSeriesArray series, final long start, final long end)
    {
        for(long time = start; time <= end; time += series.getHeader().getTimeStep().getStepMillis())
        {
            series.put(time, Float.NaN);
        }
    }

    /**
     * Sets all values between start and end (inclusive) to be missing in the given series. The time series should
     * cleared before calling this, or at least have no values defined between start and end (inclusive).
     * 
     * @param series The time series to fill.
     * @param start The time of the first value to be {@link Float#NaN}.
     * @param end The time of the last value.
     */
    public static void fillMinValue(final TimeSeriesArray series, final long start, final long end)
    {
        for(long time = start; time <= end; time += series.getHeader().getTimeStep().getStepMillis())
        {
            series.put(time, Float.MIN_VALUE);
        }
    }

//
//    /**
//     * If the given series has a valid forecast time. A valid forecast time is defined as one that exceeds
//     * {@link Long#MIN_VALUE}.
//     * 
//     * @param tsa time series to check
//     * @return if the series has a valid forecast time
//     */
//    public static boolean isForecastTimeDefined(final TimeSeriesArray tsa)
//    {
//        return tsa.getHeader().getForecastTime() > Long.MIN_VALUE;
//    }
//
//    /**
//     * Creates a DataSet containing all time series values. Column 0 is the julian hour (hours since on Jan 1, 1900,
//     * 0h). Column 1 is the value cast to a double. This method may be a bit slow since it constructs a Calendar for
//     * every time series value. However, I want it to be able to handle any time series, including irregular time
//     * series, so I cannot rely on a Calendar looping mechanism to get the time for each value.
//     * 
//     * @param tsa
//     * @return
//     */
//    public static DataSet convertToJulianHourDataSet(final TimeSeriesArray tsa)
//    {
//        final DataSet results = new DataSet(tsa.size(), 2);
//        for(int i = 0; i < tsa.size(); i++)
//        {
//            results.addSample(new double[]{
//                HCalendar.computeJulianHourFromCalendar(HCalendar.computeCalendarFromMilliseconds(tsa.getTime(i)), true),
//                tsa.getValue(i)});
//        }
//        return results;
//    }

    /**
     * Conversion is done in place. No changes are made if the unit types are not compatible. This uses the class
     * {@link MeasuringUnit}, available in the CHPS Common code base.
     * 
     * @param ts Time series of values to convert. The base unit is specified in the header.
     * @param newUnit Target unit.
     * @throws Exception If the units are not compatible.
     */
    public static void convertUnits(final TimeSeriesArray ts, final String newUnit) throws Exception
    {
        final MeasuringUnit fromUnit = MeasuringUnit.getMeasuringUnit(ts.getHeader().getUnit().toUpperCase());
        final MeasuringUnit toUnit = MeasuringUnit.getMeasuringUnit(newUnit);
        if(fromUnit == toUnit) //The getMeasuringUnit returns the same instance if units are the same
        {
            return;
        }

        final double convFactor = MeasuringUnit.getConversionFactor(fromUnit, toUnit);

        for(int i = 0; i < ts.size(); i++)
        {
            if(fromUnit.getType() == toUnit.getType())
            {
                float value = ts.getValue(i);

                //when switching temperature, MeasuringUnit.getConversionFactor cannot do it
                value = convertValue(value, fromUnit, toUnit, convFactor);

                ts.setValue(i, value);
            }
            else
            //can't convert these types
            {
                throw new Exception("Invalid unit conversion attempt from " + fromUnit.getName() + " to "
                    + toUnit.getName() + ".");
            }
        }

        //Thresholds
//        final float[] convertedThresholdValues = new float[ts.getHeader().getHighLevelThresholdCount()];
//        final String[] valueIds = new String[ts.getHeader().getHighLevelThresholdCount()];
//        final String[] valueNames = new String[ts.getHeader().getHighLevelThresholdCount()];
        final Threshold[] thresholds = new Threshold[ts.getHeader().getHighLevelThresholdCount()];
        for(int j = 0; j < thresholds.length; j++)
        {
            thresholds[j] = new DefaultThreshold(((DefaultTimeSeriesHeader)ts.getHeader()).getHighLevelThreshold(j)
                                                                                          .getId(),
                                                 ((DefaultTimeSeriesHeader)ts.getHeader()).getHighLevelThreshold(j)
                                                                                          .getName(),
                                                 convertValue(((DefaultTimeSeriesHeader)ts.getHeader()).getHighLevelThreshold(j)
                                                                                                       .getValue(),
                                                              fromUnit,
                                                              toUnit,
                                                              convFactor));

//((DefaultTimeSeriesHeader)ts.getHeader()).getHighLevelThreshold(j);
//            convertedThresholdValues[j] = convertValue(ts.getHeader().getHighLevelThresholdValue(j),
//                                                       fromUnit,
//                                                       toUnit,
//                                                       convFactor);
//            valueIds[j] = ((DefaultTimeSeriesHeader)ts.getHeader()).getHighLevelThreshold(j).getId();
//            valueNames[j] = ((DefaultTimeSeriesHeader)ts.getHeader()).getHighLevelThreshold(j).getName();
        }

//        ((DefaultTimeSeriesHeader)ts.getHeader()).setHighLevelThresholds(valueIds, valueNames, convertedThresholdValues);
        ((DefaultTimeSeriesHeader)ts.getHeader()).setHighLevelThresholds(thresholds);

        ((DefaultTimeSeriesHeader)ts.getHeader()).setUnit(newUnit);
    }

    /**
     * Eventually, this method should be moved into {@link MeasuringUnit}. Furthermore, {@link MeasuringUnit} should be
     * redesigned to allow for linear conversion with a scalar and constant (not just scalars), so that temperature is
     * properly handled. Also, the conversion method should be part of {@link MeasuringUnit}, as in
     * fromUnit.convertValue(value, targetUnit).
     * 
     * @param value Value to convert.
     * @param fromUnit The base unit.
     * @param toUnit The target unit.
     * @param conversionFactor The conversion factor, calculated outside this method so that this does not need to
     *            determine the factor multiple times.
     * @return The converted value. Note that a special unit conversion is required for temperature conversion. If
     *         {@link #isOHDMissingValue(float)} returns true, then nothing will be done to the value before returning.
     */
    private static float convertValue(final float value,
                                      final MeasuringUnit fromUnit,
                                      final MeasuringUnit toUnit,
                                      final double conversionFactor)
    {
        if(isOHDMissingValue(value))
        {
            return value;
        }
        if(fromUnit == MeasuringUnit.degreesCelsius) //Assumes conversion to fahrenheit
        {
            return 9.0f * value / 5.0f + 32.0f;
        }
        else if(fromUnit == MeasuringUnit.degreesFahrenheit) //Assumes conversion to celsius
        {
            return 5.0f * (value - 32f) / 9.0f;
        }
        else
        {
            return value * (float)conversionFactor;
        }
    }

    /**
     * @return The index of the provided time within the provided time series, or -1 if the time is not present.
     */
    public static int getIndexOfTime(final TimeSeriesArray ts, final long time)
    {
        final int index = ts.firstIndexAfterOrAtTime(time);
        if(index < 0)
        {
            return index;
        }
        if(ts.getTime(index) != time)
        {
            return -1;
        }
        return index;
    }

    /**
     * Allows for accessing the time series by time instead of index.
     * 
     * @param ts The time series from which to pull a value.
     * @param time The time in millis for which to find the value.
     * @return The found value or NaN if no value was found for that time.
     */
    public static float getValueByTime(final TimeSeriesArray ts, final long time)
    {
        if((time < ts.getStartTime()) || (time > ts.getEndTime()))
        {
            return Float.NaN;
        }
        final int index = ts.firstIndexAfterOrAtTime(time);
        if(ts.getTime(index) != time)
        {
            return Float.NaN;
        }
        return ts.getValue(index);
    }

    /**
     * Replaces one value in the time series with the provided value.
     * 
     * @param ts Time series to modify.
     * @param time The time of the value to replace if it exists.
     * @param value The new value.
     * @return True if a value existed for the specified time and was replace. Returns false if no value exists for that
     *         time.
     */
    public static boolean replaceValueByTime(final TimeSeriesArray ts, final long time, final float value)
    {
        final int index = ts.firstIndexAfterOrAtTime(time);
        if(ts.getTime(index) != time)
        {
            return false;
        }
        ts.setValue(index, value);
        return true;
    }

    /**
     * @param ts
     * @return The time step in hours, gotten via {@link TimeSeriesArray#getHeader()} and
     *         {@link TimeSeriesHeader#getTimeStep()}, truncated to an int.
     */
    public static int getTimeStepInHours(final TimeSeriesArray ts)
    {
        return (int)(ts.getHeader().getTimeStep().getStepMillis() / HCalendar.MILLIS_IN_HR);
    }

    /**
     * Replaces all missing values within a time series, those set to NaN, with the provided value. The TimeSeriesArray
     * class stores missing values as NaN by default. After calling this, the TimeSeriesArray isMissing method will not
     * work -- the previously missing values will not longer be missing!
     * 
     * @param ts Time series to modify.
     * @param value The value to replace NaN with.
     */
    public static void replaceAllMissingValuesWithValue(final TimeSeriesArray ts, final float value)
    {
        for(int i = 0; i < ts.size(); i++)
        {
            if(ts.isMissingValue(i))
            {
                ts.setValue(i, value);
            }
        }
    }

    /**
     * Replaces all occurrences of the original value to the new value within a time series.
     * 
     * @param ts Time series to modify.
     * @param originalValue The original value to replace
     * @param newValue The new value to replace the originalValue with.
     */
    public static void replaceAllInstancesOfValue(final TimeSeriesArray ts,
                                                  final float originalValue,
                                                  final float newValue)
    {
        for(int i = 0; i < ts.size(); i++)
        {
            if(ts.getValue(i) == originalValue)
            {
                ts.setValue(i, newValue);
            }
        }
    }

    /**
     * @param ts
     * @return A string that includes the locationId, parameterId, ensembleId, and member index of the time series.
     */
    public static String createHeaderString(final TimeSeriesArray ts)
    {
        return "[TimeSeries: " + ts.getHeader().getLocationId() + ", " + ts.getHeader().getParameterId() + ", "
            + ts.getHeader().getEnsembleId() + ", " + ts.getHeader().getEnsembleMemberIndex() + "]";
    }

    /**
     * I did not write this method to be an all-encompassing check. Rather, it must focus on the header items that
     * matter. Specifically, this was designed for GraphGen and the items that matter are those that can be used to
     * delineate between time series: locationId, parameterId, ensemble info, time step, qualifiers, and T0. If those
     * items equal, regardless of other header items, GraphGen will treat them as being the same time series. Of course,
     * if the data is different, then that must be handled separately.
     * 
     * @param header1
     * @param header2
     * @throws Exception If the location id, parameter id, ensemble id, or member index, time step, qualifier ids (order
     *             matters), or forecast time is different.
     */
    public static void areHeadersEqual(final TimeSeriesHeader header1, final TimeSeriesHeader header2) throws Exception
    {
        if(!header1.getLocationId().equals(header2.getLocationId()))
        {
            throw new Exception("The location ids are not identical: " + header1.getLocationId() + ", "
                + header2.getLocationId());
        }
        if(!header1.getParameterId().equals(header2.getParameterId()))
        {
            throw new Exception("The parameter ids are not identical: " + header1.getParameterId() + ", "
                + header2.getParameterId());
        }
        if(!Objects.equal(header1.getEnsembleId(), header2.getEnsembleId()))
        {
            throw new Exception("The ensemble ids are not identical: " + header1.getEnsembleId() + ", "
                + header2.getEnsembleId());
        }
        if(header1.getEnsembleMemberIndex() != header2.getEnsembleMemberIndex())
        {
            throw new Exception("The ensemble member indices are not identical: " + header1.getEnsembleMemberIndex()
                + ", " + header2.getEnsembleMemberIndex());
        }

        //Time step
        if(header1.getTimeStep().isEquidistantMillis() != header2.getTimeStep().isEquidistantMillis())
        {
            throw new Exception("One of the time series time steps are equidistant and the other is not ("
                + header1.getTimeStep().isEquidistantMillis() + ", " + header2.getTimeStep().isEquidistantMillis()
                + ")");
        }
        if(header1.getTimeStep().isEquidistantMillis())
        {
            if(header1.getTimeStep().getStepMillis() != header2.getTimeStep().getStepMillis())
            {
                throw new Exception("Time series time steps are not of the same size.");
            }
        }

        //Qualifiers
        if(header1.getQualifierCount() != header2.getQualifierCount())
        {
            throw new Exception("The number of qualifier ids are not identical: " + header1.getQualifierCount() + ", "
                + header2.getQualifierCount());
        }
        for(int i = 0; i < header1.getQualifierCount(); i++)
        {
            if(!header1.getQualifierId(i).equals(header2.getQualifierId(i)))
            {
                throw new Exception("The qualifier ids in index " + i + " are not equal: " + header1.getQualifierId(i)
                    + ", " + header2.getQualifierId(i));
            }
        }

        if(header1.getForecastTime() != header2.getForecastTime())
        {
            throw new Exception("The forecast times are not identical: " + header1.getForecastTime() + ", "
                + header2.getForecastTime());
        }
    }

    /**
     * Calls {@link #areHeadersEqual(TimeSeriesHeader, TimeSeriesHeader) areHeadersEqual} and checks the start time, end
     * time, time step size, and values.
     * 
     * @param ts1
     * @param ts2
     * @throws Exception if any are not equal.
     */
    public static void checkTimeSeriesEqual(final TimeSeriesArray ts1, final TimeSeriesArray ts2) throws Exception
    {
        checkTimeSeriesEqual(ts1, ts2, 0.0D);
    }

    /**
     * Calls {@link #areHeadersEqual(TimeSeriesHeader, TimeSeriesHeader) areHeadersEqual}. Note that the time step in
     * the header is assumed accurate; the one in the {@link TimeSeriesArray} itself is ignored.
     * 
     * @param ts1
     * @param ts2
     * @throws Exception if any are not equal.
     */
    public static void checkTimeSeriesEqual(final TimeSeriesArray ts1, final TimeSeriesArray ts2, final double tolerance) throws Exception
    {
        //First the headers.
        try
        {
            areHeadersEqual(ts1.getHeader(), ts2.getHeader());
        }
        catch(final Exception e)
        {
            throw new Exception("Time series headers are not equal: " + e.getMessage());
        }

        //Header stuff not stored in header
        if(ts1.getStartTime() != ts2.getStartTime())
        {
            throw new Exception("Time series start times are not equal.");
        }
        if(ts1.getEndTime() != ts2.getEndTime())
        {
            throw new Exception("Time series end times are not equal.");
        }
        if(ts1.size() != ts2.size())
        {
            throw new Exception("Time series do not have the same number of values (" + ts1.size()
                + " for the first and " + ts2.size() + " for the second).");
        }

        //Checks the values.  Use TimeSeriesArray equals if tolerance is zero or less.
        if(tolerance <= 0.0D)
        {
            if(!ts1.equals(ts2))
            {
                throw new Exception("Time series values are not equal.");
            }
        }
        //Otherwise, individually check each number.  There may be some things that equals does that are not
        //captured here.
        else
        {
            for(int i = 0; i < ts1.size(); i++)
            {
                if((Float.isNaN(ts1.getValue(i))) && (Float.isNaN(ts1.getValue(i))))
                {
                    //This is good -- do nothing.  I know I can combine this with the else if into one statement,
                    //but I think its more readable this way.
                }
                else if(!NumberTools.nearEquals(ts1.getValue(i), ts2.getValue(i), tolerance))
                {
                    throw new Exception("Time series value at index " + i
                        + " found to differ by more than tolerance; values = " + ts1.getValue(i) + " and "
                        + ts2.getValue(i) + ".");
                }
            }
        }
    }

    /**
     * @return true if either {@link TimeSeriesArray#isMissingValue(int)} or {@link #isOHDMissingValue(float)} is true.
     */
    public static boolean isMissing(final TimeSeriesArray ts, final int i)
    {
        return ts.isMissingValue(i) || isOHDMissingValue(ts.getValue(i));
    }

    /**
     * @param value Value to check.
     * @return True if the value is either NaN or -998f, -999f, or -9999f.
     */
    public static boolean isOHDMissingValue(final float value)
    {
        return ((Float.isNaN(value)) || (value == -998f) || (value == -999f) || (value == -9999f));
    }

    /**
     * @param value Value to check.
     * @return True if the value is either NaN or -998f, -999f, or -9999f.
     */
    public static boolean isOHDMissingValue(final double value)
    {
        return ((Double.isNaN(value)) || (value == -998d) || (value == -999d) || (value == -9999d));
    }

    /**
     * Checks to see if any value in the time series is not missing (NaN). It calls {@link Float#isNaN()} directly;
     * maybe it should call {@link TimeSeriesArray#isMissingValue(int)}? This will return true if the time series is
     * empty, as well.
     * 
     * @param ts Time series to check
     * @return True if all values are missing; false otherwise.
     */
    public static boolean isAllMissing(final TimeSeriesArray ts)
    {
        for(int i = 0; i < ts.size(); i++)
        {
            if(!isMissing(ts, i))
            {
                return false;
            }
        }
        return true;
    }

    /**
     * This can be useful to make sure all read-in time series are converted to {@link DefaultTimeSeriesHeader}, so that
     * they are identical. Identical headers are needed for any {@link TimeSeriesArrays} object.
     * 
     * @param input To be copied.
     * @return A copy of the provided {@link TimeSeriesArray} with a {@link DefaultTimeSeriesHeader}. Both the header
     *         and data are copied.
     */
    public static TimeSeriesArray copyTimeSeries(final TimeSeriesArray input)
    {
        final TimeSeriesArray output = prepareTimeSeries(input);
        for(int i = 0; i < input.size(); i++)
        {
            output.putValue(input.getTime(i), input.getValue(i), input.getFlag(i));
        }
        return output;
    }

    /**
     * This can be useful to make sure all read-in time series are converted to {@link DefaultTimeSeriesHeader}, so that
     * they are identical. Identical headers are needed for any {@link TimeSeriesArrays} object. <br>
     * <br>
     * It copies the time series (header and data), but only includes a subseries within the overall time series,
     * specified by the three given times.
     * 
     * @param input Time series to copy a subsection of.
     * @param forecastTimeMillis The forecast time of the time series to create.
     * @param startTime The start time of the subseries extracted.
     * @param endTime The end time fo the subseries extracted.
     * @return A copy of the provided {@link TimeSeriesArray} with a {@link DefaultTimeSeriesHeader}. Both header and
     *         data are copied.
     */
    public static TimeSeriesArray copyTimeSeries(final TimeSeriesArray input,
                                                 final long forecastTimeMillis,
                                                 final long startTime,
                                                 final long endTime)
    {
        final TimeSeriesArray output = prepareTimeSeries(input);
        ((DefaultTimeSeriesHeader)output.getHeader()).setForecastTime(forecastTimeMillis);
        final int startIndex = input.firstIndexAfterOrAtTime(startTime);
        int endIndex = input.firstIndexAfterOrAtTime(endTime);

        //Backup endIndex if necessary to make sure we get no values with times after endTime.
        if(input.getTime(endIndex) > endTime)
        {
            endIndex--;
        }
        for(int i = startIndex; i <= endIndex; i++)
        {
            output.putValue(input.getTime(i), input.getValue(i));
        }
        return output;
    }

    /**
     * @param input Input time series providing basis for output.
     * @return A new time series containing the same header information as input. The header will be a
     *         {@link DefaultTimeSeriesHeader}. The time series will include no data.
     */
    public static TimeSeriesArray prepareTimeSeries(final TimeSeriesArray input)
    {
        final TimeSeriesArray results = new TimeSeriesArray(new DefaultTimeSeriesHeader());
        TimeSeriesArrayTools.prepareHeader(input, results, null);
        return results;
    }

    /**
     * @param input Input time series providing basis for output.
     * @param newTimeStepStr New time step to employ.
     * @param newTimeStepMillis New time step in millis.
     * @return A new time series containing the same header information as input, but with the given time step string
     *         and corresponding millis. The header will be a {@link DefaultTimeSeriesHeader}. The time series will
     *         include no data.
     */
    public static TimeSeriesArray prepareTimeSeries(final TimeSeriesArray input,
                                                    final String newTimeStepStr,
                                                    final long newTimeStepMillis)
    {
        final TimeSeriesArray results = new TimeSeriesArray(new DefaultTimeSeriesHeader());
        TimeSeriesArrayTools.prepareHeader(input, results, null, newTimeStepStr, newTimeStepMillis);
        return results;
    }

    /**
     * @param basis The basis for the header to create.
     * @param output The header to populate.
     * @param parameterSuffix A suffix to add to the parameterId. If null, nothing is added.
     */
    public static void prepareHeader(final TimeSeriesHeader basis,
                                     final TimeSeriesHeader output,
                                     final String parameterSuffix)
    {
        if(output == null)
        {
            return;
        }

        String addOn = " " + parameterSuffix;
        if(parameterSuffix == null)
        {
            addOn = "";
        }

        final DefaultTimeSeriesHeader editableHeader = (DefaultTimeSeriesHeader)output;

        editableHeader.setEnsembleId(basis.getEnsembleId());
        editableHeader.setForecastTime(basis.getForecastTime());
        editableHeader.setLocationDescription(basis.getLocationDescription());
        editableHeader.setLocationId(basis.getLocationId());
        editableHeader.setLocationName(basis.getLocationName());

        final String[] qualIds = new String[basis.getQualifierCount()];
        for(int i = 0; i < basis.getQualifierCount(); i++)
        {
            qualIds[i] = basis.getQualifierId(i);
        }
        editableHeader.setQualifierIds(qualIds);
        editableHeader.setGeometry(basis.getGeometry());
        editableHeader.setUnit(basis.getUnit());
        editableHeader.setEnsembleMemberIndex(basis.getEnsembleMemberIndex());

        editableHeader.setParameterId(basis.getParameterId() + addOn);
        editableHeader.setParameterType(basis.getParameterType());
        editableHeader.setParameterName(basis.getParameterName() + addOn);
        editableHeader.setUnit(basis.getUnit());

        editableHeader.setTimeStep(basis.getTimeStep());
    }

    /**
     * Call to append the parameter suffix to the header of the provided time series.
     * 
     * @param ts Time series with the header to modify.
     * @param parameterSuffix The suffix to add to the parameterId.
     */
    public static void appendSuffixToParameterId(final TimeSeriesArray ts, final String parameterSuffix)
    {
        final DefaultTimeSeriesHeader editableHeader = (DefaultTimeSeriesHeader)ts.getHeader();
        String addOn = " " + parameterSuffix;
        if(parameterSuffix == null)
        {
            addOn = "";
        }
        editableHeader.setParameterId(editableHeader.getParameterId() + addOn);
        editableHeader.setParameterName(editableHeader.getParameterName() + addOn);
    }

    /**
     * Prepares the {@link TimeSeriesHeader} in output based on that in basis. <br>
     * <br>
     * This is designed to work with {@link DefaultTimeSeriesHeader} instances, but should work with
     * {@link PiTimeSeriesHeader} since that is a subclass (not sure).
     * 
     * @param basis The array that provides a basis.
     * @param output The array whose header is to be prepared for use.
     * @param parameterSuffix The suffix to add to the parameter id, if not null. The suffix is added after the basis
     *            parameterId with a space separation.
     */
    public static void prepareHeader(final TimeSeriesArray basis,
                                     final TimeSeriesArray output,
                                     final String parameterSuffix)
    {
        prepareHeader(basis.getHeader(), output.getHeader(), parameterSuffix);
    }

    /**
     * Calls {@link #prepareHeader(TimeSeriesArray, TimeSeriesArray, String)} and then sets the time step based on a
     * string and milli step length and the parameter based on the suffix.<br>
     * <br>
     * This is designed to work with {@link DefaultTimeSeriesHeader} instances, but should work with
     * {@link PiTimeSeriesHeader} since that is a subclass (not sure).
     * 
     * @param basis The array that provides a basis.
     * @param output The array whose header is to be prepared for use.
     * @param parameterSuffix Suffix to add to the parameter id; see
     *            {@link #prepareHeader(TimeSeriesArray, TimeSeriesArray, String)}
     * @param timeStepStr The time step string of the new time step. If the string contains "year" or "month", then the
     *            time step is fixed to a constant defined within {@link ComplexEquidistantTimeStep}. Otherwise it
     *            defines a {@link SimpleEquidistantTimeStep} using the next argument.
     * @param milliTimeStep The milli equivalent of the timeStepStr, used to define time steps other than year and
     *            month.
     */
    public static void prepareHeader(final TimeSeriesArray basis,
                                     final TimeSeriesArray output,
                                     final String parameterSuffix,
                                     final String timeStepStr,
                                     final Long milliTimeStep)
    {
        prepareHeader(basis, output, parameterSuffix);

        TimeStep timeStep = null;
        if(milliTimeStep > 0)
        {
            if(timeStepStr.toLowerCase().contains("year"))
            {
                timeStep = ComplexEquidistantTimeStep.YEAR;
            }
            else if(timeStepStr.toLowerCase().contains("month"))
            {
                timeStep = ComplexEquidistantTimeStep.MONTH;
            }
            else
            {
                timeStep = SimpleEquidistantTimeStep.getInstance(milliTimeStep);
            }
        }

        final DefaultTimeSeriesHeader editableHeader = (DefaultTimeSeriesHeader)output.getHeader();
        editableHeader.setTimeStep(timeStep);
    }

    /**
     * @param header In addition to the time step, the header must specify the correct forecast time.
     * @return Calls {@link #createTimeSeries(TimeSeriesHeader, float[], long, boolean)} with a data start time of the
     *         header's forecast time plus one time step.
     */
    public static TimeSeriesArray createForecastTimeSeries(final TimeSeriesHeader header,
                                                           final float[] values,
                                                           final boolean copyHeader)
    {
        return createTimeSeries(header,
                                values,
                                header.getForecastTime() + header.getTimeStep().getStepMillis(),
                                copyHeader);
    }

    /**
     * @param header Provides all the information for the time series header. The time step must be regular, meaning
     *            {@link TimeStep#getStepMillis()} returns a valid number.
     * @param values The values to put into the time series.
     * @param dataStartTime The time of the first value in the array.
     * @param copyHeader If true, then the header is not used directly; rather it is copied via
     *            {@link #prepareHeader(TimeSeriesHeader, TimeSeriesHeader, String)}.
     * @return A populated time series using the header provided in the constructor for {@link TimeSeriesArray}. The
     *         data is put in via {@link TimeSeriesArray#put(long, float)} with the first value having a time associated
     *         with it of the provided dataStartTime. Each value is added following the header's time step.
     */
    public static TimeSeriesArray createTimeSeries(final TimeSeriesHeader header,
                                                   final float[] values,
                                                   final long dataStartTime,
                                                   final boolean copyHeader)
    {
        final TimeSeriesArray results;
        if(copyHeader)
        {
            results = new TimeSeriesArray(new DefaultTimeSeriesHeader());
            prepareHeader(header, results.getHeader(), null);
        }
        else
        {
            results = new TimeSeriesArray(header);
        }
        for(int i = 0; i < values.length; i++)
        {
            results.putValue(dataStartTime + i * header.getTimeStep().getStepMillis(), values[i]);
        }
        return results;
    }

    /**
     * Calls {@link TimeSeriesArray#isMissingValue(int)}.
     * 
     * @param ts To check.
     * @return Returns true if any value in the time series array is marked as missing or is NaN.
     */
    public static boolean containsMissingValues(final TimeSeriesArray ts)
    {
        boolean missingValue = false;

        for(int i = 0; i < ts.size(); i++)
        {
            // "bug" found where -999 is not converting to NaN by the reader.  
            if(isMissing(ts, i))
            {
                missingValue = true;
                break;

            }
        }
        return missingValue;
    }

    /**
     * Calls {@link TimeSeriesArray#isMissingValue(int)}.
     * 
     * @param ts To check.
     * @param startTime The first time to check for missings.
     * @param endTime The last time to check for missings.
     * @return Returns true if any value in the time series array is marked as missing or is NaN.
     */
    public static boolean containsMissingValuesWithinWindow(final TimeSeriesArray ts,
                                                            final long startTime,
                                                            final long endTime)
    {
        boolean missingValue = false;

        for(int i = 0; i < ts.size(); i++)
        {
            //Only care about the value if it is within the window.
            if((ts.getTime(i) < startTime) && (ts.getTime(i) > endTime))
            {
                continue;
            }

            // "bug" found where -999 is not converting to NaN by the reader.  
            if(isMissing(ts, i))
            {
                missingValue = true;
                break;

            }
        }
        return missingValue;
    }

    /**
     * @return Count of the number of time series values that are not NaN for which
     *         {@link TimeSeriesArray#isMissingValue(int)} returns false and for which {@link #isOHDMissingValue(float)}
     *         returns false. This is a count of the number of non-missing values in the ts.
     */
    public static int countNumberOfNonMissingValues(final TimeSeriesArray ts)
    {
        int count = 0;
        for(int i = 0; i < ts.size(); i++)
        {
            // "bug" found where -999 is not converting to NaN by the reader.  
            if(!ts.isMissingValue(i) && !isOHDMissingValue(ts.getValue(i)))
            {
                count++;
            }
        }
        return count;
    }

    /**
     * Trims missing data, where missing are denoted with NaN vaues, from the beginning and end of a time series. It
     * uses {@link TimeSeriesArray#subArray(Period)} to return a subseries.
     * 
     * @param ts Time series to trim.
     * @return Trimmed time series created via {@link TimeSeriesArray#subArray(Period)}.
     */
    public static TimeSeriesArray trimMissingValuesFromBeginningAndEndOfTimeSeries(final TimeSeriesArray ts)
    {
        int startIndex = 0;
        if(ts.isEmpty())
        {
            return ts;
        }
        while(ts.isMissingValue(startIndex))
        {
            startIndex++;
            if(startIndex >= ts.size())
            {
                ts.clear();
                return ts;
            }
        }
        int endIndex = ts.size() - 1;
        while(ts.isMissingValue(endIndex))
        {
            endIndex--;
        }
        final long newStartTime = ts.getTime(startIndex);
        final long newEndTime = ts.getTime(endIndex);
        return ts.subArray(new Period(newStartTime, newEndTime));
    }

    /**
     * Calls {@link #extendFromOtherTimeSeries(TimeSeriesArray, TimeSeriesArray, boolean)} with false passed into the
     * method, so that the adjacent check is not performed.
     * 
     * @param base
     * @param extension
     */
    public static void extendFromOtherTimeSeries(final TimeSeriesArray base, final TimeSeriesArray extension)
    {
        extendFromOtherTimeSeries(base, extension, false);
    }

    /**
     * This does check to make sure the time steps of the two time series are equal and (optional) that the time ranges
     * either overlap or touch (i.e., start of one is within one time step end of the other or vice versa). It then
     * starts from the start time of the base time series and works backward adding values from extension if present, or
     * missing otherwise. It then does the same starting from the end of the time series. In both directions it will
     * stop once outside the start-end time range of the extension.<br>
     * <br>
     * For this to work properly, the two time series must be in the same time systems (values falling on the same time
     * of day). Otherwise, the base will be extended with all missings.
     * 
     * @param base The time series to be extended.
     * @param extension The source of the extended values.
     * @param checkForAdjacentTimeSeries If true, then the check for time ranges either overlapping or touching will be
     *            done. Otherwise, it will not be checked.
     */
    public static void extendFromOtherTimeSeries(final TimeSeriesArray base,
                                                 final TimeSeriesArray extension,
                                                 final boolean checkForAdjacentTimeSeries)
    {
        if(extension.isEmpty())
        {
            return;
        }

        if(base.getHeader().getTimeStep().getStepMillis() != extension.getHeader().getTimeStep().getStepMillis())
        {
            throw new IllegalArgumentException("The time steps of the extension time series does not "
                + "match the base time series.");
        }

        //Only check for touching time series if requested.
        if(checkForAdjacentTimeSeries)
        {
            if((base.getStartTime() > extension.getEndTime() + base.getHeader().getTimeStep().getStepMillis())
                || (base.getEndTime() + base.getHeader().getTimeStep().getStepMillis() < extension.getStartTime()))
            {
                throw new IllegalArgumentException("The two time series do not overlap or touch "
                    + "at the end points of the time range.");
            }
        }

        //Working time points to the time where values are placed in base.
        //extensionIndex points to the first value after or at the working time, initially.
        //In the loop, extension index is backed up until it reaches a time equal or before
        //working time.  Then values are placed in base: either missing if the time was not 
        //fond in extension or non-missing if found.
        long workingTime = base.getStartTime() - base.getHeader().getTimeStep().getStepMillis();
        int extensionIndex = extension.firstIndexAfterOrAtTime(workingTime);
        while(workingTime >= extension.getStartTime())
        {
            //extensionIndex can only go as low as 0.  It must always point to something.
            while((extensionIndex > 0) && (extension.getTime(extensionIndex) > workingTime))
            {
                extensionIndex--;
            }
            if(extension.getTime(extensionIndex) == workingTime)
            {
                base.putValue(workingTime, extension.getValue(extensionIndex));
            }
            else
            {
                base.putValue(workingTime, Float.NaN);
            }
            workingTime -= base.getHeader().getTimeStep().getStepMillis();
        }

        //Now for the end of the base time series.  Same idea, but everything is reversed.
        //XXX There is probably a way to pull this off without two loops that look virtually the same.
        //But, for now, I think this separation may improve readability.
        workingTime = base.getEndTime() + base.getHeader().getTimeStep().getStepMillis();
        extensionIndex = extension.firstIndexAfterOrAtTime(workingTime);
        while(workingTime <= extension.getEndTime())
        {
            //extensionIndex can only go as high as size - 1.  It must always point to something.
            while((extensionIndex < extension.size() - 1) && (extension.getTime(extensionIndex) < workingTime))
            {
                extensionIndex++;
            }
            if(extension.getTime(extensionIndex) == workingTime)
            {
                base.putValue(workingTime, extension.getValue(extensionIndex));
            }
            else
            {
                base.putValue(workingTime, Float.NaN);
            }
            workingTime += base.getHeader().getTimeStep().getStepMillis();
        }
    }

    /**
     * Fills in missing values in base with values found in filler. Values that are missing and not found in filler are
     * left as missing. THIS HAS NOT BEEN FULLY TESTED YET!!!
     * 
     * @param base Time series for which missing values will be replaced.
     * @param filler Data to be used to fill in missing values, if present.
     */
    public static void fillInMissingFromOtherTimeSeries(final TimeSeriesArray base, final TimeSeriesArray filler)
    {
        if(filler.isEmpty())
        {
            return;
        }

        //Do nothing if there is no overlap.
        if((base.getStartTime() > filler.getEndTime()) || (base.getEndTime() < filler.getStartTime()))
        {
            return;
        }

        //Initialize workingTime to be the time for which the 
        final long startTime = Math.max(base.getStartTime(), filler.getStartTime());
        int baseWorkingIndex = base.firstIndexAfterTime(startTime);
        int fillerWorkingIndex = filler.firstIndexAfterOrAtTime(base.getTime(baseWorkingIndex));

        //Base working index moves from the starting point, computed above, to the end of base or
        //until the corresponding filler index moves off the end of filler.
        while((baseWorkingIndex < base.size()) && (fillerWorkingIndex < filler.size()))
        {
            //Increment filler index until we are again equal or after the working base time,
            //or until we reach the end of filler.
            while((fillerWorkingIndex < filler.size())
                && (filler.getTime(fillerWorkingIndex) < base.getTime(baseWorkingIndex)))
            {
                fillerWorkingIndex++;
            }

            //We only replace base values if they are missing and there is a potentially usable 
            //value in filler.
            if((base.isMissingValue(baseWorkingIndex)) && (fillerWorkingIndex < filler.size()))
            {
                if(!filler.isMissingValue(fillerWorkingIndex))
                {
                    base.setValue(baseWorkingIndex, filler.getValue(fillerWorkingIndex));
                }
            }

            baseWorkingIndex++;
        }
    }

    /**
     * The forecast time is also shifted if the provided {@link TimeSeriesArray} has a header that is an instance of
     * {@link DefaultTimeSeriesHeader}.
     * 
     * @param ts The time series to shift. Shifting is done in place; this time series is modified!
     * @param shiftInMillis The number of milliseconds to ADD to the provided time series. Negative values are allowed.
     */
    public static void shift(final TimeSeriesArray ts, final long shiftInMillis)
    {
        if(ts.getHeader() instanceof DefaultTimeSeriesHeader)
        {
            ((DefaultTimeSeriesHeader)ts.getHeader()).setForecastTime(ts.getHeader().getForecastTime() + shiftInMillis);
        }
        final long[] baseTimes = new long[ts.size()];
        final float[] baseValues = new float[ts.size()];
        for(int i = 0; i < ts.size(); i++)
        {
            baseTimes[i] = ts.getTime(i);
            baseValues[i] = ts.getValue(i);
        }
        ts.clear();
        for(int i = 0; i < baseTimes.length; i++)
        {
            ts.put(baseTimes[i] + shiftInMillis, baseValues[i]);
        }
    }

    /**
     * @param ts Time series to look at.
     * @param startingPoint The starting point for the computation, typically a T0.
     * @return Crude method just subtracts the startingPoint from the time series end time and determines the number of
     *         full days that encompasses (round down). This does no checking at all on the validity of startingPoint.
     */
    public static int computeNumberOfFullDaysSpanned(final TimeSeriesArray ts, final long startingPoint)
    {
        if(ts.isEmpty())
        {
            return 0;
        }
        final long difference = ts.getEndTime() - startingPoint;
        return (int)((double)difference / (double)(HCalendar.MILLIS_IN_HR * 24));
    }

    /**
     * @return The qualifier ids for the time series as an array.
     */
    public static String[] getQualifierIds(final TimeSeriesArray ts)
    {
        final String[] qualIds = new String[ts.getHeader().getQualifierCount()];
        for(int i = 0; i < ts.getHeader().getQualifierCount(); i++)
        {
            qualIds[i] = ts.getHeader().getQualifierId(i);
        }
        return qualIds;
    }
}
