package ohd.hseb.hefs.utils.tsarrays;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.TimeZone;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamReader;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.common.collect.Lists;

import nl.wldelft.fews.pi.PiTimeSeriesHeader;
import nl.wldelft.fews.pi.PiTimeSeriesParser;
import nl.wldelft.fews.pi.PiTimeSeriesSerializer;
import nl.wldelft.fews.pi.PiVersion;
import nl.wldelft.util.FileUtils;
import nl.wldelft.util.Period;
import nl.wldelft.util.TextUtils;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.SimpleTimeSeriesContent;
import nl.wldelft.util.timeseries.SimpleTimeSeriesContentHandler;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import nl.wldelft.util.timeseries.TimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeStep;
import ohd.hseb.hefs.utils.Dyad;
import ohd.hseb.hefs.utils.piservice.FewsPiServiceProvider;
import ohd.hseb.hefs.utils.tools.ListTools;
import ohd.hseb.hefs.utils.tsarrays.agg.NDCounterAggregator;
import ohd.hseb.measurement.MeasuringUnit;

public abstract class TimeSeriesArraysTools
{
    public static final Logger LOG = LogManager.getLogger(TimeSeriesArraysTools.class);

    /**
     * If time series are being read in multiple files at the same time (diff threads), this variable may not accurately
     * reflect the read time zone when {@link #getMostRecentlyReadTimeZone()} is called.
     */
    public static TimeZone _mostRecentlyReadTimeZone = null;

    /**
     * @return The {@link TimeZone} found in the last file read via {@link #readFromFile(File)}. Note that if
     *         {@link #readFromFiles(List)} was just called, the time zone will be that for the last file in the list
     *         read.
     */
    public static TimeZone getMostRecentlyReadTimeZone()
    {
        return _mostRecentlyReadTimeZone;
    }

    /**
     * This method should not need to exist. For any {@link PiTimeSeriesHeader} in the provided ts, the file desciption
     * is set to null. This is done to make tests pass, since its output in tar balls appears to be inconsistent.
     * 
     * @param ts
     */
    public static void removeAnnoyingFileDescriptions(final TimeSeriesArrays ts)
    {
        for(int i = 0; i < ts.size(); i++)
        {
            if(ts.get(i).getHeader() instanceof PiTimeSeriesHeader)
            {
                ((PiTimeSeriesHeader)ts.get(i).getHeader()).setFileDescription(null);
            }
        }
    }

    /**
     * Prepare the headers in output to be identical to those in basis.<br>
     * <br>
     * The header for the output must be a {@link DefaultTimeSeriesHeader} (it will not work with a
     * {@link PiTimeSeriesHeader}.
     * 
     * @param basis The arrays whose headers provides the basis for each time series.
     * @param output The arrays to adjust.
     * @param parameterSuffix The suffix to add to the parameter id, if not null.
     * @throws Exception If the two TimeSeriesArrays lists are not of the same size.
     */
    public static void prepareHeaders(final TimeSeriesArrays basis,
                                      final TimeSeriesArrays output,
                                      final String parameterSuffix) throws Exception
    {
        if(basis.size() != output.size())
        {
            throw new Exception("Number of input series does not equal number of output series.");
        }
        for(int i = 0; i < basis.size(); i++)
        {
            TimeSeriesArrayTools.prepareHeader(basis.get(i), output.get(i), parameterSuffix);
        }
    }

    /**
     * Prepare the headers in output to be identical to the header in basis.<br>
     * <br>
     * The header for the output must be a {@link DefaultTimeSeriesHeader} (it will not work with a
     * {@link PiTimeSeriesHeader}.
     * 
     * @param basis The array whose header provides the basis for all time series.
     * @param output The arrays to adjust.
     * @param parameterSuffix The suffix to add to the parameter id, if not null.
     * @throws Exception If the two TimeSeriesArrays lists are not of the same size.
     */
    public static void prepareHeaders(final TimeSeriesArray basis,
                                      final TimeSeriesArrays output,
                                      final String parameterSuffix,
                                      final boolean includeEnsembleParameters)
    {
        for(int i = 0; i < output.size(); i++)
        {
            TimeSeriesArrayTools.prepareHeader(basis, output.get(i), parameterSuffix);
            if(!includeEnsembleParameters)
            {
                ((DefaultTimeSeriesHeader)output.get(i).getHeader()).setEnsembleId(null);
                ((DefaultTimeSeriesHeader)output.get(i).getHeader()).setEnsembleMemberIndex(-1);
            }
        }
    }

    /**
     * This method should be expanded as needed.
     * 
     * @param parameterId
     * @return Returns a one or two word summary of the data type. This is useful for plot labels.
     */
    public static String returnOneWordDataTypeSummary(final String parameterId)
    {
        if((parameterId.startsWith("QIN")) || (parameterId.startsWith("QINE")) || (parameterId.startsWith("SQIN"))
            || (parameterId.startsWith("SQME")) || (parameterId.startsWith("QME")))
        {
            return "Discharge";
        }
        if((parameterId.startsWith("SSTG")) || (parameterId.startsWith("STG")))
        {
            return "Stage";
        }
        if((parameterId.startsWith("MAP")) || (parameterId.startsWith("FMAP")) || (parameterId.startsWith("MAPE"))
            || (parameterId.startsWith("MAPX")) || (parameterId.startsWith("RAIM")))
        {
            return "Precipitation";
        }
        if((parameterId.startsWith("MAT")) || (parameterId.startsWith("FMAT")))
        {
            return "Temperature";
        }
        if(parameterId.startsWith("TFMX") || parameterId.startsWith("TMAX"))
        {
            return "Temperature";
        }
        if(parameterId.startsWith("TFMN") || parameterId.startsWith("TMIN"))
        {
            return "Temperature";
        }
        if(parameterId.startsWith("PELE") || parameterId.startsWith("PELV"))
        {
            return "Pool Elevation";
        }
        if(parameterId.startsWith("SPEL"))
        {
            return "Pool Stage";
        }
        if(NDCounterAggregator.isCounterParameterId(parameterId))
        {
            return NDCounterAggregator.buildReadableCounterLabelFromParameter(parameterId);
        }
        //if((parameterId.startsWith))
        return "Value";
    }

    /**
     * @param tsList A list of TimeSeriesArrays instances that need to be summarized.
     * @return A list of String[], each with 5 elements: location id, location name, location description, parameter id,
     *         and ensemble id.
     */
    public static List<String[]> retrieveListOfUniqueIdentifyingComponentsForAllTimeSeries(final TimeSeriesArrays timeSeries)
    {
        final List<String[]> results = new ArrayList<String[]>();
        for(int j = 0; j < timeSeries.size(); j++)
        {
            final String[] tsIdentifyingInfo = new String[5];
            tsIdentifyingInfo[0] = timeSeries.get(j).getHeader().getLocationId();
            tsIdentifyingInfo[1] = timeSeries.get(j).getHeader().getLocationName();
            tsIdentifyingInfo[2] = timeSeries.get(j).getHeader().getLocationDescription();
            tsIdentifyingInfo[3] = timeSeries.get(j).getHeader().getParameterId();
            tsIdentifyingInfo[4] = timeSeries.get(j).getHeader().getEnsembleId();

            //Make sure the entry does not yet exist and add it if it doesn't.
            int k;
            for(k = 0; k < results.size(); k++)
            {
                if(Arrays.equals(results.get(k), tsIdentifyingInfo))
                {
                    break;
                }
            }
            if(k == results.size())
            {
                results.add(tsIdentifyingInfo);
            }
        }
        return results;
    }

    /**
     * @param tsArrays A list of TimeSeriesArrays instances that need to be summarized.
     * @return A List of all unique locationIds
     */
    public static List<String> retrieveListofUniqueLocationIdsForAllTimeSeries(final TimeSeriesArrays tsArrays)
    {
        final List<String> locationIds = new ArrayList<String>();

        for(int i = 0; i < tsArrays.size(); i++)
        {
            final String locId = tsArrays.get(i).getHeader().getLocationId();

            int k = 0;
            for(k = 0; k < locationIds.size(); k++)
            {
                if(locationIds.get(k).equalsIgnoreCase(locId))
                {
                    break;
                }
            }
            if(k == locationIds.size())
            {
                locationIds.add(locId);
            }
        }
        return locationIds;
    }

    /**
     * @param timeSeriesXML XML specifying time series.
     * @return TimeSeriesArrays object.
     * @throws IOException
     */
    public static TimeSeriesArrays createTimeSeriesArraysFromXml(final String timeSeriesXML) throws IOException
    {
        return createTimeSeriesArraysFromXml(timeSeriesXML, null);
    }

    /**
     * @param timeSeriesXML XML to parse.
     * @param defaultTZ Default time zone to use. It will not call FewsPiServiceProvider.getTimeZoneId() and attempt to
     *            connect to CHPS unless defaultTZ is null.
     * @return Time series read in from XML.
     * @throws IOException
     */
    public static TimeSeriesArrays createTimeSeriesArraysFromXml(final String timeSeriesXML,
                                                                 TimeZone defaultTZ) throws IOException
    {
        if(defaultTZ == null)
        {
            defaultTZ = TimeZone.getTimeZone(FewsPiServiceProvider.getTimeZoneId());
        }
        final SimpleTimeSeriesContentHandler timeSeriesContentHandler = new SimpleTimeSeriesContentHandler();
        timeSeriesContentHandler.setDefaultTimeZone(defaultTZ);
        final PiTimeSeriesParser piTimeSeriesParser = new PiTimeSeriesParser();
        TextUtils.parse(timeSeriesXML, "series", piTimeSeriesParser, timeSeriesContentHandler);
        return timeSeriesContentHandler.getTimeSeriesArrays();
    }

    /**
     * This method updates the {@link #_mostRecentlyReadTimeZone} static variable appropriately for the file just read.
     * This method does NOT call {@link #readFromFile(File)} because I don't know if its safe to always assume that the
     * {@link #_mostRecentlyReadTimeZone} reflects the file contents if time series are being read in multiple threads.
     * 
     * @param fileFile from which to read time series.
     * @return {@link Dyad} containing the {@link TimeSeriesArrays} in the file along with the {@link TimeZone} pulled
     *         from the file's header via the {@link PiTimeSeriesParser}.
     * @throws IOException
     * @throws InterruptedException
     */
    public static Dyad<TimeSeriesArrays, TimeZone> readFromFile2(final File file) throws IOException,
                                                                                  InterruptedException
    {
        final SimpleTimeSeriesContentHandler timeSeriesContentHandler = new SimpleTimeSeriesContentHandler();
        final PiTimeSeriesParser piTimeSeriesParser = new PiTimeSeriesParser();
        FileUtils.parse(file, piTimeSeriesParser, timeSeriesContentHandler);
        _mostRecentlyReadTimeZone = piTimeSeriesParser.getTimeZone();
        return new Dyad<TimeSeriesArrays, TimeZone>(timeSeriesContentHandler.getTimeSeriesArrays(),
                                                    piTimeSeriesParser.getTimeZone());
    }

    /**
     * This method updates the {@link #_mostRecentlyReadTimeZone} static variable appropriately for the file just read.
     * 
     * @param file File from which to read time series.
     * @return {@link TimeSeriesArrays} containing the read time series.
     * @throws IOException
     * @throws InterruptedException
     */
    public static TimeSeriesArrays readFromFile(final File file) throws IOException, InterruptedException
    {
        final SimpleTimeSeriesContentHandler timeSeriesContentHandler = new SimpleTimeSeriesContentHandler();
//        timeSeriesContentHandler.setDefaultTimeZone(TimeZone.getTimeZone(FewsPiServiceProvider.getTimeZoneId())); I don't think I need this line, which was copied from the String read above.
        final PiTimeSeriesParser piTimeSeriesParser = new PiTimeSeriesParser();
        FileUtils.parse(file, piTimeSeriesParser, timeSeriesContentHandler);
        _mostRecentlyReadTimeZone = piTimeSeriesParser.getTimeZone();
        return timeSeriesContentHandler.getTimeSeriesArrays();
    }

    //XXX Just marking this method so it shows up in Celipse.
    /**
     * THIS METHOD IS NOT THOROUGHLY TESTED!!!  It is only intended as a starting point for a change to allow
     * PI-timeseries to be read from a large .tgz file containing many files.  Such a method would read through the 
     * archive, similar to how MEFP pulls it off for its parameter files, and pass Streams to this method to get
     * and accumulate the read in TimeSeriesArrays.
     */
    public static TimeSeriesArrays readFromStream(final InputStream stream) throws Exception
    {
        final SimpleTimeSeriesContentHandler timeSeriesContentHandler = new SimpleTimeSeriesContentHandler();
        final PiTimeSeriesParser piTimeSeriesParser = new PiTimeSeriesParser();
        final XMLInputFactory f = XMLInputFactory.newInstance();
        final XMLStreamReader r = f.createXMLStreamReader(stream);
        
        //Its unclear to me what the second argument is for the parse method of PiTimeSeriesParser.
        piTimeSeriesParser.parse(r, "WHAT IS THIS???", timeSeriesContentHandler);  
        final TimeSeriesArrays tss = timeSeriesContentHandler.getTimeSeriesArrays();
        return tss;
    }

    /**
     * Calls {@link #readFromFiles(List, boolean)} passing in false for the error flag so that reading does not error
     * out unless processed XML is invalid.
     */
    public static TimeSeriesArrays readFromFiles(final List<File> files) throws IOException, InterruptedException
    {
        return readFromFiles(files, false);
    }

    /**
     * @param files Files to read time series from.
     * @param errorOutOnFailure If true, then if any file cannot be read for any reason, it will fail out. If false,
     *            only parsing exceptions due to bad XML will result in problems.
     * @return {@link TimeSeriesArrays} read from the given files, concatenated into one object.
     */
    public static TimeSeriesArrays readFromFiles(final List<File> files,
                                                 final boolean errorOutOnFailure) throws IOException,
                                                                                  InterruptedException
    {
        TimeSeriesArrays results = null;
        for(final File file: files)
        {
            if(!file.exists() || !file.canRead())
            {
                if(errorOutOnFailure)
                {
                    throw new IOException("The file " + file.getAbsolutePath() + " does not exist or cannot be read.");
                }
                else
                {
                    continue; //skip the file.
                }
            }
            if(results == null)
            {
                results = readFromFile(file);
            }
            else
            {
                results.addAll(readFromFile(file));
            }
        }
        return results;
    }

    /**
     * Creates a string with version 1.9, which includes lat/lon and ensemble info. The missing value will be NaN. Calls
     * {@link #writeToString(TimeSeriesArrays, float)}.
     * 
     * @param ts Time series to write.
     * @throws IOException
     * @throws InterruptedException
     */
    public static String writeToString(final TimeSeriesArrays ts)
    {
        return writeToString(ts, Float.NaN);
    }

    /**
     * Creates a string with version 1.9, which includes lat/lon and ensemble info.
     * 
     * @param ts Time series to write.
     * @param missingValue The value to use for missing in the output.
     * @throws IOException
     * @throws InterruptedException
     */
    public static String writeToString(final TimeSeriesArrays ts, final float missingValue)
    {
        final TimeZone defaultTZ = TimeZone.getTimeZone("GMT");
        final SimpleTimeSeriesContent timeSeriesContent = new SimpleTimeSeriesContent(ts);
        timeSeriesContent.setDefaultTimeZone(defaultTZ);
        timeSeriesContent.setDefaultMissingValue(missingValue);
        final PiTimeSeriesSerializer serializer = new PiTimeSeriesSerializer(PiVersion.VERSION_1_9);
        return ts.toString(serializer);
    }

    /**
     * Write one time series to a file calling {@link #writeToFile(File, TimeSeriesArrays)}.
     */
    public static void writeToFile(final File file, final TimeSeriesArray ts) throws IOException, InterruptedException
    {
        writeToFile(file, new TimeSeriesArrays(ts));
    }

    /**
     * Creates a file with version 1.9, which includes lat/lon and ensemble info. The missing value will be NaN. Calls
     * {@link #writeToFile(File, TimeSeriesArrays, float)}.
     * 
     * @param file Must end in .xml or .fi.
     * @param ts Time series to write.
     * @throws IOException
     * @throws InterruptedException
     */
    public static void writeToFile(final File file, final TimeSeriesArrays ts) throws IOException, InterruptedException
    {
        writeToFile(file, ts, Float.NaN);
    }

    /**
     * Creates a file with version 1.9, which includes lat/lon and ensemble info.
     * 
     * @param file Must end in .xml or .fi.
     * @param ts Time series to write.
     * @param missingValue The value to use for missing data.
     * @throws IOException
     * @throws InterruptedException
     */
    public static void writeToFile(final File file,
                                   final TimeSeriesArrays ts,
                                   final float missingValue) throws IOException, InterruptedException
    {
        final TimeZone defaultTZ = TimeZone.getTimeZone("GMT");
        final SimpleTimeSeriesContent timeSeriesContent = new SimpleTimeSeriesContent(ts);
        timeSeriesContent.setDefaultTimeZone(defaultTZ);
        timeSeriesContent.setDefaultMissingValue(missingValue);
        final PiTimeSeriesSerializer serializer = new PiTimeSeriesSerializer(PiVersion.VERSION_1_9);
        FileUtils.write(file, timeSeriesContent, serializer);

//        ts.toString(serializer);
//old  -- had problems, but I don't know what.
//        PiTimeSeriesWriter writer = new PiTimeSeriesWriter(file);
//        writer.write(ts);
    }

//Can resurrect this if I ever figure out how to get the Deltares stuff to output the headers only.
//    public static void writeAsFloatsToBinaryFile(final File baseFile, final TimeSeriesArrays ts) throws IOException
//    {
//        final FileOutputStream stream = new FileOutputStream(FileTools.replaceExtension(baseFile, "bin"));
//        DataOutputStream writer = null;
//        try
//        {
//            writer = new DataOutputStream(stream);
//            for(int tsIndex = 0; tsIndex < ts.size(); tsIndex++)
//            {
//                for(int valueIndex = 0; valueIndex < ts.get(tsIndex).size(); valueIndex++)
//                {
//                    writer.writeFloat(ts.get(tsIndex).getValue(valueIndex));
//                }
//            }
//        }
//        finally
//        {
//            StreamTools.closeStream(writer);
//        }
//    }

    /**
     * Calls writeToFile after constructing a TimeSeriesArrays from the given collection.
     * 
     * @param file File to create.
     * @param col TimeSeriesArray collection specifying time series.
     * @throws IOException
     * @throws InterruptedException
     */
    public static void writeToFile(final File file, final Collection<TimeSeriesArray> col) throws IOException,
                                                                                           InterruptedException
    {
        writeToFile(file, convertTimeSeriesCollection(col));
    }

    /**
     * Calls writeToFile after constructing a TimeSeriesArrays from the given collection. The 2 in the name is needed to
     * avoid conflict with the other writeToFile that takes a collection of TimeSeriesArray instances.
     * 
     * @param file File to create.
     * @param col TimeSeriesArrays collection specifying time series.
     * @throws IOException
     * @throws InterruptedException
     */
    public static void writeToFile2(final File file, final Collection<TimeSeriesArrays> col) throws IOException,
                                                                                             InterruptedException
    {
        writeToFile(file, convertTimeSeriesArraysCollections(col));
    }

    /**
     * Writes the collection of time series to a binary format. When reading it back in, be sure to call one of the
     * binary/byte array readers. Also, if the binary is gzipped, be sure to create a gzipped outpout stream.
     * 
     * @param dataStream The stream to which to write. It is NOT closed when done.
     * @param col The time series to write.
     */
    public static void writeDataToBinStream(final OutputStream outputStream,
                                            final Collection<TimeSeriesArray> col) throws IOException
    {
        final DataOutputStream dataStream = new DataOutputStream(outputStream);
        for(final TimeSeriesArray ts: col)
        {
            dataStream.writeLong(ts.getHeader().getForecastTime());
            dataStream.writeLong(ts.getStartTime());
            dataStream.writeInt(ts.size());
            for(int i = 0; i < ts.size(); i++)
            {
                dataStream.writeFloat(ts.getValue(i));
            }
        }
        dataStream.flush();
    }

    /**
     * Writes a collection of {@link TimeSeriesArray} instanced to a gzipped binary file. See
     * {@link #readDataFromBinFile(TimeSeriesArray, File)} for how to read it back in. This calls
     * {@link #writeDataToBinaryStream(DataOutputStream, Collection)} after preparing a gzipped file output stream.
     * 
     * @param binFile The name of the file to write.
     * @param col The collection of time series to write.
     * @throws IOException
     */
    public static void writeDataToBinFile(final File binFile, final Collection<TimeSeriesArray> col) throws IOException
    {
        FileOutputStream outputStream = null;
        GZIPOutputStream gzipStream = null;
        try
        {
            outputStream = new FileOutputStream(binFile);
            gzipStream = new GZIPOutputStream(outputStream);
            writeDataToBinStream(gzipStream, col);
        }
        finally
        {
            if(gzipStream != null)
            {
                gzipStream.close();
            }
        }
    }

    /**
     * Writes a collection of {@link TimeSeriesArray} instanced to a gzipped binary format byte array similar to the
     * {@link #writeDataToBinFile(File, Collection)} method. This calls
     * {@link #writeDataToBinStream(DataOutputStream, Collection)} after preparing a gzipped byte array output stream.
     * Note that a limited amount of information is output to the stream and the result should be read in by called
     * {@link #readDataFromByteArray(TimeSeriesArray, byte[])}; see that method for more info on how to read it back in.
     * 
     * @param col Time series to write.
     * @return Array of bytes specifying the binary, gzipped time series.
     * @throws IOException
     */
    public static byte[] writeDataToByteArray(final Collection<TimeSeriesArray> col,
                                              final boolean gzipped) throws IOException
    {
        ByteArrayOutputStream outputStream = null;
        GZIPOutputStream gzipStream = null;
        try
        {
            outputStream = new ByteArrayOutputStream();
            if(gzipped)
            {
                gzipStream = new GZIPOutputStream(outputStream);
                writeDataToBinStream(gzipStream, col);
            }
            else
            {
                writeDataToBinStream(outputStream, col);
            }
            return outputStream.toByteArray();
        }
        finally
        {
            if(gzipStream != null)
            {
                gzipStream.close();
            }
        }
    }

    /**
     * Reads time series from a stream that contains data in a binary format generated by
     * {@link #writeDataToBinaryStream(OutputStream, Collection)}. The input stream is NOT closed after reading.<br>
     * <br>
     * If the time series to be read in are part of an ensemble, the indexing of the ensemble must be done AFTER this is
     * called. This will not index the ensemble; all time series will have the same index and ensembleId as the provided
     * template time series!<br>
     * <br>
     * Also, if the time series to be read in have different parameter ids, that must be handled after calling this. All
     * returned time series will have the same parameterId as the provided template!
     * 
     * @param template Template for all read in time series. This template is used to call
     *            {@link TimeSeriesArrayTools#prepareTimeSeries(TimeSeriesArray)} to create a holder for the data.
     * @param inputStream Stream from which to read binary data.
     * @return {@link List} of {@link TimeSeriesArray} read in.
     */
    public static List<TimeSeriesArray> readDataFromBinStream(final TimeSeriesArray template,
                                                              final InputStream inputStream) throws IOException
    {
        final List<TimeSeriesArray> ts = new ArrayList<TimeSeriesArray>();

        DataInputStream dataStream = null;
        try
        {
            dataStream = new DataInputStream(inputStream);
            boolean atEndOfFile = false;
            while(!atEndOfFile)
            {
                long forecastTime = Long.MIN_VALUE;
                long startTime = Long.MIN_VALUE;
                int size = -1;

                //Reading in the forecast time is the check to determine if the end-of-file was reached legally.
                try
                {
                    forecastTime = dataStream.readLong();
                }
                catch(final EOFException e)
                {
                    atEndOfFile = true;
                }

                //Only continue if we are not at the end of the file.
                if(!atEndOfFile)
                {
                    startTime = dataStream.readLong();
                    size = dataStream.readInt();

                    //Create a time series to contain results, set its forecast time, and clear its values.  Then
                    //read in the values assigning them to time starting from startTime.  Read in only size elements.
                    final TimeSeriesArray series = TimeSeriesArrayTools.prepareTimeSeries(template);
                    ((DefaultTimeSeriesHeader)series.getHeader()).setForecastTime(forecastTime);
                    series.clear();
                    for(int i = 0; i < size; i++)
                    {
                        series.put(startTime + i * series.getHeader().getTimeStep().getStepMillis(),
                                   dataStream.readFloat());
                    }
                    ts.add(series);
                }
            }

            return ts;
        }
        catch(final EOFException e)
        {
            throw new IOException("File badly formatted: end-of-file reached before expected.");
        }
    }

    /**
     * Calls {@link #readDataFromBinFile(TimeSeriesArray, File)}, initializing the {@link TimeSeriesArray} used as a
     * template based on the given header.
     * 
     * @param header The header to use.
     * @param binFile The binary file to read.
     * @return The time series found.
     * @throws IOException
     */
    public static List<TimeSeriesArray> readDataFromBinFile(final TimeSeriesHeader header,
                                                            final File binFile) throws IOException
    {
        return readDataFromBinFile(new TimeSeriesArray(header), binFile);
    }

    /**
     * Meant to work in conjunction with {@link #writeDataToBinFile(File, Collection)}, this reads the data back in, but
     * requires a template time series that contains all the header information, including location, parameter, time
     * step, and so on. The provided time series is copied to create new time series, with the forecast time and values
     * (both times and numbers) being set based on data from the file. This calls
     * {@link #readDataFromBinaryStream(TimeSeriesArray, InputStream)}.<br>
     * <br>
     * If the time series to be read in are part of an ensemble, the indexing of the ensemble must be done AFTER this is
     * called. This will not index the ensemble; all time series will have the same index and ensembleId as the provided
     * template time series!<br>
     * <br>
     * Also, if the time series to be read in have different parameter ids, that must be handled after calling this. All
     * returned time series will have the same parameterId as the provided template!
     * 
     * @param template The template to use.
     * @param binFile The file to read.
     * @return A {@link Collection} of {@link TimeSeriesArray} specifying the found time series.
     * @throws IOException
     */
    public static List<TimeSeriesArray> readDataFromBinFile(final TimeSeriesArray template,
                                                            final File binFile) throws IOException
    {
        FileInputStream inputStream = null;
        GZIPInputStream gzipStream = null;
        try
        {
            inputStream = new FileInputStream(binFile);
            gzipStream = new GZIPInputStream(inputStream);
            return readDataFromBinStream(template, gzipStream);
        }
        catch(final EOFException e)
        {
            throw new IOException("File badly formatted: end-of-file reached before expected.");
        }
        finally
        {
            if(gzipStream != null)
            {
                gzipStream.close();
            }
        }
    }

    /**
     * This does the same thing as {@link #readDataFromBinFile(TimeSeriesArray, File)}, but reads from the provided byte
     * array. See that method for more info. The byte array MUST be in a gzip, binary format.
     */
    public static List<TimeSeriesArray> readDataFromByteArray(final TimeSeriesArray template,
                                                              final byte[] bytes) throws IOException
    {
        ByteArrayInputStream inputStream = null;
        GZIPInputStream gzipStream = null;
        try
        {
            inputStream = new ByteArrayInputStream(bytes);
            gzipStream = new GZIPInputStream(inputStream);
            return readDataFromBinStream(template, gzipStream);
        }
        catch(final EOFException e)
        {
            throw new IOException("Array badly formatted: end-of-file reached before expected.");
        }
        finally
        {
            if(gzipStream != null)
            {
                gzipStream.close();
            }
        }
    }

    /**
     * @param col
     * @return {@link TimeSeriesArrays} containing all time series in the collection. Null is returned if col is empty.
     */
    public static TimeSeriesArrays convertTimeSeriesCollection(final Collection<TimeSeriesArray> col)
    {
        if(col.isEmpty())
        {
            return null;
        }

        final Iterator<TimeSeriesArray> iter = col.iterator();
        final TimeSeriesArrays results = new TimeSeriesArrays(iter.next());
        while(iter.hasNext())
        {
            results.add(iter.next());
        }
        return results;
    }

    /**
     * @param col The collection to convert, which can be a collection of {@link TimeSeriesArrays} or any subclass,
     *            including {@link TimeSeriesEnsemble}.
     * @return {@link TimeSeriesArrays} containing all time series in the collection.
     */
    public static TimeSeriesArrays convertTimeSeriesArraysCollections(final Collection<? extends TimeSeriesArrays> col)
    {
        if(col.isEmpty())
        {
            return null;
        }

        final Iterator<? extends TimeSeriesArrays> iter = col.iterator();
        TimeSeriesArrays results = null;
        while(iter.hasNext())
        {
            final TimeSeriesArrays ts = iter.next();
            if(!ts.isEmpty())
            {
                //Initialize results here...
                if(results == null)
                {
                    results = new TimeSeriesArrays(ts.get(0));
                    for(int i = 1; i < ts.size(); i++)
                    {
                        results.add(ts.get(i));
                    }
                }
                //Otherwise, just add
                else
                {
                    results.addAll(ts);
                }
            }
        }
        return results;
    }

    /**
     * See {@link #extractTimeSeries(TimeSeriesArrays, int, int)}. A negative value is passed in for lastIndex.
     * 
     * @param base {@link TimeSeriesArrays} from which to extract time series.
     * @param firstIndex Index of first time series to include. Pass in a negative number to acquire all time series.
     * @return A new {@link TimeSeriesArrays} object containing only those items in the base with indices after and
     *         including startIndex.
     */
    public static TimeSeriesArrays extractTimeSeries(final TimeSeriesArrays base, final int startIndex)
    {
        return extractTimeSeries(base, startIndex, -1);
    }

    /**
     * If both firstIndex and lastIndex are negative, all time series are extracted.
     * 
     * @param base {@link TimeSeriesArrays} from which to extract time series.
     * @param firstIndex Index of first time series to include. Pass in a negative number to acquire all time series
     *            before lastIndex.
     * @param lastIndex Index of last time series to include (note that this is inclusive!). Pass in a negative number
     *            to acquire all time series after and including firstIndex.
     * @return A new {@link TimeSeriesArrays} object containing on those items in the base with indices of base within
     *         the specified index range.
     */
    public static TimeSeriesArrays extractTimeSeries(final TimeSeriesArrays base, int firstIndex, int lastIndex)
    {
        if((firstIndex >= base.size()) || (lastIndex >= base.size()))
        {
            throw new IllegalArgumentException("First index or last index exceeds size of list, " + base.size()
                + " (first = " + firstIndex + "; last = " + lastIndex + ").");
        }

        if(firstIndex < 0)
        {
            firstIndex = 0;
        }
        if(lastIndex < 0)
        {
            lastIndex = base.size() - 1;
        }

        if(lastIndex < firstIndex)
        {
            throw new IllegalArgumentException("Last index is before first index (first = " + firstIndex + "; last = "
                + lastIndex + ").");
        }

        final TimeSeriesArrays results = new TimeSeriesArrays(base.get(firstIndex));
        for(int i = firstIndex + 1; i <= lastIndex; i++)
        {
            results.add(base.get(i));
        }
        return results;
    }

    /**
     * Adds the time series in the provided iterable to the {@link TimeSeriesArrays} passed as an argument. The
     * {@link TimeSeriesArrays} addAll methods only work with arrays or other {@link TimeSeriesArrays} instances. This
     * one will work with any {@link Iterable}, including collections, lists, and so on.
     */
    public static void addAll(final TimeSeriesArrays ts, final Iterable<TimeSeriesArray> toAdd)
    {
        for(final TimeSeriesArray oneTS: toAdd)
        {
            ts.add(oneTS);
        }
    }

    /**
     * I'm not sure how this is different than {@link #convertTimeSeriesCollection(Collection)}, except it is much
     * simpler due to indexing being usable..
     * 
     * @param ts
     * @return A {@link TimeSeriesArrays} containing all of the time series.
     */
    public static TimeSeriesArrays convertListOfTimeSeriesToTimeSeriesArrays(final List<TimeSeriesArray> ts)
    {
        return convertListOfTimeSeriesToTimeSeriesArrays(DefaultTimeSeriesHeader.class, ts);
//        final TimeSeriesArrays results = new TimeSeriesArrays(ts.get(0));
//        if(ts.size() > 1)
//        {
//            addAll(results, ListTools.createSubList(ts, 1, true, -1));
//        }
//        return results;
    }

    /**
     * I'm not sure how this is different than {@link #convertTimeSeriesCollection(Collection)}, except it is much
     * simpler due to indexing being usable..
     * 
     * @param ts
     * @return A {@link TimeSeriesArrays} containing all of the time series.
     */
    public static TimeSeriesArrays convertListOfTimeSeriesToTimeSeriesArrays(final Class tsHeaderClass,
                                                                             final List<TimeSeriesArray> ts)
    {
        final TimeSeriesArrays results = new TimeSeriesArrays(tsHeaderClass, ts.size());
        addAll(results, ts);
        return results;
    }

    /**
     * Convert the provided {@link Iterable} to {@link TimeSeriesArrays}.
     * 
     * @param ts Must not be empty!
     * @return A {@link TimeSeriesArrays} containing all of the time series.
     */
    public static TimeSeriesArrays convertToTimeSeriesArrays(final Iterable<TimeSeriesArray> ts)
    {
        final Iterator<TimeSeriesArray> iter = ts.iterator();
        if(!iter.hasNext())
        {
            throw new IllegalArgumentException("No time series were provided (empty iterable).");
        }
        final TimeSeriesArrays results = new TimeSeriesArrays(iter.next());
        while(iter.hasNext())
        {
            results.add(iter.next());
        }
        return results;
    }

    /**
     * I tried to use the {@link TimeSeriesArrays#toArray()} combined with {@link Arrays#asList(Object...)}, but it
     * failed. So I created a manual converter.
     * 
     * @param ts
     * @return Return the time series as a {@link List}. This wraps {@link Arrays#asList(Object...)}.
     */
    public static List<TimeSeriesArray> convertTimeSeriesArraysToList(final TimeSeriesArrays ts)
    {
        final List<TimeSeriesArray> results = new ArrayList<>();
        for(int i = 0; i < ts.size(); i++)
        {
            results.add(ts.get(i));
        }
        return results;
    }

//    /**
//     * This is used in HEFSEnsPost testing. It should be refactored to call the {@link TimeSeriesComparisonUtilities}
//     * methods.
//     * 
//     * @param tsList1
//     * @param tsList2
//     * @return
//     */
//    public static boolean compareTwoTimeSeriesArraysOrderMatters(final TimeSeriesArrays tsList1,
//                                                                 final TimeSeriesArrays tsList2)
//    {
//        if(tsList1.size() != tsList2.size())
//        {
//            System.err.println("The two lists of RegularTimeSeries are of unequal length: " + tsList1.size() + " and "
//                + tsList2.size());
//            return false;
//        }
//
//        for(int i = 0; i < tsList1.size(); i++)
//        {
//            try
//            {
//                TimeSeriesArrayTools.checkTimeSeriesEqual(tsList1.get(i), tsList2.get(i), 0.0001d);
//            }
//            catch(final Exception e)
//            {
//                System.err.println("Time series in index " + i + " are not equal: " + e.getMessage());
//                return false;
//            }
//        }
//        return true;
//    }

    /**
     * Trims missing data, where missing are denoted with NaN vaues, from the beginning and end of a time series. See
     * {@link TimeSeriesArrayTools#trimMissingValuesFromBeginningAndEndOfTimeSeries(TimeSeriesArray)}.
     * 
     * @param ts Time series to trim.
     * @return Trimmed time series created via
     *         {@link TimeSeriesArrayTools#trimMissingValuesFromBeginningAndEndOfTimeSeries(TimeSeriesArray)}.
     */
    public static TimeSeriesArrays trimMissingValuesFromBeginningAndEndOfTimeSeries(final TimeSeriesArrays ts)
    {
        for(int i = 0; i < ts.size(); i++)
        {
            ts.set(i, TimeSeriesArrayTools.trimMissingValuesFromBeginningAndEndOfTimeSeries(ts.get(i)));
        }
        return ts;
    }

    /**
     * Conversion is done in place. This method will throw an exception the first time a time series fails to convert
     * due to incompatible units. This means some time series may be converted while others are not if one in the middle
     * is incompatible. So, it is critical that all time series in the TimeSeriesArrays passed in are of unit types
     * compatible with the target unit, otherwise. See
     * {@link TimeSeriesArrayTools#convertUnits(TimeSeriesArray, String)}.
     * 
     * @param ts Time series to convert. The base unit is specified in the header.
     * @param newUnit Target unit.
     * @throws Exception If the units are not compatible.
     */
    public static void convertUnits(final TimeSeriesArrays ts, final String unit) throws Exception
    {
        for(int i = 0; i < ts.size(); i++)
        {
            TimeSeriesArrayTools.convertUnits(ts.get(i), unit);
        }
    }

    /**
     * @return A copy of the passed in time series in which all headers are {@link DefaultTimeSeriesHeader} instances.
     */
    public static TimeSeriesArrays copyTimeSeries(final TimeSeriesArrays tss)
    {
        final TimeSeriesArrays results = new TimeSeriesArrays(DefaultTimeSeriesHeader.class, 1);
        for(int i = 0; i < tss.size(); i++)
        {
            results.add(TimeSeriesArrayTools.copyTimeSeries(tss.get(i)));
        }
        return results;
    }

    /**
     * Replace all values in the time series with another value. See
     * {@link TimeSeriesArrayTools#replaceAllInstancesOfValues(TimeSeriesArray, float, float)}.
     * 
     * @param ts Time series.
     * @param originalValue Value to be replaced.
     * @param newValue Replacement value.
     */
    public static void replaceAllInstancesOfValues(final TimeSeriesArrays ts,
                                                   final float originalValue,
                                                   final float newValue)
    {
        for(int i = 0; i < ts.size(); i++)
        {
            TimeSeriesArrayTools.replaceAllInstancesOfValue(ts.get(i), originalValue, newValue);
        }
    }

    /**
     * Calls {@link TimeSeriesArrayTools#containsMissingValues(TimeSeriesArray)}.
     * 
     * @param ts To check.
     * @return True if any value for any time series is missing. False if not.
     */
    public static boolean containsMissingValues(final TimeSeriesArrays ts)
    {
        for(int i = 0; i < ts.size(); i++)
        {
            if(TimeSeriesArrayTools.containsMissingValues(ts.get(i)))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * Calls {@link #containsMissingValues(TimeSeriesArrays)}.
     */
    public static boolean containsMissingValues(final Collection<TimeSeriesArray> ts)
    {
        return containsMissingValues(TimeSeriesArraysTools.convertTimeSeriesCollection(ts));
    }

    /**
     * @param ts Time series to search.
     * @param time Time to look for.
     * @param treatAsMissingIfOutsideTimeSeries If true, then if the time provided is outside the time period covered by
     *            one of the time series, it is treated as missing.
     * @return If the value at the provided time is missing or not present in any one of the time series provided. If
     *         the time is outside the period of the time series, the aforementioned flag dictates the behavior.
     */
    public static boolean isAnyValueMissingAtTime(final Collection<TimeSeriesArray> ts,
                                                  final long time,
                                                  final boolean treatAsMissingIfOutsideTimeSeries)
    {
        for(final TimeSeriesArray tsa: ts)
        {
            if((time < tsa.getStartTime()) || (time > tsa.getEndTime()))
            {
                if(treatAsMissingIfOutsideTimeSeries)
                {
                    return true;
                }
            }
            else
            {
                final int indexToCheck = TimeSeriesArrayTools.getIndexOfTime(tsa, time);
                if(indexToCheck < 0)
                {
                    return true;
                }
                else if(TimeSeriesArrayTools.isMissing(tsa, indexToCheck))
                {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * @return True if all provided time series are missing, false if any one is not empty.
     */
    public static boolean areAllMissing(final TimeSeriesArrays ts)
    {
        for(int i = 0; i < ts.size(); i++)
        {
            if(!TimeSeriesArrayTools.isAllMissing(ts.get(i)))
            {
                return false;
            }
        }
        return true;
    }

    /**
     * @return Results of {@link #areAllMissing(TimeSeriesArrays)}.
     */
    public static boolean areAllMissing(final Collection<TimeSeriesArray> ts)
    {
        return areAllMissing(TimeSeriesArraysTools.convertTimeSeriesCollection(ts));
    }

    /**
     * Replace all missing values in a time series, NaN, with a new value to denote missing. See
     * {@link TimeSeriesArrayTools#replaceAllMissingValuesWithValue(TimeSeriesArray, float)}.
     * 
     * @param ts The time series.
     * @param value The value to be used in place of all NaN (missing) values.
     */
    public static void replaceAllMissingValuesWithValue(final TimeSeriesArrays ts, final float value)
    {
        for(int i = 0; i < ts.size(); i++)
        {
            TimeSeriesArrayTools.replaceAllMissingValuesWithValue(ts.get(i), value);
        }
    }

    /**
     * Performs a binary search.
     * 
     * @param ts {@link List} of ts to search (must be sorted in ascending order).
     * @param forecastTime Forecast time to search for.
     * @return Index of the found time series. If the value is negative, it is -(insertion point - 1), where the
     *         insertion point is where the forecastTime would be inserted into the list, or the index of the first time
     *         series with a forecast time after the desired time.
     */
    public static int searchByForecastTime(final List<TimeSeriesArray> list, final long forecastTime)
    {
        int low = 0;
        int high = list.size() - 1;

        while(low <= high)
        {
            final int mid = (low + high) >>> 1; //Divides by two and truncates... stole this from Collections.binarySearch.
            final TimeSeriesArray ts = list.get(mid);
            if(ts.getHeader().getForecastTime() < forecastTime)
            {
                low = mid + 1;
            }
            else if(ts.getHeader().getForecastTime() > forecastTime)
            {
                high = mid - 1;
            }
            else
            {
                return mid;
            }
        }

        return -(low + 1);

    }

    /**
     * @return The smallest time step of any time series in the collection. It calls
     *         {@link TimeSeriesHeader#getTimeStep()} to get the step and {@link TimeStep#getStepMillis()} to gets its
     *         millis.
     */
    public static long findSmallestTimeStep(final Collection<TimeSeriesArray> arrays)
    {
        long step = Long.MAX_VALUE;
        for(final TimeSeriesArray tsa: arrays)
        {
            if(tsa.getHeader().getTimeStep().getStepMillis() < step)
            {
                step = tsa.getHeader().getTimeStep().getStepMillis();
            }
        }
        return step;
    }

    /**
     * Returns the minimum and maximum times for which data is recorded, using the methods
     * {@link TimeSeriesArray#getStartTime()} and {@link TimeSeriesArray#getEndTime()} to determine the times to check.
     * 
     * @param arrays the arrays to search; empty arrays are skipped.
     * @return the minimum and maximum times for which data is stored in the array (data might be missing). If the times
     *         are illegal, i.e., no time series has any data, then null is returned.
     */
    public static long[] getRangeOfTimes(final Collection<TimeSeriesArray> arrays)
    {
        final long[] times = {Long.MAX_VALUE, Long.MIN_VALUE};

        for(final TimeSeriesArray tsa: arrays)
        {
            if(!tsa.isEmpty())
            {
                final long startTime = tsa.getStartTime();
                if(startTime < times[0])
                {
                    times[0] = startTime;
                }
                if(startTime > times[1])
                {
                    times[1] = startTime;
                }

                final long endTime = tsa.getEndTime();
                if(endTime < times[0])
                {
                    times[0] = endTime;
                }
                if(endTime > times[1])
                {
                    times[1] = endTime;
                }
            }
        }

        // If range is illegal, return null instead.
        if(times[0] > times[1])
        {
            return null;
        }
        return times;
    }

    /**
     * @return A {@link LinkedHashMap} of forecast time mapped to {@link TimeSeriesArrays} such that each contains the
     *         time series that have the same forecast time. Typically, this means they form an ensemble if they all
     *         share the same location and data type.
     */
    public static LinkedHashMap<Long, TimeSeriesArrays> createMapOfForecastTimeToTimeSeries(final Collection<TimeSeriesArray> tss)
    {
        final LinkedHashMap<Long, TimeSeriesArrays> resultsMap = new LinkedHashMap<>();
        for(final TimeSeriesArray ts: tss)
        {
            TimeSeriesArrays oneTimeTS = resultsMap.get(ts.getHeader().getForecastTime());
            if(oneTimeTS == null)
            {
                oneTimeTS = new TimeSeriesArrays(ts);
                resultsMap.put(ts.getHeader().getForecastTime(), oneTimeTS);
            }
            else
            {
                oneTimeTS.add(ts);
            }
        }
        return resultsMap;
    }

    /**
     * @return A {@link LinkedHashMap} of forecast time to time series, with only one time series allowed per forecast
     *         time. The last one found with a forecast time is the one that will be kept in the map when it is
     *         returned.
     */
    public static LinkedHashMap<Long, TimeSeriesArray> createMapOfForecastTimeToSingleTimeSeries(final Collection<TimeSeriesArray> tss)
    {
        final LinkedHashMap<Long, TimeSeriesArray> resultsMap = new LinkedHashMap<>();
        for(final TimeSeriesArray ts: tss)
        {
            resultsMap.put(ts.getHeader().getForecastTime(), ts);
        }
        return resultsMap;
    }

    /**
     * Returns the minimum and maximum forecast time from the array
     * 
     * @param arrays the arrays to search for forecast times through
     * @param ignoreIllegal if true, ignores forecast times <= 0
     * @return the minimum and maximum forecast times in an array, or null if there were no times
     */
    public static long[] getRangeOfForecastTimes(final Collection<TimeSeriesArray> arrays, final boolean ignoreIllegal)
    {
        final long[] times = {Long.MAX_VALUE, Long.MIN_VALUE};

        for(final TimeSeriesArray tsa: arrays)
        {
            final long ftime = tsa.getHeader().getForecastTime();
            if(!ignoreIllegal || ((ftime != Long.MIN_VALUE) && (ftime != Long.MAX_VALUE)))
            {
                if(ftime < times[0])
                {
                    times[0] = ftime;
                }
                if(ftime > times[1])
                {
                    times[1] = ftime;
                }
            }
        }

        // If range is illegal, return null instead.
        if(times[0] > times[1])
        {
            return null;
        }
        return times;
    }

    /**
     * Calls {@link TimeSeriesArrayTools#shift(TimeSeriesArray, long)} for each time series.
     */
    public static void shift(final Collection<TimeSeriesArray> tsToShift, final long shiftInMillis)
    {
        for(final TimeSeriesArray ts: tsToShift)
        {
            TimeSeriesArrayTools.shift(ts, shiftInMillis);
        }
    }

    /**
     * Calls {@link #shift(Collection, long)} after converting the TimeSeriesArrays.
     */
    public static void shift(final TimeSeriesArrays tsToShift, final long shiftInMillis)
    {
        shift(convertTimeSeriesArraysToList(tsToShift), shiftInMillis);
    }

    /**
     * Calls {@link TimeSeriesArrayTools#extendFromOtherTimeSeries(TimeSeriesArray, TimeSeriesArray)} for each element
     * in the provided arrays. For each time series in base, the time series at the same index from extension will be
     * used to extend it.
     */
    public static void extendFromOtherTimeSeries(final TimeSeriesArrays base, final TimeSeriesArrays extension)
    {
        for(int i = 0; i < base.size(); i++)
        {
            TimeSeriesArrayTools.extendFromOtherTimeSeries(base.get(i), extension.get(i));
        }
    }

    /**
     * @param tss The time series to modify.
     * @param parameterId The new parameter id.
     */
    public static void setAllParameterIds(final List<TimeSeriesArrays> tss, final String parameterId)
    {
        for(final TimeSeriesArrays ts: tss)
        {
            setAllParameterIds(ts, parameterId);
        }
    }

    /**
     * @param ts The time series to modify.
     * @param parameterId The new parameter id. The time series header must be an instance of
     *            {@link DefaultTimeSeriesHeader}, which includes {@link PiTimeSeriesHeader}.
     */
    public static void setAllParameterIds(final TimeSeriesArrays ts, final String parameterId)
    {
        for(int i = 0; i < ts.size(); i++)
        {
            ((DefaultTimeSeriesHeader)ts.get(i).getHeader()).setParameterId(parameterId);
        }
    }

    /**
     * @param ts The time series to modify.
     * @param ensembleId The new ensemble id. The time series header must be an instance of
     *            {@link DefaultTimeSeriesHeader}, which includes {@link PiTimeSeriesHeader}.
     */
    public static void setAllEnsembleIds(final TimeSeriesArrays ts, final String ensembleId)
    {
        for(int i = 0; i < ts.size(); i++)
        {
            ((DefaultTimeSeriesHeader)ts.get(i).getHeader()).setEnsembleId(ensembleId);
        }
    }

    /**
     * @return True if all time series provided have the same start and end.
     */
    public static boolean areTimeSeriesStartAndEndAllIdentical(final Collection<TimeSeriesArray> tss)
    {
        final TimeSeriesArray firstTS = ListTools.first(tss);
        for(final TimeSeriesArray ts: tss)
        {
            if((ts.getStartTime() != firstTS.getStartTime() || (ts.getEndTime() != firstTS.getEndTime())))
            {
                return false;
            }
        }
        return true;
    }

    /**
     * Finds the largest start time and smallest end time of the provided time series and trims all time series to the
     * same times. This uses {@link TimeSeriesArray#subArray(Period)}.
     * 
     * @param tss Time series to trim.
     * @return {@link List} of {@link TimeSeriesArray} that all share the same reduced start time and end time. If null
     *         is returned, then either only one time series was provided or trimming would result in empty time series
     *         because the largest start time is not smaller than the smallest end time.
     */
    public static List<TimeSeriesArray> trimTimeSeriesToHaveSameStartAndEnd(final Collection<TimeSeriesArray> tss)
    {
        if((tss.size() < 2) || (areTimeSeriesStartAndEndAllIdentical(tss)))
        {
            return null;
        }

        //Find the times.
        long startTime = Long.MIN_VALUE;
        long endTime = Long.MAX_VALUE;
        for(final TimeSeriesArray ts: tss)
        {
            if(ts.getStartTime() > startTime)
            {
                startTime = ts.getStartTime();
            }
            if(ts.getEndTime() < endTime)
            {
                endTime = ts.getEndTime();
            }
        }

        if(startTime >= endTime)
        {
            return null;
        }

        //Trim the time series.
        final List<TimeSeriesArray> results = Lists.newArrayList();
        for(final TimeSeriesArray ts: tss)
        {
            results.add(ts.subArray(new Period(startTime, endTime)));
        }
        return results;
    }

    /**
     * Sorts the provided time series in place by forecast time.
     * 
     * @param tss Time series to sort.
     * @return The passed in list of time series is returned for convenience (allowing for one liners, for example:
     *         return sortListByForecastTime(...)).
     */
    public static List<TimeSeriesArray> sortListByForecastTime(final List<TimeSeriesArray> tss)
    {
        Collections.sort(tss, new Comparator<TimeSeriesArray>()
        {
            @Override
            public int compare(final TimeSeriesArray o1, final TimeSeriesArray o2)
            {
                return Long.valueOf(o1.getHeader().getForecastTime()).compareTo(o2.getHeader().getForecastTime());
            }
        });
        return tss;
    }

    /**
     * @return All time series in the provided {@link TimeSeriesArrays} with the given location id.
     */
    public static TimeSeriesArrays getTimeSeriesForLocationId(final TimeSeriesArrays tss, final String locationId)
    {
        final List<TimeSeriesArray> results = new ArrayList<>();
        for(int i = 0; i < tss.size(); i++)
        {
            if(tss.get(i).getHeader().getLocationId().equals(locationId))
            {
                results.add(tss.get(i));
            }
        }
        return TimeSeriesArraysTools.convertListOfTimeSeriesToTimeSeriesArrays(results);
    }

    /**
     * This is a test main used for converting time series. Modify as needed.
     * 
     * @param arg List of argument to be specified via eclipse and likely corresponding to a file on the file system. Be
     *            sure to include a directory in all path names, even if its the working directory. FEWS time series
     *            reader is pretty stoopid and unable to see that a file with no directory is supposed to assume the
     *            working directory.
     */
    public static void main(final String[] arg)
    {
        try
        {
            final TimeSeriesArrays tss = TimeSeriesArraysTools.readFromFile(new File("./tempts.xml"));
            TimeSeriesArraysTools.writeToFile(new File("./tempts.fi"), tss);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
        }

    }

    /**
     * @param tss Time series to convert.
     * @param toMetric True to convert to metric units, false to English.
     */
    public static void convertTimeSeriesUnits(final TimeSeriesArrays tss, final boolean toMetric)
    {
        for(int i = 0; i < tss.size(); i++)
        {
            final TimeSeriesArray ts = tss.get(i);
            final MeasuringUnit baseUnit = MeasuringUnit.getMeasuringUnit(ts.getHeader().getUnit().toUpperCase());
            MeasuringUnit targetUnit = null;
            if(!toMetric)
            {
                targetUnit = MeasuringUnit.getEnglishUnit(baseUnit);
            }
            else
            {
                targetUnit = MeasuringUnit.getMetricUnit(baseUnit);
            }

            if(targetUnit != null)
            {
                try
                {
                    TimeSeriesArrayTools.convertUnits(ts, targetUnit.getName());
                }
                catch(final Throwable t)
                {
                    LOG.error("INTERNAL ERROR converting ts units (will leave units unchanged); reason: "
                        + t.getMessage());
                }
            }
        }
    }
}
