package ohd.hseb.hefs.utils.tsarrays;

import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.function.Consumer;

import nl.wldelft.util.Period;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import nl.wldelft.util.timeseries.TimeSeriesHeader;
import ohd.hseb.hefs.utils.tools.GeneralTools;
import ohd.hseb.hefs.utils.tsarrays.agg.AggregationTools;
import ohd.hseb.hefs.utils.tsarrays.agg.TimeSeriesAggregationException;
import ohd.hseb.util.fews.ensmodels.FewsEnsembleException;
import ohd.hseb.util.misc.HCalendar;

import com.google.common.base.Objects;

/**
 * Wrapper on TimeSeriesArrays that treats the time series as an ensemble. This is for use only with a standard forecast
 * ensemble where all added members are compatible according to
 * {@link #areTimeSeriesCompatible(TimeSeriesArray, TimeSeriesArray)}. Thus, all members must have the same location id,
 * forecast time, start/end times, etc.<br>
 * <br>
 * Do NOT use this to store a lagged ensemble, unless you plan to force the members pass the compatibility check.
 * 
 * @author hank.herr
 */
public class TimeSeriesEnsemble extends TimeSeriesArrays implements Cloneable, Iterable<TimeSeriesArray>
{
    /**
     * Constant used for a standard historical water year for passing into
     * {@link TimeSeriesEnsemble#TimeSeriesEnsemble(TimeSeriesArray, long, long, int, int, Calendar, String)}.
     */
    public final static Calendar STANDARD_HISTORICAL_WATER_YEAR = HCalendar.processDate("1990-10-01 00:00:00 GMT");

    private TimeSeriesArray _memberTemplateTS;

    private String _ensembleId;

    /**
     * The times associated with each member. This mapping is built up as needed; entries are added the first time they
     * are needed.
     */
    private HashMap<Integer, long[]> _ensembleMemberIndexToTimesMap;

    /**
     * The values for an ensemble member. This mapping is built up as needed; entries are added the first time they are
     * needed.
     */
    private HashMap<Integer, double[]> _ensembleMemberIndexToValuesMap;

    /**
     * The values for a time step. Time series index refers to an index in time (i.e. the array or index of a value
     * within a RegularTimeSeries list of values), not a member index. This mapping is built up as needed; entries are
     * added the first time they are needed.
     */
    private HashMap<Integer, double[]> _timeSeriesIndexToValuesMap;

    /**
     * If true, the end times of the members are checked for equality. If false, they are not checked, so that members
     * can have different lengths.
     */
    private boolean _checkEndTimesEqual = true;

    public TimeSeriesEnsemble()
    {
        super(new DefaultTimeSeriesHeader().getClass(), 50);
    }

    public TimeSeriesEnsemble(final int initialCapacity)
    {
        super(new DefaultTimeSeriesHeader().getClass(), initialCapacity);
    }

    /**
     * @param ts TimeSeriesArrays to be placed in this ensemble. The ensembleId is set based on the first member
     */
    public TimeSeriesEnsemble(final TimeSeriesArrays members) throws TimeSeriesEnsembleException
    {
        //replace the clone() method by duplicate()
        super(members.get(0).duplicate());
        for(int i = 1; i < members.size(); i++)
        {
            addMember(members.get(i));
        }
        _ensembleId = members.get(0).getHeader().getEnsembleId();
        validateConsistencyOfEnsembleMembers();
        synchronizeListOrderToMemberIndex();
    }

    /**
     * Extracts members from the given time series and sorts list according to member index.
     * 
     * @param ensembleId Ensemble id to extract
     * @param timeSeries Time series providing members.
     */
    public TimeSeriesEnsemble(final String ensembleId, final TimeSeriesArrays timeSeries) throws TimeSeriesEnsembleException
    {
        super(timeSeries.getHeaderClass(), 1);
        for(int i = 0; i < timeSeries.size(); i++)
        {
            if(Objects.equal(ensembleId, timeSeries.get(i).getHeader().getEnsembleId()))
            {
                this.addMember(timeSeries.get(i));
            }
        }
        _ensembleId = ensembleId;
        validateConsistencyOfEnsembleMembers();
        synchronizeListOrderToMemberIndex();
    }

    /**
     * Extracts members from the given time series and sorts list according to member index.
     * 
     * @param ensembleId Ensemble id to extract
     * @param timeSeries Time series providing members.
     */
    public TimeSeriesEnsemble(final String ensembleId, final List<TimeSeriesArray> timeSeries) throws TimeSeriesEnsembleException
    {
        super(timeSeries.get(0).getHeader().getClass(), 1);
        for(int i = 0; i < timeSeries.size(); i++)
        {
            if(timeSeries.get(i).getHeader().getEnsembleId().equals(ensembleId))
            {
                this.addMember(timeSeries.get(i));
            }
        }
        _ensembleId = ensembleId;
        validateConsistencyOfEnsembleMembers();
        synchronizeListOrderToMemberIndex();
    }

    /**
     * Calls {@link TimeSeriesEnsemble#TimeSeriesEnsemble(TimeSeriesArray, long, long, int, int, Calendar, String)}
     * passing in null for the {@link Calendar} if useHistoricalWaterYears is false. Otherwise, it passes in
     * {@link #STANDARD_HISTORICAL_WATER_YEAR}, indicating the use of a standard water year.
     */
    public TimeSeriesEnsemble(final TimeSeriesArray longSeries,
                              final long forecastTime,
                              final long numberOfValesPerMember,
                              final int initialYear,
                              final int lastYear,
                              final boolean useHistoricalWaterYears,
                              final String ensembleId) throws Exception
    {
        this(longSeries,
             forecastTime,
             numberOfValesPerMember,
             initialYear,
             lastYear,
             useHistoricalWaterYears ? STANDARD_HISTORICAL_WATER_YEAR : null, //Using this notation to simplify stuff.
             ensembleId);
    }

    /**
     * Constructs an ensemble by cutting out parts of the given long time series. This is used to construct HS
     * ensembles.
     * 
     * @param longSeries Long, possibly calibration or historical, time series from which to cutout the ensemble.
     *            Members will have member indices assigned based on the year.
     * @param forecastTime The forecast time of the time series to construct. Should be a time in the time series or one
     *            step just prior to the start value of the time series.
     * @param numberOfValesPerMember The number of values to include in each member. The first value is assumed to be
     *            one step after the forecastTime. Each value after that proceeds one time step at a time.
     * @param initialYear The initial year of the ensemble to construct.
     * @param lastYear Last year of the ensemble to construct.
     * @param memberIndexingCal Defines the first day of the member indexing calendar year. For a given T0, if the
     *            month/day of that T0 is AFTER the month/day within this {@link Calendar}, then the calendar year for
     *            the first member and first value of the ensemble is set to be one LESS than the initial year. For
     *            example, a standard historical water year is defined such that the 1950 water year is 10/1/1949 -
     *            9/30/1950. Thus, the 1950 member index if T0 is between Oct and Dec will start in 1949. Pass in null
     *            to use a standard calendar year (1/1 - 12/31).
     * @param ensembleId The ensemble id to assign to the constructed ensemble.
     */
    public TimeSeriesEnsemble(final TimeSeriesArray longSeries,
                              final long forecastTime,
                              final long numberOfValesPerMember,
                              final int initialYear,
                              final int lastYear,
                              final Calendar memberIndexingCal,
                              final String ensembleId) throws Exception
    {
        super(DefaultTimeSeriesHeader.class, lastYear - initialYear + 1);
        _ensembleId = ensembleId;
        final int numberOfMembers = lastYear - initialYear + 1;

        //Determine the firstCalendarYear.
        final Calendar forecastTimeCal = HCalendar.computeCalendarFromMilliseconds(forecastTime);
        int firstCalendarYear = initialYear;
        if(memberIndexingCal != null)
        {
            final int fcstMonth = forecastTimeCal.get(Calendar.MONTH);
            final int memberIndexingMonth = memberIndexingCal.get(Calendar.MONTH);
            if((fcstMonth > memberIndexingMonth)
                || ((fcstMonth == memberIndexingMonth) && (forecastTimeCal.get(Calendar.DAY_OF_MONTH) >= memberIndexingCal.get(Calendar.DAY_OF_MONTH))))
            {
                firstCalendarYear--;
            }
        }

        //The size of this tracks the current member index.
        while(this.size() < numberOfMembers)
        {
            //Compute the times for the member.
            final Calendar sourceCal = (Calendar)forecastTimeCal.clone();
            sourceCal.set(Calendar.YEAR, firstCalendarYear + this.size());
            final long sourceStartMillis = sourceCal.getTimeInMillis()
                + longSeries.getHeader().getTimeStep().getStepMillis();

            //Determine the start index.  This will never be less than 0.
            final int startIndex = longSeries.firstIndexAfterOrAtTime(sourceStartMillis); //TODO should be an exact match!

            //If the member is to start before the beginning of long series or ends after the end, throw an error.
            if((sourceStartMillis < longSeries.getStartTime()) || (startIndex < 0)
                || (startIndex + numberOfValesPerMember > longSeries.size()))
            {
                throw new Exception("Ensemble could not be constructed for " + numberOfValesPerMember
                    + " time steps starting one step after " + HCalendar.buildDateTimeTZStr(sourceCal)
                    + ", because the long time series, which starts on "
                    + HCalendar.buildDateTimeTZStr(longSeries.getStartTime()) + " and ends on "
                    + HCalendar.buildDateTimeTZStr(longSeries.getEndTime())
                    + ", does not include all the required data.");
            }

            //Build the member.
            final TimeSeriesArray member = TimeSeriesArrayTools.prepareTimeSeries(longSeries);
            for(int i = 0; i < numberOfValesPerMember; i++)
            {
                member.putValue(forecastTime + (1 + i) * longSeries.getHeader().getTimeStep().getStepMillis(),
                                longSeries.getValue(startIndex + i));
            }

            //All member should include the same forecast time.  Also set the ensemble info here.
            ((DefaultTimeSeriesHeader)member.getHeader()).setForecastTime(forecastTime);
            ((DefaultTimeSeriesHeader)member.getHeader()).setEnsembleId(getEnsembleId());
            ((DefaultTimeSeriesHeader)member.getHeader()).setEnsembleMemberIndex(initialYear + this.size());

            this.add(member);
        }
    }

    /**
     * Constructor copies members from the base via
     * {@link TimeSeriesArrayTools#copyTimeSeries(TimeSeriesArray, long, long, long)}, constructing an ensemble of
     * subseries.
     * 
     * @param base The base ensemble.
     * @param forecastTime The new forecastTime.
     * @param startTime The new start time.
     * @param endTime The new end time.
     */
    public TimeSeriesEnsemble(final TimeSeriesEnsemble base,
                              final long forecastTime,
                              final long startTime,
                              final long endTime)
    {
        super(DefaultTimeSeriesHeader.class, base.size());

        _ensembleId = base.getEnsembleId();
        for(int i = 0; i < base.size(); i++)
        {
            final TimeSeriesArray baseTS = base.get(i);
            final TimeSeriesArray member = TimeSeriesArrayTools.copyTimeSeries(baseTS, forecastTime, startTime, endTime);
            add(member);
        }
    }

    /**
     * Wraps {@link TimeSeriesArraysTools#convertTimeSeriesArraysToList(TimeSeriesArrays)}.
     * 
     * @return List of {@link TimeSeriesArray} instances contained in this.
     */
    public List<TimeSeriesArray> asList()
    {
        return TimeSeriesArraysTools.convertTimeSeriesArraysToList(this);
    }

    public void isTimeSeriesCompatible(final TimeSeriesArray ts) throws TimeSeriesEnsembleException
    {
        if(isEmpty())
        {
            return;
        }
        areTimeSeriesCompatible(ts, get(0), _checkEndTimesEqual);
    }

    /**
     * @throws TimeSeriesEnsembleException If the members are not consistent. This checks forecast time, start time, end
     *             time, location id, qualifier ids, ensemble id, time step, parameter id, and parameter type.
     */
    public void validateConsistencyOfEnsembleMembers() throws TimeSeriesEnsembleException
    {
        int i;
        for(i = 1; i < this.size(); i++)
        {
            try
            {
                isTimeSeriesCompatible(get(i));
            }
            catch(final TimeSeriesEnsembleException e)
            {
                throw new TimeSeriesEnsembleException("Ensemble member with member index "
                    + get(i).getHeader().getEnsembleMemberIndex() + " is not compatible with that with member index "
                    + get(0).getHeader().getEnsembleMemberIndex() + ": " + e.getMessage());
            }

        }
    }

    /**
     * Sets the ensembleId to be the id within this FewsEnsemble object. Sets the time series type, location id, long
     * name, source org, source system, and file description to those values in the member template. Synchronizes the
     * member index to match the list order.<br>
     * <br>
     * Use this after building the ensemble and before outputting it. DO NOT USE IT if the ensembleMemberIndex within
     * each member is correct, but the list has not been synchronized to match the member indices. If you do, then the
     * indices will be changed to match the list order, possibly making the indices incorrect.
     * 
     * @throws Exception if any time series in the ensemble is not compatible with the units of the template.
     */
    public void applyTemplateToAllMembers() throws Exception
    {
        if(_memberTemplateTS == null)
        {
            return;
        }

        //Convert the measurements, first.
        TimeSeriesArraysTools.convertUnits(this, _memberTemplateTS.getHeader().getUnit());

        //Now set the rest.
        for(int i = 0; i < size(); i++)
        {
            final DefaultTimeSeriesHeader hdr = (DefaultTimeSeriesHeader)get(i).getHeader();

            //Use the _ensembleId as the ensemble id for each time series.  Do not use the template.
            hdr.setEnsembleId(_ensembleId);
            hdr.setParameterId(_memberTemplateTS.getHeader().getParameterId());
            hdr.setParameterName(_memberTemplateTS.getHeader().getParameterName());
            hdr.setParameterType(_memberTemplateTS.getHeader().getParameterType());
            hdr.setLocationId(_memberTemplateTS.getHeader().getLocationId());
            hdr.setLocationName(_memberTemplateTS.getHeader().getLocationName());
            hdr.setLocationDescription(_memberTemplateTS.getHeader().getLocationDescription());
            final String[] qualIds = new String[_memberTemplateTS.getHeader().getQualifierCount()];
            for(int j = 0; j < _memberTemplateTS.getHeader().getQualifierCount(); j++)
            {
                qualIds[j] = _memberTemplateTS.getHeader().getQualifierId(j);
            }
            hdr.setQualifierIds(qualIds);
        }

        //I know this synchronizing could be done in the for loop above, but I want to do it
        //via this method so that I synchronize consistently.
        synchronizeListOrderToMemberIndex();
    }

    /**
     * Synchronizes the list ordering to be the member indices of the individual time series.
     */
    public void synchronizeListOrderToMemberIndex()
    {
        final Comparator<TimeSeriesArray> comp = new Comparator<TimeSeriesArray>()
        {

            @Override
            public int compare(final TimeSeriesArray o1, final TimeSeriesArray o2)
            {
                if(o1.getHeader().getEnsembleMemberIndex() == o2.getHeader().getEnsembleMemberIndex())
                {
                    return 0;
                }
                else if(o1.getHeader().getEnsembleMemberIndex() < o2.getHeader().getEnsembleMemberIndex())
                {
                    return -1;
                }
                else
                {
                    return 1;
                }
            }
        };
        final List<TimeSeriesArray> tss = TimeSeriesArraysTools.convertTimeSeriesArraysToList(this);
        Collections.sort(tss, comp);
        clear();
        for(final TimeSeriesArray ts: tss)
        {
            try
            {
                this.addMember(ts);
            }
            catch(final TimeSeriesEnsembleException e)
            {
                // Should never happen
                e.printStackTrace();
            }
        }
    }

    /**
     * Get sub time series.
     */
    public TimeSeriesEnsemble generateSubSeries(final long startTime, final long endTime)
    {
        final Period pd = new Period(startTime, endTime);
        try
        {
            return new TimeSeriesEnsemble(subArrays(pd));
        }
        catch(final Exception e)
        {
            //Should never happen!
            e.printStackTrace();
            return null;
        }
    }

    /**
     * @return The index that points to the first forecast value, or that value one time interval beyond the forecast
     *         basis time. Returns -1 if the forecast basis time is not defined, or if there are no ensemble members.
     * @throws FewsEnsembleException
     */
    public int computeIndexOfFirstForecastValue() throws TimeSeriesEnsembleException
    {
        if(this.getEarliestForecastTime() < 0)
        {
            return -1;
        }

        final long firstForecastValueTime = getEarliestForecastTime();
        if(firstForecastValueTime == Long.MIN_VALUE)
        {
            return -1;
        }
        for(int i = 0; i < get(0).size(); i++)
        {
            if(get(0).getTime(i) == firstForecastValueTime)
            {
                return i;
            }
            if(get(0).getTime(i) > firstForecastValueTime)
            {
                break;
            }
        }
        throw new TimeSeriesEnsembleException("Forecast basis time is defined, but cannot find the first forecast value "
            + "that should exist in the ensemble.");
    }

    /**
     * Assumes null for the argument to the other version of the method.
     */
    public TimeSeriesEnsemble generateForecastedEnsemble() throws TimeSeriesEnsembleException
    {
        return generateForecastedEnsemble(null);
    }

    /**
     * Generate an ensemble of time series for which each member contains only forecast values; i.e. those values
     * starting from the index returned by computeIndexOfFirstForecastValue and going to the end of the series.
     * 
     * @return modelRunStartTime The start time to assume for the first forecast value to include in the forecast
     *         ensemble. If null, it is assumed that the start time should be calculated based on the index of the first
     *         forecast value herein.
     * @return FewsEnsemble where each time series contains only those values that are forecasted values. Any value
     *         equal to or before the passed in time is discarded. It will return this if the entire series is part of
     *         the forecast ensemble or if _forecastBasisTime is null (its default value).
     * @throws FewsEnsembleException if the ensemble does not appear to be valid. That means that the first forecast
     *             value could not be found, because the basis time and forecast times do not appear to correspond
     *             correctly.
     */
    public TimeSeriesEnsemble generateForecastedEnsemble(final Long timeOfFirstValueInForecastedEnsemble) throws TimeSeriesEnsembleException
    {
        long firstValueTimeInMillis;
        if(timeOfFirstValueInForecastedEnsemble == null)
        {
            final int forecastStartIndex = computeIndexOfFirstForecastValue();
            if(forecastStartIndex <= 0)
            {
                return this;
            }
            firstValueTimeInMillis = this.getEarliestForecastTime();
        }
        else
        {
            firstValueTimeInMillis = timeOfFirstValueInForecastedEnsemble;
        }
        final TimeSeriesEnsemble subSeries = generateSubSeries(firstValueTimeInMillis, get(0).getEndTime());
        return subSeries;
    }

    /**
     * Calls {@link #aggregate(int, long, Long, String, String, String)} passing in null for the aggregationPeriodStr
     * and periodAnchorStr, forcing those to default values (asTimeStep and ending, respectively). This assumes that the
     * missing values should not be ignored and that no zero will be prefixed to the output.
     */
    public TimeSeriesEnsemble aggregate(final int type,
                                        final long startDate,
                                        final Long endDate,
                                        final String timeStepStr) throws TimeSeriesAggregationException
    {
        return aggregate(type, startDate, endDate, timeStepStr, null, null, false, false);
    }

    /**
     * All arguments are those passed through to the
     * {@link AggregationTools#aggregate(TimeSeriesArrays, long, Long, String, String, String, int)}.
     * 
     * @return A {@link TimeSeriesEnsemble} containing the aggregated members.
     * @throws TimeSeriesAggregationException
     */
    public TimeSeriesEnsemble aggregate(final int type,
                                        final long startDate,
                                        final Long endDate,
                                        final String timeStepStr,
                                        final String aggregationPeriodStr,
                                        final String periodAnchorStr,
                                        final boolean ignoreMissingValues,
                                        final boolean prefixWithZero) throws TimeSeriesAggregationException
    {
        final TimeSeriesArrays ts = AggregationTools.aggregate(this,
                                                               startDate,
                                                               endDate,
                                                               timeStepStr,
                                                               aggregationPeriodStr,
                                                               periodAnchorStr,
                                                               ignoreMissingValues,
                                                               prefixWithZero,
                                                               type);

        try
        {
            return new TimeSeriesEnsemble(ts);
        }
        catch(final TimeSeriesEnsembleException e)
        {
            //Should never happen.
            e.printStackTrace();
            throw new TimeSeriesAggregationException("INTERNAL ERROR: " + e.getMessage());
        }
    }

    /**
     * Makes use of the {@link TimeSeriesArray#subArray(Period)} method. Does this cause a significant slowdown?
     * 
     * @param ts {@link TimeSeriesArray} to add.
     * @param ensureEndTimesMatch If true adn the passed in ts has an end time that does not match the rest of the
     *            ensemble, trim either it or the ensemble to ensure it matches. It will then call
     *            {@link #isTimeSeriesCompatible(TimeSeriesArray)} which will check the rest of the fields for
     *            compatibility, including the forecast time and start time.
     * @throws TimeSeriesEnsembleException If the provided ts is not compatible after trimming.
     */
    public void addMember(final TimeSeriesArray ts, final boolean ensureEndTimesMatch) throws TimeSeriesEnsembleException
    {
        TimeSeriesArray usedTS = ts;
        if((ensureEndTimesMatch) && (!isEmpty()))
        {
            //uses sub-array stuff.
            //Modify the ts coming end using subArray to a period with the ensemble's end time, because the ensemble is shorter.
            if(getEndTime() < ts.getEndTime())
            {
                final Period newPeriod = new Period(ts.getStartTime(), getEndTime());
                usedTS = ts.subArray(newPeriod);
            }
            //Modify the ts in the ensemble using subArray for a period with the new time series' end time, because it is shorter.
            else if(getEndTime() > ts.getEndTime())
            {
                final Period newPeriod = new Period(getStartTime(), ts.getEndTime());
                for(int i = 0; i < size(); i++)
                {
                    set(i, get(i).subArray(newPeriod));
                }
            }
        }

        isTimeSeriesCompatible(usedTS);
        super.add(usedTS);
    }

    /**
     * Call {@link #addMember(TimeSeriesArray, boolean)} passing in false for trimming so that the provided time series
     * must have the same end time as the rest of the ensemble.
     * 
     * @throws TimeSeriesEnsembleException If the provided ts is not compatible.
     */
    public void addMember(final TimeSeriesArray ts) throws TimeSeriesEnsembleException
    {
        addMember(ts, false);
    }

    public void addAllMembers(final TimeSeriesArrays ts) throws TimeSeriesEnsembleException
    {
        if(ts.isEmpty())
        {
            return;
        }
        for(int i = 1; i < ts.size(); i++)
        {
            areTimeSeriesCompatible(ts.get(0), ts.get(i), _checkEndTimesEqual);
        }
        isTimeSeriesCompatible(ts.get(0));
        super.addAll(ts);
    }

    /**
     * @param index The index for which values are needed.
     * @return Array of values if found or null if the index is invalid. It is possible, though not likely, that the
     *         returned array could contain get(i).getMissingValue() values.
     */
    public double[] getEnsembleValuesForIndex(final int index)
    {
        if((index < 0) || (index > get(0).size()))
        {
            return null;
        }
        if(_timeSeriesIndexToValuesMap == null)
        {
            _timeSeriesIndexToValuesMap = new HashMap<Integer, double[]>();
        }
        if(_timeSeriesIndexToValuesMap.get(index) == null)
        {
            final double[] values = new double[this.size()];
            for(int i = 0; i < size(); i++)
            {
                values[i] = get(i).getValue(index);
            }
            return values;
        }
        return _timeSeriesIndexToValuesMap.get(index);
    }

    /**
     * Get the values across the ensemble for a particular time.
     * 
     * @param time The time to check in milliseconds.
     * @param exacthMatchingTime If false, then the time need not exactly match. In this case, the array, if the times
     *            don't exactly match, will contain the first values with a time AFTER the desired time. This follows
     *            the same logic of the aggregator which associates the time at the end of each aggregation period with
     *            that period. So, this routine will return the values associated with the period encompassing the
     *            passed in time.
     * @return Array of double representing the values found for that time. Null is returned if the time is not exactly
     *         matched or is outside the measuring time range, or if there are no ensemble members.
     */
    public double[] getEnsembleValuesForTime(final long time, final boolean exactMatchingTime)
    {
        if(size() == 0)
        {
            return null;
        }
        if((time < get(0).getStartTime()) || (time > get(0).getEndTime()))
        {
            return null;
        }

        //Find the index that corresponds to this time by brute force.
        int index = 0;
        while(index < get(0).size())
        {
            //Check 1 -- exact match.
            //Check 2 -- first time AFTER the desired time, and the caller does not need
            //an exact matching time.
            if((get(0).getTime(index) == time) || ((!exactMatchingTime) && (get(0).getTime(index) > time)))
            {
                return getEnsembleValuesForIndex(index);
            }
            //If we get here, then we the know the caller does not want an exact matching time.
            //We also know that there is no exact matching time, because we've already passed
            //the desired time, so return null.
            if(get(0).getTime(index) > time)
            {
                return null;
            }
            index++;
        }
        return null;
    }

    public long getStartTime()
    {
        if(!isEmpty())
        {
            return get(0).getStartTime();
        }
        return Long.MIN_VALUE;
    }

    public long getEndTime()
    {
        if(!isEmpty())
        {
            return get(0).getEndTime();
        }
        return Long.MIN_VALUE;
    }

    public long getForecastTime()
    {
        if(!isEmpty())
        {
            return get(0).getHeader().getForecastTime();
        }
        return Long.MIN_VALUE;
    }

    public String getLocationId()
    {
        if(!isEmpty())
        {
            return get(0).getHeader().getLocationId();
        }
        return null;
    }

    public String getParameterId()
    {
        if(!isEmpty())
        {
            return get(0).getHeader().getParameterId();
        }
        return null;
    }

    public long getTimeStepMillis()
    {
        if(!isEmpty())
        {
            return this.get(0).getHeader().getTimeStep().getStepMillis();
        }
        return Long.MIN_VALUE;
    }

    public int getTimeStepHours()
    {
        if(!isEmpty())
        {
            return (int)((double)get(0).getHeader().getTimeStep().getStepMillis() / (double)HCalendar.MILLIS_IN_HR);
        }
        return -1;
    }

    /**
     * @return The number of values in the first member. The {@link #validateConsistencyOfEnsembleMembers()} should
     *         ensure that all members have the same number of values.
     */
    public int getValueCountPerMember()
    {
        return get(0).size();
    }

    /**
     * @param memberIndex The ensemble member for which values are to be gotten.
     * @return Array of millsecond longs.
     */
    public long[] getMemberTimes(final int memberIndex)
    {
        if(_ensembleMemberIndexToTimesMap == null)
        {
            _ensembleMemberIndexToTimesMap = new HashMap<Integer, long[]>();
        }
        if(_ensembleMemberIndexToTimesMap.get(memberIndex) == null)
        {
            final long[] times = TimeSeriesEnsemble.buildTimesFromTimeSeriesArray(get(memberIndex));
            _ensembleMemberIndexToTimesMap.put(memberIndex, times);
        }
        return _ensembleMemberIndexToTimesMap.get(memberIndex);
    }

    /**
     * @param memberIndex The ensemble member for which values are to be gotten.
     * @return Array of values ready for charting.
     */
    public double[] getMemberValues(final int memberIndex)
    {
        if(_ensembleMemberIndexToValuesMap == null)
        {
            _ensembleMemberIndexToValuesMap = new HashMap<Integer, double[]>();
        }
        if(_ensembleMemberIndexToValuesMap.get(memberIndex) == null)
        {
            final double[] ySeries = TimeSeriesEnsemble.buildValuesFromTimeSeriesArray(get(memberIndex));
            _ensembleMemberIndexToValuesMap.put(memberIndex, ySeries);
        }
        return _ensembleMemberIndexToValuesMap.get(memberIndex);
    }

    public TimeSeriesArray getMemberTemplateTS()
    {
        return _memberTemplateTS;
    }

    public void setMemberTemplateTS(final TimeSeriesArray memberTemplateTS)
    {
        _memberTemplateTS = memberTemplateTS;
    }

    public String getEnsembleId()
    {
        return _ensembleId;
    }

    public void setCheckEndTimesEqual(final boolean b)
    {
        _checkEndTimesEqual = b;
    }

    public void setEnsembleId(final String ensembleId)
    {
        _ensembleId = ensembleId;
        for(int i = 0; i < size(); i++)
        {
            ((DefaultTimeSeriesHeader)get(i).getHeader()).setEnsembleId(_ensembleId);
        }
    }

    public void setParameterId(final String parameterId)
    {
        for(int i = 0; i < size(); i++)
        {
            ((DefaultTimeSeriesHeader)get(i).getHeader()).setParameterId(parameterId);
        }
    }

    @Override
    public TimeSeriesEnsemble clone()
    {
        //I'm using super.add because I know the members are valid.  No need to call addMember and validate
        //them.
        final TimeSeriesEnsemble ens = new TimeSeriesEnsemble(size());
        for(int i = 0; i < size(); i++)
        {
            // replace the clone() method by duplicate()
            ens.add(get(i).duplicate());
        }
        ens.setMemberTemplateTS(_memberTemplateTS); //This is NOT cloned.  I don't see a reason to clone a template.
        ens.setEnsembleId(_ensembleId);
        return ens;
    }

    public static long[] buildTimesFromTimeSeriesArray(final TimeSeriesArray ts)
    {
        final long[] times = new long[ts.size()];
        int j;
        for(j = 0; j < ts.size(); j++)
        {
            times[j] = ts.getTime(j);
        }
        return times;
    }

    public static double[] buildValuesFromTimeSeriesArray(final TimeSeriesArray ts)
    {
        final double[] values = new double[ts.size()];
        int j;
        for(j = 0; j < ts.size(); j++)
        {
            values[j] = ts.getValue(j);
        }
        return values;
    }

    /**
     * Checks done: {@link TimeSeriesHeader#getForecastTime()}, {@link TimeSeriesArray#getStartTime()},TimeSeriesArray
     * {@link #getEndTime()}, {@link TimeSeriesHeader#getLocationId()}, {@link TimeSeriesHeader#getQualifierId(int)},
     * {@link TimeSeriesHeader#getEnsembleId()}, {@link TimeSeriesHeader#getTimeStep()},
     * {@link TimeSeriesHeader#getParameterId()}, and {@link TimeSeriesHeader#getParameterType()}.
     * 
     * @param ts
     * @param base
     * @param checkEndTimesEqual If false, the end times are not checked for equality.
     * @throws TimeSeriesEnsembleException If the two time series do not match. The message identifies the unmatching
     *             field.
     */
    private static void areTimeSeriesCompatible(final TimeSeriesArray ts,
                                                final TimeSeriesArray base,
                                                final boolean checkEndTimesEqual) throws TimeSeriesEnsembleException
    {
        if(base.getHeader().getForecastTime() != ts.getHeader().getForecastTime())
        {
            throw new TimeSeriesEnsembleException("forecast times are different.");
        }
        if(base.getStartTime() != ts.getStartTime())
        {
            throw new TimeSeriesEnsembleException("start times are different.");
        }
        if(checkEndTimesEqual && (base.getEndTime() != ts.getEndTime()))
        {
            throw new TimeSeriesEnsembleException("end times are different.");
        }
        if(!base.getHeader().getLocationId().equalsIgnoreCase(ts.getHeader().getLocationId()))
        {
            throw new TimeSeriesEnsembleException("location ids are different.");
        }
        for(int qualIndex = 0; qualIndex < base.getHeader().getQualifierCount(); qualIndex++)
        {
            if(!base.getHeader().getQualifierId(qualIndex).equalsIgnoreCase(ts.getHeader().getQualifierId(qualIndex)))
            {
                throw new TimeSeriesEnsembleException("qualifier ids at index " + qualIndex + " are different: "
                    + base.getHeader().getQualifierId(qualIndex) + " and " + ts.getHeader().getQualifierId(qualIndex));
            }
        }

        //Ensemble id requires a null check first.
        if(!GeneralTools.checkForFullEqualityOfObjects(base.getHeader().getEnsembleId(), ts.getHeader().getEnsembleId()))
//        if(!base.getHeader().getEnsembleId().equalsIgnoreCase(ts.getHeader().getEnsembleId()))
        {
            throw new TimeSeriesEnsembleException("ensemble ids are different.");
        }

        if(base.getHeader().getTimeStep().isRegular() != ts.getHeader().getTimeStep().isRegular())
        {
            throw new TimeSeriesEnsembleException("one isRegular differs.");
        }
        if(base.getHeader().getTimeStep().isRegular())
        {
            if(base.getHeader().getTimeStep().getStepMillis() != ts.getHeader().getTimeStep().getStepMillis())
            {
                throw new TimeSeriesEnsembleException("time steps are different.");
            }
        }
        if(!base.getHeader().getParameterId().equalsIgnoreCase(ts.getHeader().getParameterId()))
        {
            throw new TimeSeriesEnsembleException("time series data type is different("
                + base.getHeader().getParameterId() + " and " + ts.getHeader().getParameterId() + ").");
        }
        if(base.getHeader().getParameterType().compareTo(ts.getHeader().getParameterType()) != 0)
        {
            throw new TimeSeriesEnsembleException("type of time series (acc or inst) is different; "
                + base.getHeader().getParameterType() + " and " + ts.getHeader().getParameterType() + ".");
        }
    }

    /**
     * Makes this thing iterable... much more useful.
     */
    @Override
    public Iterator<TimeSeriesArray> iterator()
    {
        return new Iterator<TimeSeriesArray>()
        {
            int nextPosition = 0;

            @Override
            public boolean hasNext()
            {
                return nextPosition < size();
            }

            @Override
            public TimeSeriesArray next()
            {
                return get(nextPosition++); //Adds one AFTER getting the time series
            }

            @Override
            public void remove()
            {
                throw new IllegalStateException("Cannot call remove on a TimeSeriesArrays.");
            }

            //TODO: Need to verify if this method is needed or need to remove it.
            @Override
            public void forEachRemaining(final Consumer<? super TimeSeriesArray> action)
            {
                while(hasNext())
                    action.accept(next());
            }

        };
    }

    //TODO: Need to verify it with Deltares and Hank
    // Looks like Deltares is implementing this method.
    public void forEach(final Consumer<? super TimeSeriesArray> action)
    {
        for(final TimeSeriesArray t: this)
            action.accept(t);
    }
}
