package ohd.hseb.hefs.utils.tsarrays;

import java.util.Calendar;

import org.xml.sax.Attributes;

import nl.wldelft.util.coverage.PointsGeometry;
import nl.wldelft.util.geodatum.GeoPoint;
import nl.wldelft.util.geodatum.Wgs1984Point;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.ParameterType;
import nl.wldelft.util.timeseries.SimpleEquidistantTimeStep;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesHeader;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.util.misc.HCalendar;

public class TimeSeriesHeaderInfo implements XMLReader, Comparable
{
//    private static final Logger LOG = Logger.getLogger(TimeSeriesHeaderInfo.class);

    private TimeSeriesHeader _timeSeriesHeader = null;
//    private TimeSeriesArray _timeSeries;
//    private StringBuffer _timeSeriesXMLString;

    private long _startTime = Long.MIN_VALUE;
    private long _endTime = Long.MIN_VALUE;

    private Double _readLat;
    private Double _readLon;
    private Double _readZ;

    private String _creationDateStr;
    private String _creationTimeStr;
    private long _creationTime = Long.MIN_VALUE;

    public TimeSeriesHeaderInfo()
    {
    }

    public TimeSeriesHeaderInfo(final TimeSeriesArray ts)
    {
        if(ts.size() > 0)
        {
            _startTime = ts.getStartTime();
            _endTime = ts.getEndTime();
        }
        _creationTime = ts.getHeader().getCreationTime();
        _timeSeriesHeader = ts.getHeader();
    }

    public void shiftTime(final float timeZoneShiftInHours)
    {
        final long shift = (long)(timeZoneShiftInHours * HCalendar.MILLIS_IN_HR);
        _startTime += shift;
        _endTime += shift;
        _creationTime += shift;
    }

    public TimeSeriesHeader getTimeSeriesHeader()
    {
        return _timeSeriesHeader;
    }

    public void setTimeSeriesHeader(final TimeSeriesHeader timeSeriesHeader)
    {
        this._timeSeriesHeader = timeSeriesHeader;
    }

    public long getStartTime()
    {
        return _startTime;
    }

    public long getEndTime()
    {
        return _endTime;
    }

    public long getCreationTime()
    {
        return _creationTime;
    }

    @Override
    public String toString()
    {
        String results = "TimeSeriesInformation: locationId = " + _timeSeriesHeader.getLocationId() + "; ";
        results += "parameterId = " + _timeSeriesHeader.getParameterId() + "; ";
        results += "units = " + _timeSeriesHeader.getUnit() + "; ";
        results += "startTime = " + HCalendar.buildDateTimeTZStr(HCalendar.computeCalendarFromMilliseconds(_startTime))
            + "; ";
        results +=
                "endTime = " + HCalendar.buildDateTimeTZStr(HCalendar.computeCalendarFromMilliseconds(_endTime)) + "; ";
        results += "creationTime = "
            + HCalendar.buildDateTimeTZStr(HCalendar.computeCalendarFromMilliseconds(_creationTime)) + ".";
        return results;
    }

    @Override
    public String getXMLTagName()
    {
        return "series";
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
//        _timeSeriesXMLString.append(value + "</" + elementName + ">");
        if(elementName.equals("type"))
        {
            ((DefaultTimeSeriesHeader)_timeSeriesHeader).setParameterType(ParameterType.valueOf(value.toUpperCase()));
        }
        else if(elementName.equals("locationId"))
        {
            ((DefaultTimeSeriesHeader)_timeSeriesHeader).setLocationId(value);
        }
        else if(elementName.equals("parameterId"))
        {
            ((DefaultTimeSeriesHeader)_timeSeriesHeader).setParameterId(value);
        }
        else if(elementName.equals("stationName"))
        {
            ((DefaultTimeSeriesHeader)_timeSeriesHeader).setLocationName(value);
        }
        else if(elementName.equals("units"))
        {
            ((DefaultTimeSeriesHeader)_timeSeriesHeader).setUnit(value);
        }
        else if(elementName.equals("lat"))
        {
            try
            {
                this._readLat = Double.parseDouble(value);
                if(_readLat == 0.0D)
                {
                    _readLat = null;
                }
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Provided latitude is not a number.");
            }
        }
        else if(elementName.equals("lon"))
        {
            try
            {
                this._readLon = Double.parseDouble(value);
                if(_readLon == 0.0D)
                {
                    _readLon = null;
                }
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Provided latitude is not a number.");
            }
        }
        else if(elementName.equals("z"))
        {
            try
            {
                this._readZ = Double.parseDouble(value);
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Provided latitude is not a number.");
            }
        }
        else if(elementName.equals("creationDate"))
        {
            _creationDateStr = value;
        }
        else if(elementName.equals("creationTime"))
        {
            _creationTimeStr = value;
        }
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName,
                                                  final Attributes attr) throws XMLReaderException
    {
        if(elementName.equals(getXMLTagName()))
        {
            _timeSeriesHeader = null;
            _startTime = Long.MIN_VALUE;
            _endTime = Long.MIN_VALUE;
            _creationDateStr = "";
            _creationTimeStr = "";
            _creationTime = Long.MIN_VALUE;
        }
        else if(elementName.equals("header"))
        {
            _timeSeriesHeader = new DefaultTimeSeriesHeader();
        }
        else if(elementName.equals("timeStep"))
        {
            final String unitStr = attr.getValue("unit");
            final String multStr = attr.getValue("multiplier");
            final long millis = HCalendar.computeIntervalValueInMillis(multStr + " " + unitStr);
            ((DefaultTimeSeriesHeader)_timeSeriesHeader).setTimeStep(SimpleEquidistantTimeStep.getInstance(millis));
        }
        else if(elementName.equals("startDate"))
        {
            final String dateStr = attr.getValue("date");
            final String timeStr = attr.getValue("time");
            this._startTime = HCalendar.convertStringToCalendar(dateStr + " " + timeStr, HCalendar.DEFAULT_DATE_FORMAT)
                                       .getTimeInMillis();
        }
        else if(elementName.equals("endDate"))
        {
            final String dateStr = attr.getValue("date");
            final String timeStr = attr.getValue("time");
            this._endTime = HCalendar.convertStringToCalendar(dateStr + " " + timeStr, HCalendar.DEFAULT_DATE_FORMAT)
                                     .getTimeInMillis();
        }
        return null;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
        //Creation time.
        if(!_creationDateStr.isEmpty() && !_creationTimeStr.isEmpty())
        {
            final String fullString = _creationDateStr + " " + _creationTimeStr;
            final Calendar cal = HCalendar.processDate(fullString);
            if(cal == null)
            {
                throw new XMLReaderException("Combining creationDate and creationTime yields an invalid date/time string: "
                    + fullString + ".");
            }
            _creationTime = cal.getTimeInMillis();
        }

        //Geometry
        final Wgs1984Point point = new Wgs1984Point();
        if((_readLat != null) && (_readLon != null) && (_readZ != null))
        {
            point.setLatLongZ(_readLat, _readLon, _readZ);
            final PointsGeometry ptsGeo = new PointsGeometry(new GeoPoint[]{point});
            ((DefaultTimeSeriesHeader)this._timeSeriesHeader).setGeometry(ptsGeo);
        }
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public int compareTo(final Object o)
    {
        final TimeSeriesHeaderInfo other = (TimeSeriesHeaderInfo)o;
        final int compareResults = _timeSeriesHeader.getLocationId()
                                                    .compareTo(other.getTimeSeriesHeader().getLocationId());
        if(compareResults != 0)
        {
            return compareResults;
        }
        return _timeSeriesHeader.getParameterId().compareTo(other.getTimeSeriesHeader().getParameterId());
    }

    @Override
    public XMLReader getReader()
    {
        return this;
    }
}
