package ohd.hseb.hefs.utils.tsarrays.agg;

import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;

/**
 * This method is used by Deltares functionality. It somehow identifies the input and output from outside the class,
 * meaning that input and output attributes should not include a preceding '_'. To be consistent, no other attributes in
 * this class include a '_'.
 * 
 * @author hank.herr
 */
public class AccumulativeAggregator extends OHDAggregator
{
    public static final int AGGREGATE_TO_SUM = 1;
    public static final int AGGREGATE_TO_MEAN = 2;

    double _preAggregationValueMultiplier = 1.0D;

    public AccumulativeAggregator()
    {
        super();
    }

    public AccumulativeAggregator(final int type)
    {
        this.setAggregationType(type);
    }

    public AccumulativeAggregator(final int type, final double preAggregationValueMultiplier)
    {
        this.setAggregationType(type);
        this.setPreAggregationValueMultiplier(preAggregationValueMultiplier);
    }

    public void setPreAggregationValueMultiplier(final double mult)
    {
        _preAggregationValueMultiplier = mult;
    }

    @Override
    public float aggregate() throws TimeSeriesAggregationException
    {
        if(isAnyValueIsMissingPeriodNotCoveredOrNoInput())
        {
            if(ignoreMissingValues())
            {
                String message = "At least one missing data value was found and ignored.";
                if(getAggregationType() == AGGREGATE_TO_SUM)
                {
                    message += " For a sum aggregation, this means the computed sum may be invalid.";
                }
                addWarningMessage(message);
            }
            else
            {
                return Float.NaN;
            }
        }

        boolean valueAdded = false;
        float results = 0;
        if(getAggregationType() == AGGREGATE_TO_SUM)
        {
            for(int i = 0; i < getInputCount(); i++)
            {
                if(!TimeSeriesArrayTools.isOHDMissingValue(getInputValue(i)))
                {
                    results += (float)(getProportionsOfInputWindowInPeriod(i) * getInputValue(i) * _preAggregationValueMultiplier);
                    valueAdded = true;
                }
            }
        }
        else if(getAggregationType() == AGGREGATE_TO_MEAN)
        {
            for(int i = 0; i < getInputCount(); i++)
            {
                if(!TimeSeriesArrayTools.isOHDMissingValue(getInputValue(i)))
                {
                    results += (float)(getProportionsOfPeriodCoveredByInputWindow(i) * getInputValue(i) * _preAggregationValueMultiplier);
                    valueAdded = true;
                }
            }
        }
        else
        {
            throw new TimeSeriesAggregationException("Type index of " + getAggregationType()
                + " is not valid for an accumulative aggregation.");
        }
        if(!valueAdded)
        {
            return Float.NaN;
        }
        return results;
    }

    @Override
    public String getAggregationDisplayName()
    {
        if(getAggregationType() == AGGREGATE_TO_SUM)
        {
            return "sum";
        }
        else if(getAggregationType() == AGGREGATE_TO_MEAN)
        {
            return "mean";
        }
        return "UNSPECIFIED";
    }

//    @Override
//    public boolean canBeAccumulatedWithPrevioutValue()
//    {
//        return true;
//    }
//
//    @Override
//    public float accumulateWithPreviousValue(final float previousValue,
//                                             final long timeWindowForPreviousValue,
//                                             final int timeSeriesIndexOfPreviousValue,
//                                             final float currentValue,
//                                             final long timeWindowForCurrentValue) throws TimeSeriesAggregationException
//    {
//        if(getAggregationType() == AGGREGATE_TO_SUM)
//        {
//            return previousValue + currentValue;
//        }
//        else if(getAggregationType() == AGGREGATE_TO_MEAN)
//        {
//            final float proportionOfPrevious = timeWindowForPreviousValue
//                / (timeWindowForPreviousValue + timeWindowForCurrentValue);
//            final float proportionOfCurrent = 1 - proportionOfPrevious;
//            return proportionOfPrevious * previousValue + proportionOfCurrent * currentValue;
//        }
//        else
//        {
//            throw new TimeSeriesAggregationException("Type index of " + getAggregationType()
//                + " is not valid for an accumulative aggregation.");
//        }
//    }

}
