package ohd.hseb.hefs.utils.tsarrays.agg;

import java.util.Arrays;
import java.util.Calendar;
import java.util.List;

import nl.wldelft.util.Period;
import ohd.hseb.hefs.utils.xml.vars.XMLTimeStep;
import ohd.hseb.util.misc.HCalendar;

/**
 * Implementation of XMLTimeStep that uses special units of {@link #AGGEGATION_PERIOD_SPECIAL_UNITS}, which includes
 * {@link #AS_TIME_STEP_UNIT} and {@link #ACCUMULATED_UNIT}.
 * 
 * @author Hank.Herr
 */
public class AggregationPeriod extends XMLTimeStep
{
    public final static String AS_TIME_STEP_UNIT = "asTimeStep";
    public final static String ACCUMULATED_UNIT = "accumulated";
    public final static List<String> AGGEGATION_PERIOD_SPECIAL_UNITS = Arrays.asList(AS_TIME_STEP_UNIT,
                                                                                     ACCUMULATED_UNIT);

    public AggregationPeriod()
    {
        super(AGGEGATION_PERIOD_SPECIAL_UNITS);
        setUnit(AS_TIME_STEP_UNIT);
        setMultiplier(1);
    }

    public AggregationPeriod(final String xmlTag)
    {
        super(xmlTag, AGGEGATION_PERIOD_SPECIAL_UNITS);
        setUnit(AS_TIME_STEP_UNIT);
        setMultiplier(1);
    }

    public AggregationPeriod(final String tag, final String unit, final int multiplier)
    {
        super(tag, unit, multiplier, AGGEGATION_PERIOD_SPECIAL_UNITS);
    }

    /**
     * @return Returns the provided anchor, unless this aggregation is accumulated. If so, then the period anchor is
     *         forced to be ending; accumulated and centered makes no sense.
     */
    public PeriodAnchor determinedUsedAnchor(final PeriodAnchor anchor)
    {
        if(isAccumulated())
        {
            return new PeriodAnchor("unused", AnchorsEnum.ending);
        }
        return anchor;
    }

    /**
     * A yearly aggregation period uses the {@link Calendar#set(int, int)} method to add multiplies of
     * {@link Calendar#YEAR} to the computation time in order to determine the period. If the aggregation is centered,
     * then it will add and subtract six months from the computation time.<br>
     * <br>
     * A monthly aggregation is essentially the same as yearly but modifies the {@link Calendar#MONTH} field. However, a
     * centered period is not understood and results in an {@link IllegalArgumentException}. At this point in time, its
     * not clear to me how to center a month.
     * 
     * @param overallStartTime The overall aggregation start time, used when the aggregation unit is
     *            {@link #ACCUMULATED_UNIT}.
     * @param computationTime The time for which the aggregation period will be recorded.
     * @param computationTimeStep The computationTimeStep used for the aggregation computations; this is only needed if
     *            the {@link #getUnit()} is {@link #AS_TIME_STEP_UNIT}.
     * @param anchor The {@link PeriodAnchor} to apply.
     * @return A {@link Period} that specifies the aggregation period to use for the provided computationTimeStep. For
     *         accumulated aggregation, the period is always the overall start time to the computation time. For all
     *         other cases, it is computed based on the multiplier and unit. If the unit is {@link #AS_TIME_STEP_UNIT},
     *         the used period will be computed using the multiplier and unit of the computation time step provided
     *         instead of those within this object.
     */
    public Period computePeriod(final long overallStartTime,
                                final long computationTime,
                                final ComputationTimeStep computationTimeStep,
                                final PeriodAnchor anchor)
    {
        //Setup the used step.  
        //If this aggregation is as time step, then copy the settings from the computation time step.
        //Note that in such a case, accumulated aggregation if clause below is never triggered.
        XMLTimeStep usedPd = this;
        if(isAsTimeStep())
        {
            usedPd = computationTimeStep;
        }

        //Accumulated always starts at the overall start time and ends at the computation time.
        if(isAccumulated())
        {
            return new Period(overallStartTime, computationTime);
        }

        //A year is either 12 months before the comp time to the comp time, or it is 6-months on either side
        //of the comp time, dependingo the anchor.
        else if(usedPd.isUnitYear())
        {
            final Calendar computeTimeCal = HCalendar.computeCalendarFromMilliseconds(computationTime);
            final Calendar adjustedCal = HCalendar.computeCalendarFromMilliseconds(computationTime);
            if(anchor.isCentered())
            {
                adjustedCal.add(Calendar.MONTH, -6 * usedPd.getMultiplier());
                computeTimeCal.add(Calendar.MONTH, 6 * usedPd.getMultiplier());
            }
            else
            {
                adjustedCal.add(Calendar.YEAR, -1 * usedPd.getMultiplier());
            }
            return new Period(adjustedCal.getTimeInMillis(), computeTimeCal.getTimeInMillis());
        }

        //A month period is always computed as one month before the comp time.  I'm not sure how to handled
        //a centered month, so, for now, we don't allow it.
        else if(usedPd.isUnitMonth())
        {
            final Calendar computeTimeCal = HCalendar.computeCalendarFromMilliseconds(computationTime);
            final Calendar adjustedCal = HCalendar.computeCalendarFromMilliseconds(computationTime);
            if(anchor.isCentered())
            {
                throw new IllegalArgumentException("Currently it is not possble to do a month aggregation period "
                    + "that is centered on the computation time, since the width of the centered window may"
                    + " vary; try using a fixed aggregation period of '30 days'.");
            }
            else
            {
                adjustedCal.add(Calendar.MONTH, -1 * usedPd.getMultiplier());
            }
            return new Period(adjustedCal.getTimeInMillis(), computeTimeCal.getTimeInMillis());
        }
        //Straightforward in the case of regularly spaced intervals
        else
        {
            if(anchor.isCentered())
            {
                final long millis = usedPd.computeIntervalInMillis();
                return new Period(computationTime - (long)((double)millis / 2), computationTime
                    + (long)((double)millis / 2));
            }
            else
            {
                return new Period(computationTime - usedPd.computeIntervalInMillis(), computationTime);
            }
        }
    }

    /**
     * @return True if this period has the unit {@link #AS_TIME_STEP_UNIT}.
     */
    public boolean isAsTimeStep()
    {
        return getUnit().equals(AS_TIME_STEP_UNIT);
    }

    /**
     * @return True if this period has the unit {@link #ACCUMULATED_UNIT}.
     */
    public boolean isAccumulated()
    {
        return getUnit().equals(ACCUMULATED_UNIT);
    }
}
