package ohd.hseb.hefs.utils.tsarrays.agg;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import nl.wldelft.util.Period;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.ParameterType;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import nl.wldelft.util.timeseries.TimeStep;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.util.misc.HCalendar;


/**
 * A collection of aggregation methods that can be applied to {@link TimeSeriesArray} instances. These are wrappers to
 * calls to {@link TimeSeriesArrayAggregator} using implementations of {@link OHDAggregator}. It provides only the basic
 * aggregators: mean, sum, min, max, and instantaneous. It does not provide acces to the {@link NDCounterAggregator}.
 * 
 * @author hank.herr
 */
public abstract class AggregationTools
{
    private static final Logger LOG = LogManager.getLogger(AggregationTools.class);

    public static int INSTANTANEOUS_AGGREGATION = 0;
    public static int MEAN_AGGREGATION = 1;
    public static int SUM_AGGREGATION = 2;
    public static int MIN_AGGREGATION = 3;
    public static int MAX_AGGREGATION = 4;

    /**
     * @param timeStepStr Time step string (e.g., "6 hours", "period", "1 year", "2 months", etc.). See
     *            {@link HCalendar#computeIntervalValueInMillis(String)} for guidance.
     * @param baseTimeSeries Base time series for step computation, particularly if period, month, or year.
     * @return Step length in milliseconds.
     */
    public static long computeTimeStepLength(final String timeStepStr, final TimeSeriesArray baseTimeSeries)
    {
        if(timeStepStr.contains("period"))
        {
            return baseTimeSeries.getEndTime() - baseTimeSeries.getStartTime();
        }
        //fake value! currently not used. Need change if it will be used.
        else if(timeStepStr.contains("year"))
        {
            final long currTime = baseTimeSeries.getEndTime();
            final Calendar cal = HCalendar.computeCalendarFromMilliseconds(currTime);
            final Calendar prevCal = HCalendar.computeCalendarFromMilliseconds(currTime);
            prevCal.add(Calendar.YEAR, -1);
            return cal.getTimeInMillis() - prevCal.getTimeInMillis();
        }
        //fake value! currently not used. Need change if it will be used.
        else if(timeStepStr.contains("month"))
        {
            final Calendar cal = HCalendar.computeCalendarFromMilliseconds(baseTimeSeries.getEndTime());
            final long endOfMonth = HCalendar.computeLastOfMonth(cal).getTimeInMillis();
            final long startOfMonth = HCalendar.computeFirstOfMonth(cal).getTimeInMillis();
            return endOfMonth - startOfMonth;
        }
        else
        {
            return HCalendar.computeIntervalValueInMillis(timeStepStr);
        }
    }

    /**
     * Uses an {@link AccumulativeAggregator} with an aggregation type of
     * {@link AccumulativeAggregator#AGGREGATE_TO_MEAN}.
     * 
     * @param input TimeSeriesArray object specifying the time series to aggregate.
     * @param startDate Start date of aggregation, or the lower bound of the first time period for which values are
     *            computed. Null is not allowed!
     * @param endDate End date. If null, it will be pulled from the time series.
     * @param timeStepStr String identifying the time step between computed aggregated values. Null is not allowed!
     * @param aggPeriodStr String identifying the period of aggregation for the computation at each time step. Null is
     *            allowed and will result in the default {@link AggregationPeriod#AS_TIME_STEP_UNIT} being used.
     * @param periodAnchorStr String identifying the anchor position of the period relative to the computation time
     *            step; either {@link AnchorsEnum#ending} or {@link AnchorsEnum#centered}. Null is allowed and will
     *            result in {@link AnchorsEnum#ending} being used.
     * @return Aggregated TimeSeriesArray. The parameterId will be modified to include a suffix specifying the
     *         aggregation. Other header parameters may also be modified. Missing values are denoted by NaN.
     * @throws TimeSeriesAggregationException
     */
    public static TimeSeriesArray aggregateToMean(final TimeSeriesArray input,
                                                  final long startDate,
                                                  final Long endDate,
                                                  final String timeStepStr,
                                                  final String aggPeriodStr,
                                                  final String periodAnchorStr,
                                                  final boolean ignoreMissingValues,
                                                  final boolean prefixWithZero) throws TimeSeriesAggregationException
    {
        final OHDAggregator aggregator = new AccumulativeAggregator();
        aggregator.setIgnoreMissingValues(ignoreMissingValues);
        aggregator.setAggregationType(AccumulativeAggregator.AGGREGATE_TO_MEAN);
        final TimeSeriesArray ts = aggregate(input,
                                             startDate,
                                             endDate,
                                             timeStepStr,
                                             aggPeriodStr,
                                             periodAnchorStr,
                                             prefixWithZero,
                                             aggregator);
        TimeSeriesArrayTools.appendSuffixToParameterId(ts, "MEAN");
        ((DefaultTimeSeriesHeader)ts.getHeader()).setParameterType(ParameterType.MEAN); //this is a special case
        return ts;
    }

    /**
     * Uses an {@link AccumulativeAggregator} with an aggregation type of
     * {@link AccumulativeAggregator#AGGREGATE_TO_SUM}.
     * 
     * @param input TimeSeriesArray object specifying the time series to aggregate.
     * @param startDate Start date of aggregation, or the lower bound of the first time period for which values are
     *            computed. Null is not allowed!
     * @param endDate End date. If null, it will be pulled from the time series.
     * @param timeStepStr String identifying the time step between computed aggregated values. Null is not allowed!
     * @param aggPeriodStr String identifying the period of aggregation for the computation at each time step. Null is
     *            allowed and will result in the default {@link AggregationPeriod#AS_TIME_STEP_UNIT} being used.
     * @param periodAnchorStr String identifying the anchor position of the period relative to the computation time
     *            step; either {@link AnchorsEnum#ending} or {@link AnchorsEnum#centered}. Null is allowed and will
     *            result in {@link AnchorsEnum#ending} being used.
     * @param preAggregationValueMultiplier A value to multiply all time series values by before aggregating. Useful for
     *            computing volumes where a time factor is used.
     * @return Aggregated TimeSeriesArray. The parameterId will be modified to include a suffix specifying the
     *         aggregation. Other header parameters may also be modified. Missing values are denoted by NaN.
     * @throws TimeSeriesAggregationException
     */
    public static TimeSeriesArray aggregateToSum(final TimeSeriesArray input,
                                                 final long startDate,
                                                 final Long endDate,
                                                 final String timeStepStr,
                                                 final String aggPeriodStr,
                                                 final String periodAnchorStr,
                                                 final boolean ignoreMissingValues,
                                                 final boolean prefixWithZero,
                                                 final double preAggregationValueMultiplier) throws TimeSeriesAggregationException
    {
        final AccumulativeAggregator aggregator = new AccumulativeAggregator();
        aggregator.setPreAggregationValueMultiplier(preAggregationValueMultiplier);
        aggregator.setAggregationType(AccumulativeAggregator.AGGREGATE_TO_SUM);
        aggregator.setIgnoreMissingValues(ignoreMissingValues);
        final TimeSeriesArray ts = aggregate(input,
                                             startDate,
                                             endDate,
                                             timeStepStr,
                                             aggPeriodStr,
                                             periodAnchorStr,
                                             prefixWithZero,
                                             aggregator);
        TimeSeriesArrayTools.appendSuffixToParameterId(ts, "SUM");

        return ts;
    }

    /**
     * Calls {@link #aggregateToSum(TimeSeriesArray, long, Long, String, String, String, double)} passing in 1.0D for
     * the last argument.
     */
    public static TimeSeriesArray aggregateToSum(final TimeSeriesArray input,
                                                 final long startDate,
                                                 final Long endDate,
                                                 final String timeStepStr,
                                                 final String aggPeriodStr,
                                                 final String periodAnchorStr,
                                                 final boolean ignoreMissingValues,
                                                 final boolean prefixWithZero) throws TimeSeriesAggregationException
    {
        return aggregateToSum(input,
                              startDate,
                              endDate,
                              timeStepStr,
                              aggPeriodStr,
                              periodAnchorStr,
                              ignoreMissingValues,
                              prefixWithZero,
                              1.0D);
    }

    /**
     * Uses an {@link InstantaneousAggregator}.
     * 
     * @param input TimeSeriesArray object specifying the time series to aggregate.
     * @param startDate Start date of aggregation, or the lower bound of the first time period for which values are
     *            computed. Null is not allowed!
     * @param endDate End date. If null, it will be pulled from the time series.
     * @param timeStepStr String identifying the time step between computed aggregated values. Null is not allowed!
     * @param aggPeriodStr String identifying the period of aggregation for the computation at each time step. Null is
     *            allowed and will result in the default {@link AggregationPeriod#AS_TIME_STEP_UNIT} being used.
     * @param periodAnchorStr String identifying the anchor position of the period relative to the computation time
     *            step; either {@link AnchorsEnum#ending} or {@link AnchorsEnum#centered}. Null is allowed and will
     *            result in {@link AnchorsEnum#ending} being used.
     * @return Aggregated TimeSeriesArray. The parameterId will be modified to include a suffix specifying the
     *         aggregation. Other header parameters may also be modified. Missing values are denoted by NaN.
     * @throws TimeSeriesAggregationException
     */
    public static TimeSeriesArray aggregateToInstantaneous(final TimeSeriesArray input,
                                                           final long startDate,
                                                           final Long endDate,
                                                           final String timeStepStr,
                                                           final String aggPeriodStr,
                                                           final String periodAnchorStr,
                                                           final boolean prefixWithZero) throws TimeSeriesAggregationException
    {
        final InstantaneousAggregator aggregator = new InstantaneousAggregator();
        final TimeSeriesArray ts = aggregate(input,
                                             startDate,
                                             endDate,
                                             timeStepStr,
                                             aggPeriodStr,
                                             periodAnchorStr,
                                             prefixWithZero,
                                             aggregator);
        TimeSeriesArrayTools.appendSuffixToParameterId(ts, "INSTANTANEOUS");
        ((DefaultTimeSeriesHeader)ts.getHeader()).setParameterType(ParameterType.INSTANTANEOUS);

        return ts;
    }

    /**
     * Calls {@link #aggregateToMaximum(TimeSeriesArray, String)} for every time series in input.
     */
    public static TimeSeriesArrays aggregateToMinimum(final TimeSeriesArrays input,
                                                      final long startTime,
                                                      final Long endTime,
                                                      final String timeStepStr,
                                                      final String aggPeriodStr,
                                                      final String periodAnchorStr,
                                                      final boolean ignoreMissingValues,
                                                      final boolean prefixWithZero) throws TimeSeriesAggregationException
    {
        final TimeSeriesArrays results = new TimeSeriesArrays(input.getHeaderClass(), input.size());
        for(int i = 0; i < input.size(); i++)
        {
            results.add(aggregateToMinimum(input.get(i),
                                           startTime,
                                           endTime,
                                           timeStepStr,
                                           aggPeriodStr,
                                           periodAnchorStr,
                                           ignoreMissingValues,
                                           prefixWithZero));
        }
        return results;
    }

    /**
     * Calls {@link #aggregateToMinimum(TimeSeriesArray, String)} for every time series in input.
     */
    public static TimeSeriesArrays aggregateToMinimum(final TimeSeriesArrays input,
                                                      final String timeStepStr,
                                                      final String aggPeriodStr,
                                                      final String periodAnchorStr,
                                                      final boolean ignoreMissingValues,
                                                      final boolean prefixWithZero) throws TimeSeriesAggregationException
    {
        final TimeSeriesArrays results = new TimeSeriesArrays(input.getHeaderClass(), input.size());
        for(int i = 0; i < input.size(); i++)
        {
            results.add(aggregateToMinimum(input.get(i),
                                           timeStepStr,
                                           aggPeriodStr,
                                           periodAnchorStr,
                                           ignoreMissingValues,
                                           prefixWithZero));
        }
        return results;
    }

    /**
     * Calls {@link #aggregateToMinimum(TimeSeriesArray, long, Long, String)} passing in the forecast time of the time
     * series as the start date and null for the end date.
     * 
     * @param input TimeSeriesArray object specifying the time series to aggregate.
     * @param timeStepStr String identifying the time step between computed aggregated values. Null is not allowed!
     * @param aggPeriodStr String identifying the period of aggregation for the computation at each time step. Null is
     *            allowed and will result in the default {@link AggregationPeriod#AS_TIME_STEP_UNIT} being used.
     * @param periodAnchorStr String identifying the anchor position of the period relative to the computation time
     *            step; either {@link AnchorsEnum#ending} or {@link AnchorsEnum#centered}. Null is allowed and will
     *            result in {@link AnchorsEnum#ending} being used.
     * @return Aggregated time series.
     * @throws TimeSeriesAggregationException
     */
    public static TimeSeriesArray aggregateToMinimum(final TimeSeriesArray input,
                                                     final String timeStepStr,
                                                     final String aggPeriodStr,
                                                     final String periodAnchorStr,
                                                     final boolean ignoreMissingValues,
                                                     final boolean prefixWithZero) throws TimeSeriesAggregationException
    {
        return aggregateToMinimum(input,
                                  input.getHeader().getForecastTime(),
                                  null,
                                  timeStepStr,
                                  aggPeriodStr,
                                  periodAnchorStr,
                                  ignoreMissingValues,
                                  prefixWithZero);
    }

    /**
     * Uses an {@link ExtremeAggregator} with type {@link ExtremeAggregator#MINIMUM_TYPE}.
     * 
     * @param input TimeSeriesArray object specifying the time series to aggregate.
     * @param startDate Start date of aggregation, or the lower bound of the first time period for which values are
     *            computed. Null is not allowed!
     * @param endDate End date. If null, it will be pulled from the time series.
     * @param timeStepStr String identifying the time step between computed aggregated values. Null is not allowed!
     * @param aggPeriodStr String identifying the period of aggregation for the computation at each time step. Null is
     *            allowed and will result in the default {@link AggregationPeriod#AS_TIME_STEP_UNIT} being used.
     * @param periodAnchorStr String identifying the anchor position of the period relative to the computation time
     *            step; either {@link AnchorsEnum#ending} or {@link AnchorsEnum#centered}. Null is allowed and will
     *            result in {@link AnchorsEnum#ending} being used.
     * @return Aggregated TimeSeriesArray. The parameterId will be modified to include a suffix specifying the
     *         aggregation. Other header parameters may also be modified. Missing values are denoted by NaN.
     * @throws TimeSeriesAggregationException
     */
    public static TimeSeriesArray aggregateToMinimum(final TimeSeriesArray input,
                                                     final long startDate,
                                                     final Long endDate,
                                                     final String timeStepStr,
                                                     final String aggPeriodStr,
                                                     final String periodAnchorStr,
                                                     final boolean ignoreMissingValues,
                                                     final boolean prefixWithZero) throws TimeSeriesAggregationException
    {
        final ExtremeAggregator aggregator = new ExtremeAggregator();
        aggregator.setAggregationType(ExtremeAggregator.MINIMUM_TYPE);
        aggregator.setIgnoreMissingValues(ignoreMissingValues);
        final TimeSeriesArray ts = aggregate(input,
                                             startDate,
                                             endDate,
                                             timeStepStr,
                                             aggPeriodStr,
                                             periodAnchorStr,
                                             prefixWithZero,
                                             aggregator);
        TimeSeriesArrayTools.appendSuffixToParameterId(ts, "MIN");

        return ts;
    }

    /**
     * Calls {@link #aggregateToMaximum(TimeSeriesArray, String)} for every time series in input.
     */
    public static TimeSeriesArrays aggregateToMaximum(final TimeSeriesArrays input,
                                                      final String timeStepStr,
                                                      final String aggPeriodStr,
                                                      final String periodAnchorStr,
                                                      final boolean ignoreMissingValues,
                                                      final boolean prefixWithZero) throws TimeSeriesAggregationException
    {
        final TimeSeriesArrays results = new TimeSeriesArrays(input.getHeaderClass(), input.size());
        for(int i = 0; i < input.size(); i++)
        {
            results.add(aggregateToMaximum(input.get(i),
                                           timeStepStr,
                                           aggPeriodStr,
                                           periodAnchorStr,
                                           ignoreMissingValues,
                                           prefixWithZero));
        }
        return results;
    }

    /**
     * Calls {@link #aggregateToMaximum(TimeSeriesArray, String)} for every time series in input.
     */
    public static TimeSeriesArrays aggregateToMaximum(final TimeSeriesArrays input,
                                                      final long startTime,
                                                      final Long endTime,
                                                      final String timeStepStr,
                                                      final String aggPeriodStr,
                                                      final String periodAnchorStr,
                                                      final boolean ignoreMissingValues,
                                                      final boolean prefixWithZero) throws TimeSeriesAggregationException
    {
        final TimeSeriesArrays results = new TimeSeriesArrays(input.getHeaderClass(), input.size());
        for(int i = 0; i < input.size(); i++)
        {
            results.add(aggregateToMaximum(input.get(i),
                                           startTime,
                                           endTime,
                                           timeStepStr,
                                           aggPeriodStr,
                                           periodAnchorStr,
                                           ignoreMissingValues,
                                           prefixWithZero));
        }
        return results;
    }

    /**
     * Calls {@link #aggregateToMaximum(TimeSeriesArray, long, Long, String)} passing in the forecast time of the time
     * series as the start date and null for the end date.
     * 
     * @param input TimeSeriesArray object specifying the time series to aggregate.
     * @param timeStepStr String identifying the time step between computed aggregated values. Null is not allowed!
     * @param aggPeriodStr String identifying the period of aggregation for the computation at each time step. Null is
     *            allowed and will result in the default {@link AggregationPeriod#AS_TIME_STEP_UNIT} being used.
     * @param periodAnchorStr String identifying the anchor position of the period relative to the computation time
     *            step; either {@link AnchorsEnum#ending} or {@link AnchorsEnum#centered}. Null is allowed and will
     *            result in {@link AnchorsEnum#ending} being used.
     * @return Aggregated time series.
     * @throws TimeSeriesAggregationException
     */
    public static TimeSeriesArray aggregateToMaximum(final TimeSeriesArray input,
                                                     final String timeStepStr,
                                                     final String aggPeriodStr,
                                                     final String periodAnchorStr,
                                                     final boolean ignoreMissingValues,
                                                     final boolean prefixWithZero) throws TimeSeriesAggregationException
    {
        return aggregateToMaximum(input,
                                  input.getHeader().getForecastTime(),
                                  null,
                                  timeStepStr,
                                  aggPeriodStr,
                                  periodAnchorStr,
                                  ignoreMissingValues,
                                  prefixWithZero);
    }

    /**
     * Uses an {@link ExtremeAggregator} with type {@link ExtremeAggregator#MAXIMUM_TYPE}.
     * 
     * @param input TimeSeriesArray object specifying the time series to aggregate.
     * @param startDate Start date of aggregation, or the lower bound of the first time period for which values are
     *            computed. Null is not allowed!
     * @param endDate End date. If null, it will be pulled from the time series.
     * @param timeStepStr String identifying the time step between computed aggregated values. Null is not allowed!
     * @param aggPeriodStr String identifying the period of aggregation for the computation at each time step. Null is
     *            allowed and will result in the default {@link AggregationPeriod#AS_TIME_STEP_UNIT} being used.
     * @param periodAnchorStr String identifying the anchor position of the period relative to the computation time
     *            step; either {@link AnchorsEnum#ending} or {@link AnchorsEnum#centered}. Null is allowed and will
     *            result in {@link AnchorsEnum#ending} being used.
     * @return Aggregated TimeSeriesArray. The parameterId will be modified to include a suffix specifying the
     *         aggregation. Other header parameters may also be modified. Missing values are denoted by NaN.
     * @throws TimeSeriesAggregationException
     */
    public static TimeSeriesArray aggregateToMaximum(final TimeSeriesArray input,
                                                     final long startDate,
                                                     final Long endDate,
                                                     final String timeStepStr,
                                                     final String aggPeriodStr,
                                                     final String periodAnchorStr,
                                                     final boolean ignoreMissingValues,
                                                     final boolean prefixWithZero) throws TimeSeriesAggregationException
    {
        final ExtremeAggregator aggregator = new ExtremeAggregator();
        aggregator.setAggregationType(ExtremeAggregator.MAXIMUM_TYPE);
        aggregator.setIgnoreMissingValues(ignoreMissingValues);
        final TimeSeriesArray ts = aggregate(input,
                                             startDate,
                                             endDate,
                                             timeStepStr,
                                             aggPeriodStr,
                                             periodAnchorStr,
                                             prefixWithZero,
                                             aggregator);
        TimeSeriesArrayTools.appendSuffixToParameterId(ts, "MAX");

        return ts;
    }

    /**
     * The full version of the single time series aggregate method.
     * 
     * @param input The time series to aggregate.
     * @param startDate The start time of the aggregation.
     * @param endDate The end time.
     * @param computationTimeStepStr String specifying the computation time step. See {@link ComputationTimeStep}.
     * @param aggregationPeriodStr String specifying the period of aggregation to compute for each computation time. See
     *            {@link AggregationPeriod}.
     * @param periodAnchorStr String specifying the anchor position of the aggregation. See {@link PeriodAnchor}.
     * @param aggregator {@link OHDAggregator} specifying the computation.
     * @return An aggregated {@link TimeSeriesArray}, but without the time step and type specified in the header.
     * @throws TimeSeriesAggregationException
     */
    public static TimeSeriesArray aggregate(final TimeSeriesArray input,
                                            final long startDate,
                                            Long endDate,
                                            final String computationTimeStepStr,
                                            final String aggregationPeriodStr,
                                            final String periodAnchorStr,
                                            final boolean prefixWithZero,
                                            final OHDAggregator aggregator) throws TimeSeriesAggregationException
    {
        if(endDate == null)
        {
            endDate = input.getEndTime();
        }

        try
        {
            final ComputationTimeStep compStep = new ComputationTimeStep();
            compStep.parseTimeStepStr(computationTimeStepStr);

            //Default is set in the constructor.
            final AggregationPeriod aggPd = new AggregationPeriod();
            if(aggregationPeriodStr != null)
            {
                aggPd.parseTimeStepStr(aggregationPeriodStr);
            }

            //Default is set in the constructor.
            final PeriodAnchor anchor = new PeriodAnchor();
            if(periodAnchorStr != null)
            {
                anchor.set(AnchorsEnum.valueOf(periodAnchorStr));
            }

            //Do the aggregation.
            final TimeSeriesArrayAggregator tsAgg = new TimeSeriesArrayAggregator();
            final TimeSeriesArray output = tsAgg.calculateUserDefinedAggregation(input,
                                                                                 compStep,
                                                                                 new Period(startDate, endDate),
                                                                                 aggPd,
                                                                                 anchor,
                                                                                 prefixWithZero,
                                                                                 aggregator);
            processWarningMessages(input, aggregator, tsAgg.getWarningMessages());

            return output;
        }
        catch(final Exception e)
        {
//            e.printStackTrace();
            final TimeSeriesAggregationException exc = new TimeSeriesAggregationException(e.getMessage());
            exc.setStackTrace(e.getStackTrace());
            throw exc;
        }
    }

    /**
     * Posts every warning message to the {@link #LOG}, but as debug messages. Otherwise, entirely too many messages
     * will likely be posted.
     * 
     * @param warningMessages Messages to post.
     */
    private static void processWarningMessages(final TimeSeriesArray ts,
                                               final OHDAggregator agg,
                                               final Collection<String> warningMessages)
    {
        if(!warningMessages.isEmpty())
        {
            for(final String message: warningMessages)
            {
                LOG.debug("The following occured while performing an " + agg.getAggregationDisplayName()
                    + "aggregation on time series " + TimeSeriesArrayTools.createHeaderString(ts) + ": " + message);
            }
        }
    }

    /**
     * As {@link #aggregate(TimeSeriesArray, long, Long, String, String, String, OHDAggregator)}, but calls it for each
     * time series in the provided input, which is a {@link TimeSeriesArrays} instance.
     */
    public static TimeSeriesArrays aggregate(final TimeSeriesArrays input,
                                             final long startDate,
                                             final Long endDate,
                                             final String computationTimeStepStr,
                                             final String aggregationPeriodStr,
                                             final String periodAnchorStr,
                                             final boolean prefixWithZero,
                                             final OHDAggregator aggregator) throws TimeSeriesAggregationException
    {
        final List<TimeSeriesArray> resultList = new ArrayList<TimeSeriesArray>();
        for(int i = 0; i < input.size(); i++)
        {
            resultList.add(aggregate(input.get(i),
                                     startDate,
                                     endDate,
                                     computationTimeStepStr,
                                     aggregationPeriodStr,
                                     periodAnchorStr,
                                     prefixWithZero,
                                     aggregator));
        }
        return TimeSeriesArraysTools.convertListOfTimeSeriesToTimeSeriesArrays(resultList);
    }

    /**
     * Calls {@link #aggregate(TimeSeriesArray, long, Long, String, String, String, OHDAggregator)} with the appropriate
     * {@link OHDAggregator} based on the type specified.
     * 
     * @param type An integer indicating the type of aggregation: {@link #INSTANTANEOUS_AGGREGATION},
     *            {@link #MAX_AGGREGATION}, {@link #MIN_AGGREGATION}, {@link #MEAN_AGGREGATION},
     *            {@link #SUM_AGGREGATION}.
     */
    public static TimeSeriesArray aggregate(final TimeSeriesArray input,
                                            final long startDate,
                                            final Long endDate,
                                            final String timeStepStr,
                                            final String aggPeriodStr,
                                            final String periodAnchorStr,
                                            final boolean ignoreMissingValues,
                                            final boolean prefixWithZero,
                                            final int type) throws TimeSeriesAggregationException
    {
        if(type == INSTANTANEOUS_AGGREGATION)
        {
            return aggregateToInstantaneous(input,
                                            startDate,
                                            endDate,
                                            timeStepStr,
                                            aggPeriodStr,
                                            periodAnchorStr,
                                            prefixWithZero);
        }
        else if(type == MEAN_AGGREGATION)
        {
            return aggregateToMean(input,
                                   startDate,
                                   endDate,
                                   timeStepStr,
                                   aggPeriodStr,
                                   periodAnchorStr,
                                   ignoreMissingValues,
                                   prefixWithZero);
        }
        else if(type == SUM_AGGREGATION)
        {
            return aggregateToSum(input,
                                  startDate,
                                  endDate,
                                  timeStepStr,
                                  aggPeriodStr,
                                  periodAnchorStr,
                                  ignoreMissingValues,
                                  prefixWithZero);
        }
        else if(type == MAX_AGGREGATION)
        {
            return aggregateToMaximum(input,
                                      startDate,
                                      endDate,
                                      timeStepStr,
                                      aggPeriodStr,
                                      periodAnchorStr,
                                      ignoreMissingValues,
                                      prefixWithZero);
        }
        else if(type == MIN_AGGREGATION)
        {
            return aggregateToMinimum(input,
                                      startDate,
                                      endDate,
                                      timeStepStr,
                                      aggPeriodStr,
                                      periodAnchorStr,
                                      ignoreMissingValues,
                                      prefixWithZero);
        }
        throw new TimeSeriesAggregationException("Aggregation type int is invalid; was " + type
            + " but must be from 0 - 4.");
    }

    /**
     * Partial argument version calls
     * {@link #aggregate(TimeSeriesArray, long, Long, String, String, String, boolean, boolean, int, float)} that
     * assumes the ignore missing and prefix flags are false.
     */
    public static TimeSeriesArray aggregate(final TimeSeriesArray input,
                                            final long startDate,
                                            final Long endDate,
                                            final String timeStepStr,
                                            final String aggPeriodStr,
                                            final String periodAnchorStr,
                                            final int type,
                                            final float missingValue) throws TimeSeriesAggregationException
    {
        return aggregate(input,
                         startDate,
                         endDate,
                         timeStepStr,
                         aggPeriodStr,
                         periodAnchorStr,
                         false,
                         false,
                         type,
                         missingValue);
    }

    /**
     * As {@link #aggregate(TimeSeriesArray, long, Long, String, String, String, int)}, but with the value to use for
     * missing values specified.
     * 
     * @param missingValue The value to use for all missing values in the computed time series.
     */
    public static TimeSeriesArray aggregate(final TimeSeriesArray input,
                                            final long startDate,
                                            final Long endDate,
                                            final String timeStepStr,
                                            final String aggPeriodStr,
                                            final String periodAnchorStr,
                                            final boolean ignoreMissingValues,
                                            final boolean prefixWithZero,
                                            final int type,
                                            final float missingValue) throws TimeSeriesAggregationException
    {
        final TimeSeriesArray results = aggregate(input,
                                                  startDate,
                                                  endDate,
                                                  timeStepStr,
                                                  aggPeriodStr,
                                                  periodAnchorStr,
                                                  ignoreMissingValues,
                                                  prefixWithZero,
                                                  type);
        TimeSeriesArrayTools.replaceAllMissingValuesWithValue(results, missingValue);
        return results;
    }

    /**
     * As {@link #aggregate(TimeSeriesArray, long, Long, String, String, String, int)}, but calls it for each time
     * series in the provided input, which is a {@link TimeSeriesArrays} instance.
     */
    public static TimeSeriesArrays aggregate(final TimeSeriesArrays input,
                                             final long startDate,
                                             final Long endDate,
                                             final String timeStepStr,
                                             final String aggPeriodStr,
                                             final String periodAnchorStr,
                                             final boolean ignoreMissingValues,
                                             final boolean prefixWithZero,
                                             final int type) throws TimeSeriesAggregationException
    {
        final List<TimeSeriesArray> resultList = new ArrayList<TimeSeriesArray>();
        for(int i = 0; i < input.size(); i++)
        {
            resultList.add(aggregate(input.get(i),
                                     startDate,
                                     endDate,
                                     timeStepStr,
                                     aggPeriodStr,
                                     periodAnchorStr,
                                     ignoreMissingValues,
                                     prefixWithZero,
                                     type));
        }
        return TimeSeriesArraysTools.convertListOfTimeSeriesToTimeSeriesArrays(resultList);
    }

    /**
     * Partial version calls
     * {@link #aggregate(TimeSeriesArrays, long, Long, String, String, String, boolean, boolean, int, float)} passing in
     * false for the ignore missing and prefix with zero flags.
     */
    public static TimeSeriesArrays aggregate(final TimeSeriesArrays input,
                                             final long startDate,
                                             final Long endDate,
                                             final String timeStepStr,
                                             final String aggPeriodStr,
                                             final String periodAnchorStr,
                                             final int type,
                                             final float missingValue) throws TimeSeriesAggregationException
    {
        return aggregate(input,
                         startDate,
                         endDate,
                         timeStepStr,
                         aggPeriodStr,
                         periodAnchorStr,
                         false,
                         false,
                         type,
                         missingValue);
    }

    /**
     * As {@link #aggregate(TimeSeriesArrays, long, Long, String, String, String, int)}, but with the value to use for
     * missing values specified.
     * 
     * @param missingValue The value to use for all missing values in the computed time series.
     */
    public static TimeSeriesArrays aggregate(final TimeSeriesArrays input,
                                             final long startDate,
                                             final Long endDate,
                                             final String timeStepStr,
                                             final String aggPeriodStr,
                                             final String periodAnchorStr,
                                             final boolean ignoreMissingValues,
                                             final boolean prefixWithZero,
                                             final int type,
                                             final float missingValue) throws TimeSeriesAggregationException
    {
        final TimeSeriesArrays results = aggregate(input,
                                                   startDate,
                                                   endDate,
                                                   timeStepStr,
                                                   aggPeriodStr,
                                                   periodAnchorStr,
                                                   ignoreMissingValues,
                                                   prefixWithZero,
                                                   type);
        TimeSeriesArraysTools.replaceAllMissingValuesWithValue(results, missingValue);
        return results;
    }

    /**
     * Compute a single aggregated value via the
     * {@link #aggregate(TimeSeriesArray, long, Long, String, String, String, int)} method. It assumes the ignore
     * missing and prefix with zero flags are false.
     * 
     * @return The single aggregated value computed over startTime to endTime within ts using the specified aggregation
     *         type. The computation time step is period, which forces the aggregation and anchors to default values.
     * @throws TimeSeriesAggregationException
     */
    public static float computeAggregatedValue(final TimeSeriesArray ts,
                                               final long startTime,
                                               final long endTime,
                                               final int type) throws TimeSeriesAggregationException
    {
        final TimeSeriesArray outputTS = aggregate(ts, startTime, endTime, "period", null, null, false, false, type);
        if(outputTS.size() != 1)
        {
            throw new TimeSeriesAggregationException("Expected one number in agg time series, but got "
                + outputTS.size() + ".");
        }
        return outputTS.getValue(0);
    }

    /**
     * This method assumes a mean aggregation was used to compute an aggregated time series arguments.
     * 
     * @param tsToBeDisaggregated Time series in the same step as relativeRawAggregatedTimeSeries.
     * @param relativeRawTimeSeries The original time series.
     * @param relativeRawAggregatedTimeSeries The mean aggregated time series based on relativeRawTimeSeries.
     * @return Time series, each of which is constructed by disaggregating the corresponding time series in
     *         tsToBeDisaggregated based on the relationship between relativeRawTimeSeries and
     *         relativeRawAggregatedTimeSeries.
     * @throws TimeSeriesAggregationException
     */
    public static TimeSeriesArrays disaggregateTimeSeriesRelativeToAnother(final TimeSeriesArrays tsToBeDisaggregated,
                                                                           final TimeSeriesArrays relativeRawTimeSeries,
                                                                           final TimeSeriesArrays relativeRawAggregatedTimeSeries) throws TimeSeriesAggregationException
    {
        final TimeStep inputTimeStep = tsToBeDisaggregated.get(0).getHeader().getTimeStep();
        final TimeStep targetTimeStep = relativeRawTimeSeries.get(0).getHeader().getTimeStep();

        if(targetTimeStep.getStepMillis() < inputTimeStep.getStepMillis())
        {

            final TimeSeriesArrays results = new TimeSeriesArrays(tsToBeDisaggregated.getHeaderClass(),
                                                                  tsToBeDisaggregated.size());

            if(tsToBeDisaggregated.isEmpty())
            {
                return results;
            }

            for(int i = 0; i < tsToBeDisaggregated.size(); i++)
            {
                final TimeSeriesArray newSeries = disaggregateTimeSeriesRelativeToAnother(tsToBeDisaggregated.get(i),
                                                                                          relativeRawTimeSeries.get(i),
                                                                                          relativeRawAggregatedTimeSeries.get(i));

                results.add(newSeries);
            }
            return results;
        }
        else
        {
            return tsToBeDisaggregated;
        }
    }

    /**
     * This method assumes a mean aggregation was used to compute an aggregated time series arguments. It does a
     * simplified 4-point disaggregation; the 5-pt version does not work right (5-pt code is commented out).
     * 
     * @param tsToBeDisaggregated A time series in the same step as relativeRawAggregatedTimeSeries.
     * @param relativeRawTimeSeries The original time series.
     * @param relativeRawAggregatedTimeSeries The mean aggregated time series based on relativeRawTimeSeries.
     * @return A time series that was constructed by disaggregating tsToBeDisaggregated based on the relationship
     *         between relativeRawTimeSeries and relativeRawAggregatedTimeSeries.
     * @throws TimeSeriesAggregationException
     */
    public static TimeSeriesArray disaggregateTimeSeriesRelativeToAnother(final TimeSeriesArray tsToBeDisaggregated,
                                                                          final TimeSeriesArray relativeRawTimeSeries,
                                                                          final TimeSeriesArray relativeRawAggregatedTimeSeries) throws TimeSeriesAggregationException
    {   // change the method clone() by duplicate()
        final TimeSeriesArray newTraces = relativeRawTimeSeries.duplicate();

//        System.err.println(tsToBeDisaggregated.getHeader().getEnsembleId());
        final int toScale = (int)((double)relativeRawTimeSeries.getHeader().getTimeStep().getStepMillis() / (double)HCalendar.MILLIS_IN_HR);
        long startDate, endDate;
        float origDailyAvg = Float.NaN;
        float adjDailyAvg = Float.NaN;
//        float prevOrigAvg = Float.NaN;
//        float prevAdjAvg = Float.NaN;
        float working = Float.NaN;

        startDate = relativeRawTimeSeries.getStartTime();
        endDate = relativeRawTimeSeries.getEndTime();

        //The starting point is one step before the start time of tsToBeDisaggregated
        long currentAggregatedDate = tsToBeDisaggregated.getStartTime()
            - tsToBeDisaggregated.getHeader().getTimeStep().getStepMillis();

        for(long currentDate = startDate; currentDate <= endDate; currentDate += (HCalendar.MILLIS_IN_HR * toScale))
        {
            if(((currentDate > currentAggregatedDate) || (currentDate == startDate)) && (currentDate != endDate))
            {
//                prevOrigAvg = origDailyAvg;
//                prevAdjAvg = adjDailyAvg;

                currentAggregatedDate += tsToBeDisaggregated.getHeader().getTimeStep().getStepMillis();

                adjDailyAvg = TimeSeriesArrayTools.getValueByTime(tsToBeDisaggregated, currentAggregatedDate);
                origDailyAvg = TimeSeriesArrayTools.getValueByTime(relativeRawAggregatedTimeSeries,
                                                                   currentAggregatedDate);

            }

            if(TimeSeriesArrayTools.isOHDMissingValue(adjDailyAvg)
                || TimeSeriesArrayTools.isOHDMissingValue(origDailyAvg))
            {
                working = Float.NaN;
            }
            else
            {
                working = TimeSeriesArrayTools.getValueByTime(relativeRawTimeSeries, currentDate)
                    * (adjDailyAvg / origDailyAvg);
//                working = TimeSeriesArrayTools.getValueByTime(relativeRawTimeSeries, currentDate);
//                if(!Float.isNaN(prevOrigAvg) && !Float.isNaN(prevAdjAvg)) // right on the 24 hour marker
//                {
//                    if origdailyavg == 0, set origdailyavg == prevorigavg == working.   
//                    working = (float)(working * (0.5 * (adjDailyAvg / origDailyAvg) + 0.5 * (prevAdjAvg / prevOrigAvg)));
//                }
//                else
//                within the aggregation period
//                {
//                if origdailyavg is zero, set working = adjdailyavg else do what it's doing now.
//                     working = working * (adjDailyAvg / origDailyAvg);
//                }
            }

            if(!TimeSeriesArrayTools.replaceValueByTime(newTraces, currentDate, working))
            {
                throw new TimeSeriesAggregationException("In disaggragated traces, trying to set the value for time "
                    + new Date(currentDate) + " with " + working + ", but the time was not found.");
            }

//            System.out.println ( "adjDailyAvg/OrigDailyAvg prevAdj/PrevOrig working: " + adjDailyAvg + "/" + origDailyAvg + " " + prevAdjAvg+"/"+prevOrigAvg + "   " + working );

//            prevOrigAvg = Float.NaN;
//            prevAdjAvg = Float.NaN;
        }
        return newTraces;
    }

    /**
     * Disaggregates the aggregated time series assuming a straight accumulative disaggregation is needed. That means
     * that the aggregatedTS is considered to a sum of the disaggregated values and that the aggregated values have a
     * time that is the end of its associated time period, as an accumulative time series would..
     * 
     * @param aggregatedTS The time series to disaggregated.
     * @param forecastTime The forecast time of the disaggregated time series. Should be the same as the aggregatedTS
     *            forecast time but it doesn't have to be.
     * @param stepInMillis The target step in millis. For proper results, this time step should divide the aggregatedTS
     *            time step evenly.
     * @return A disaggregated time series with the same header as the aggregated time series but a shorter time step.
     * @throws TimeSeriesAggregationException
     */
    public static TimeSeriesArray disaggregateAccumulativeTimeSeries(final TimeSeriesArray aggregatedTS,
                                                                     final long forecastTime,
                                                                     final long stepInMillis) throws TimeSeriesAggregationException
    {
        final TimeSeriesArray disaggregatedTS = TimeSeriesArrayTools.prepareTimeSeries(aggregatedTS, "", stepInMillis);
        int disaggIndex = 0;

        //Determine the value to divide each aggregated value by in order to set the disaggregated values.
        final float divider = (float)((double)aggregatedTS.getHeader().getTimeStep().getStepMillis() / (double)stepInMillis);

        //Working time starts at the forecast time and increments up to the end of the aggregated time series.
        for(long workingTime = disaggregatedTS.getHeader().getForecastTime(); workingTime <= aggregatedTS.getEndTime(); workingTime += stepInMillis)
        {
            //Note that working time will never be larger than aggregatedTS.getEndTime(), so the loop below will not allow the index to get 
            //too large.
            while(aggregatedTS.getTime(disaggIndex) < workingTime)
            {
                disaggIndex++;
            }
            disaggregatedTS.put(workingTime, aggregatedTS.getValue(disaggIndex) / divider);
        }

        return disaggregatedTS;
    }
}
