package ohd.hseb.hefs.utils.tsarrays.agg;

import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;

/**
 * Implementation of {@link OHDAggregator} that performs a minimum and maximum computation.
 * 
 * @author Hank.Herr
 */
public class ExtremeAggregator extends OHDAggregator
{
    public static final int MAXIMUM_TYPE = 1;
    public static final int MINIMUM_TYPE = 2;

    public ExtremeAggregator()
    {
        super();
    }

    public ExtremeAggregator(final int type)
    {
        this.setAggregationType(type);
    }

    @Override
    public float aggregate() throws TimeSeriesAggregationException
    {
        if(isAnyValueIsMissingPeriodNotCoveredOrNoInput())
        {
            if(ignoreMissingValues())
            {
                addWarningMessage("At least one missing data value was found and ignored.");
            }
            else
            {
                return Float.NaN;
            }
        }

        float result;
        long time = Long.MIN_VALUE;

        if(getAggregationType() == MAXIMUM_TYPE)
        {
            result = Float.NEGATIVE_INFINITY;
            for(int i = 0; i < getInputCount(); i++)
            {
                if((!TimeSeriesArrayTools.isOHDMissingValue(getInputValue(i)) && (getInputValue(i) > result)))
                {
                    result = getInputValue(i);
                    time = getInputValueTime(i);
                }
            }
        }
        else if(getAggregationType() == MINIMUM_TYPE)
        {
            result = Float.POSITIVE_INFINITY;
            for(int i = 0; i < getInputCount(); i++)
            {
                if((!TimeSeriesArrayTools.isOHDMissingValue(getInputValue(i)) && (getInputValue(i) < result)))
                {
                    result = getInputValue(i);
                    time = getInputValueTime(i);
                }
            }
        }
        else
        {
            throw new TimeSeriesAggregationException("Type index of " + getAggregationType()
                + " is not valid for an extreme aggregation.");
        }

        //Check for bad results
        if((result == Float.NEGATIVE_INFINITY) || (result == Float.POSITIVE_INFINITY))
        {
            return Float.NaN;
        }

        setAggregatedValueValidTime(time);
        return result;
    }

    @Override
    public String getAggregationDisplayName()
    {

        if(getAggregationType() == MAXIMUM_TYPE)
        {
            return "maximum";
        }
        else if(getAggregationType() == MINIMUM_TYPE)
        {
            return "minimum";
        }
        return "UNSPECIFIED";
    }

//    @Override
//    public boolean canBeAccumulatedWithPrevioutValue()
//    {
//        return true;
//    }
//
//    @Override
//    public float accumulateWithPreviousValue(final float previousValue,
//                                             final long timeWindowForPreviousValue,
//                                             final int timeSeriesIndexOfPreviousValue,
//                                             final float currentValue,
//                                             final long timeWindowForCurrentValue) throws TimeSeriesAggregationException
//    {
//        if(getAggregationType() == MAXIMUM_TYPE)
//        {
//            return Math.max(previousValue, currentValue);
//        }
//        else if(getAggregationType() == MINIMUM_TYPE)
//        {
//            return Math.min(previousValue, currentValue);
//        }
//        else
//        {
//            throw new TimeSeriesAggregationException("Type index of " + getAggregationType()
//                + " is not valid for an extreme aggregation.");
//        }
//    }

}
