package ohd.hseb.hefs.utils.tsarrays.agg;

import nl.wldelft.util.timeseries.TimeStep;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.util.misc.SegmentedLine;

/**
 * A CHPS/FEWS function for computing counter statistics on an ensemble. Refer to the counter aggregator in Graphics
 * Generator for an example. Counter types are specified by constants: NDTO, NDIS, NDMX, NDMN.
 * 
 * @author hank.herr
 */
public class NDCounterAggregator extends OHDAggregator
{
    /**
     * Constant identifier of count type "number of time steps to...". Based on ESPADP.
     */
    public static final String NDTO = "NDTO";

    /**
     * Constant identifier of count type "number of events...". Based on ESPADP.
     */
    public static final String NDIS = "NDIS";

    /**
     * Constant identifier of count type "number of time steps to the max". Based on ESPADP.
     */
    public static final String NDMX = "NDMX";

    /**
     * Constant identifier of count type "number of time steps to the min". Based on ESPADP.
     */
    public static final String NDMN = "NDMN";

    private String _countType = NDCounterAggregator.NDTO;
    private boolean _isAbove = true;
    private double _threshold = 0;
    private Long _basisDate = null;
    private TimeStep _dataTimeStep = null;

    public NDCounterAggregator(final String countType, final boolean isAbove, final double threshold)
    {
        super();
        setCountType(countType);
        setIsAbove(isAbove);
        setThreshold(threshold);
    }

    /**
     * @return The number of times the value is above/below the threshold.
     */
    private int calculateNDIS()
    {
        final int startIndex = 0;
        int count = 0;

        for(int i = startIndex; i < getInputCount(); i++)
        {
            if(TimeSeriesArrayTools.isOHDMissingValue(getInputValue(i)))
            {
                continue;
            }
            if(isAbove())
            {
                if(getInputValue(i) > getThreshold())
                {
                    count++;
                }
            }
            else
            {
                if(getInputValue(i) < getThreshold())
                {
                    count++;
                }
            }
        }

        return count;
    }

    /**
     * @return A count, as a double, of the number of steps between the {@link #_basisDate} and one time step before the
     *         first input value provided, calculated by calling {@link #getInputValueTime(int)} and subtracting one
     *         {@link #_dataTimeStep}.
     */
    private double computeCountBase()
    {
        if(_basisDate == null)
        {
            throw new IllegalStateException("An NDCounterAggregator must have its basis date set.");
        }
        if(_dataTimeStep == null)
        {
            throw new IllegalStateException("The _basisDate was set without the _dataTimeStep being set.");
        }
        final long defaultCountStartTime = getInputValueTime(0) - _dataTimeStep.getStepMillis();
//            System.err.println("####>> components -- " + defaultCountStartTime + ", " + _basisDate + ", " + " --- "
//                + (defaultCountStartTime - _basisDate) + " / " + _dataTimeStep.getStepMillis() + " ----> "
//                + ((double)(defaultCountStartTime - _basisDate) / (double)_dataTimeStep.getStepMillis()));
        return (double)(defaultCountStartTime - _basisDate) / (double)_dataTimeStep.getStepMillis();
    }

    /**
     * @return The number of steps to the minimum value of the time series. The count makes use of
     *         {@link #computeCountBase()}.
     */
    private int calculateNDMN() throws TimeSeriesAggregationException
    {
        final int startIndex = 0;
        int count = 0;

        final ExtremeAggregator minAgg = new ExtremeAggregator();
        minAgg.copySettingsFrom(this);
        minAgg.setAggregationType(ExtremeAggregator.MINIMUM_TYPE);
        minAgg.setIgnoreMissingValues(ignoreMissingValues());
        final float extremeValue = minAgg.aggregate();
        if(extremeValue == Float.NaN)
        {
            return Integer.MIN_VALUE;
        }

        for(int i = startIndex; i < getInputCount(); i++)
        {
            count++;
            if(TimeSeriesArrayTools.isOHDMissingValue(getInputValue(i)))
            {
                continue;
            }
//                System.out.println("i: " + i + " value: " + values[i]);
            if(Math.abs(getInputValue(i) - extremeValue) <= 0.00001)
            {
                break;
            }
        }

        return count + (int)computeCountBase();
//        System.out.println("extremeValue: " + extremeValue);
//        System.out.println("count: " + count + "\n");
    }

    /**
     * @return The number of steps to the maximum value of the time series. The count makes use of
     *         {@link #computeCountBase()}.
     */
    private int calculateNDMX() throws TimeSeriesAggregationException
    {

        final int startIndex = 0;
        int count = 0;

        final ExtremeAggregator maxAgg = new ExtremeAggregator();
        maxAgg.copySettingsFrom(this);
        maxAgg.setAggregationType(ExtremeAggregator.MAXIMUM_TYPE);
        maxAgg.setIgnoreMissingValues(ignoreMissingValues());
        final float extremeValue = maxAgg.aggregate();
        if(extremeValue == Float.NaN)
        {
            //TODO What to do about warning messages from maxAgg???
            return Integer.MIN_VALUE;
        }

        for(int i = startIndex; i < getInputCount(); i++)
        {
            count++;
            if(TimeSeriesArrayTools.isOHDMissingValue(getInputValue(i)))
            {
                continue;
            }
//                System.out.println("i: " + i + " value: " + values[i]);
            if(Math.abs(getInputValue(i) - extremeValue) <= 0.00001)
            {
                break;
            }
        }

        return count + (int)computeCountBase();
//        System.out.println("extremeValue: " + extremeValue);
//        System.out.println("count: " + count + "\n");
    }

    /**
     * @return The number of days/steps until the time series is above/below the threshold. The count makes use of
     *         {@link #computeCountBase()}.
     */
    private int calculateNDTO()
    {
        final int startIndex = 0;
        int count = 0;

        int i;
        for(i = startIndex; i < getInputCount(); i++)
        {
            count++;
            if(TimeSeriesArrayTools.isOHDMissingValue(getInputValue(i)))
            {
                continue;
            }
//                System.out.println("i: " + i + " value: " + values[i]);

            if(isAbove())
            {
                if(getInputValue(i) > getThreshold())
                {
                    break;
                }
            }
            else
            {
                if(getInputValue(i) < getThreshold())
                {
                    break;
                }
            }
        }
        if(i == getInputCount())
        {
            return Integer.MIN_VALUE;
        }

        return count + (int)computeCountBase();
//        System.out.println("threshold: " + getThreshold());
//        System.out.println("count: " + count + "\n");

    }

    public String getCountType()
    {
        return _countType;
    }

    public double getThreshold()
    {
        return _threshold;
    }

    public boolean isAbove()
    {
        return _isAbove;
    }

    public void setCountType(final String countType)
    {
        this._countType = countType;
    }

    public void setIsAbove(final boolean isAbove)
    {
        this._isAbove = isAbove;
    }

    public void setThreshold(final double threshold)
    {
        this._threshold = threshold;
    }

    /**
     * @param date
     * @param step
     */
    public void setBasisDateAndTimeStep(final Long date, final TimeStep step)
    {
        _basisDate = date;
        _dataTimeStep = step;
    }

    /**
     * @param parameterId Parameter id that was built via the CounterTimeSeriesAggregator class.
     * @return A string that can be used as an axis title, or null if the parameterId does not appear to be a valid
     *         counter id.
     */
    public static String buildReadableCounterLabelFromParameter(final String parameterId)
    {
        if(!parameterId.startsWith("COUNT_"))
        {
            return null;
        }

        String result = null;
        final SegmentedLine segLine = new SegmentedLine(parameterId, "_", SegmentedLine.MODE_NO_EMPTY_SEGS);
        if(segLine.getSegment(1).equalsIgnoreCase(NDTO))
        {
            result = "Number of Steps to ";
            if(segLine.getSegment(2).equalsIgnoreCase("ABOVE"))
            {
                result += " Go Above";
            }
            else
            {
                result += " Go Below";
            }
            result += segLine.getSegment(3);
        }
        else if(segLine.getSegment(1).equalsIgnoreCase(NDIS))
        {
            result = "Number of Values ";
            if(segLine.getSegment(2).equalsIgnoreCase("ABOVE"))
            {
                result += " Above";
            }
            else
            {
                result += " Below";
            }
            result += segLine.getSegment(3);
        }
        else if(segLine.getSegment(1).equalsIgnoreCase(NDMX))
        {
            result = "Number of Steps to Reach Maximum";
        }
        else if(segLine.getSegment(1).equalsIgnoreCase(NDMN))
        {
            result = "Number of Steps to Reach Minimum";
        }
        return result;
    }

    /**
     * @return True if the parmaeterId corresponds to data generated by this aggregator.
     */
    public static boolean isCounterParameterId(final String parameterId)
    {
        if(parameterId.startsWith("COUNT_"))
        {
            return true;
        }
        return false;
    }

    /**
     * Designed to work with the NDTO, NDIS, NDMX, and NDMN count types (see constants in this class).
     * 
     * @param countType The count type;
     * @param isAbove
     * @param threshold
     * @return A parameter id designed for counter parameter types, such as counts of the number of values above or
     *         below a threshold and counts of the number of time steps to an event.
     */
    public static String buildCounterParameterId(final String countType, final boolean isAbove, final double threshold)
    {
        String id = "COUNT_" + countType;
        if((countType.equalsIgnoreCase(NDTO)) || (countType.equalsIgnoreCase(NDIS)))
        {
            if(isAbove)
            {
                id += "_ABOVE";
            }
            else
            {
                id += "_BELOW";
            }
            id += "_" + threshold;
        }
        return id;
    }

    @Override
    public float aggregate() throws TimeSeriesAggregationException
    {
        if(isAnyValueIsMissingPeriodNotCoveredOrNoInput())
        {
            if(ignoreMissingValues())
            {
                addWarningMessage("At least one missing data value was found and ignored.");
            }
            else
            {
                return Float.NaN;
            }
        }

        if(getCountType().equalsIgnoreCase(NDCounterAggregator.NDTO))
        {
            final int value = calculateNDTO();
            if(value == Integer.MIN_VALUE)
            {
                return Float.NaN;
            }
            return value;
        }
        else if(getCountType().equalsIgnoreCase(NDCounterAggregator.NDIS))
        {
            return calculateNDIS();
        }
        else if(getCountType().equalsIgnoreCase(NDCounterAggregator.NDMX))
        {
            final int value = calculateNDMX();
            if(value == Integer.MIN_VALUE)
            {
                return Float.NaN;
            }
            return value;
        }
        else if(getCountType().equalsIgnoreCase(NDCounterAggregator.NDMN))
        {
            final int value = calculateNDMN();
            if(value == Integer.MIN_VALUE)
            {
                return Float.NaN;
            }
            return value;
        }
        throw new TimeSeriesAggregationException("Count type of '" + this.getCountType() + "' is not recognized.");
    }

    @Override
    public String getAggregationDisplayName()
    {
        if(getCountType().equalsIgnoreCase(NDCounterAggregator.NDTO))
        {
            return "number-of-steps-to";
        }
        else if(getCountType().equalsIgnoreCase(NDCounterAggregator.NDIS))
        {
            return "number-of-values-above-or-below";
        }
        else if(getCountType().equalsIgnoreCase(NDCounterAggregator.NDMX))
        {
            return "number-of-steps-to-max";
        }
        else if(getCountType().equalsIgnoreCase(NDCounterAggregator.NDMN))
        {
            return "number-of-steps-to-min";
        }
        return "UNSPECIFIED";
    }
}
