package ohd.hseb.hefs.utils.tsarrays.agg;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import nl.wldelft.util.Period;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;

import com.google.common.collect.Lists;

/**
 * Top level super class of all {@link OHDAggregator} implementations. It provides basic tools for setting/getting input
 * aggregation values, identifying the working aggregation period, computing needed proportions, and so on.
 * 
 * @author Hank.Herr
 */
public abstract class OHDAggregator
{
    private int _aggregationType = -1;

    /**
     * {@link List} of times for the values that apply to the current working period.
     */
    private final List<Long> _inputValueTimes = Lists.newArrayList();

    /**
     * The values that apply to this working period.
     */
    private final List<Float> _inputValues = Lists.newArrayList();

    /**
     * A {@link List} of the {@link Period} for which each input value applies.
     */
    private final List<Period> _inputAffectedWindow = Lists.newArrayList();

    /**
     * For each input value/time, the proportion of its window covered by the working period.
     */
    private final List<Double> _proportionsOfInputWindowInPeriod = Lists.newArrayList();

    /**
     * For each input value/time, the proportion of the working period covered by its applicable input window.
     */
    private final List<Double> _proportionsOfPeriodCoveredByInputWindow = Lists.newArrayList();

    /**
     * The current period being aggregated.
     */
    private Period _workingPeriod = null;

    /**
     * The computation time for which the aggregation is performed. If the aggregation uses an ending anchor, this
     * should equal the {@link #_workingPeriod}'s end time. Otherwise, it will be the midpoint of the
     * {@link #_workingPeriod}.
     */
    private long _computationTime = Long.MIN_VALUE;

    /**
     * Stores the time associated with a single aggregated value calculated via {@link #aggregate()}, if appropriate. If
     * either {@link Long#MIN_VALUE} or {@link Long#MAX_VALUE}, then it is assumed that this field is in appropriate for
     * a subclass.
     */
    private long _aggregatedValueValidTime = Long.MIN_VALUE;

    /**
     * True if missing values should be ignored by the aggregator, not resulting in an error. It is up to the aggregator
     * to determine how or if to use this flag.
     */
    private boolean _ignoreMissingValues = false;

    /**
     * Records warning messages generated by the aggregator, so that only one of each type of warning message is kept.
     */
    private final HashSet<String> _warningMessages = new HashSet<>();

    /**
     * An instance should call this method to queue up warning messages.
     * 
     * @param message The message to add, but only if it has not already been added.
     */
    protected void addWarningMessage(final String message)
    {
        _warningMessages.add(message);
    }

    /**
     * For use by the {@link TimeSeriesArrayAggregator} class only.
     */
    protected Collection<String> getWarningMessages()
    {
        return _warningMessages;
    }

    /**
     * @return True if {@link #_proportionsOfPeriodCoveredByInputWindow} specifies a total that equals 1.
     */
    public boolean isPeriodCompletelyCovered()
    {
        Double sum = 0D;
        for(final Double prop: _proportionsOfPeriodCoveredByInputWindow)
        {
            sum += prop;
        }
        if(sum - 1.0D > 0.000001D)
        {
            System.err.println("####>> A SUM THAT EXCEEDS 1 HAS BEEN FOUND FOR THE PROPORTIONS, "
                + "WHICH SHOULD NEVER HAPPEN!!!");
        }
        return (Math.abs(sum - 1.0D) < 0.000001D);
    }

    public void copySettingsFrom(final OHDAggregator base)
    {
        clear();
        _aggregationType = base._aggregationType;
        _inputValueTimes.addAll(base._inputValueTimes);
        _inputValues.addAll(base._inputValues);
        _inputAffectedWindow.addAll(base._inputAffectedWindow);
        _proportionsOfInputWindowInPeriod.addAll(base._proportionsOfInputWindowInPeriod);
        _proportionsOfPeriodCoveredByInputWindow.addAll(base._proportionsOfPeriodCoveredByInputWindow);

    }

    public void setIgnoreMissingValues(final boolean b)
    {
        _ignoreMissingValues = b;
    }

    public boolean ignoreMissingValues()
    {
        return _ignoreMissingValues;
    }

    public void setAggregationType(final int type)
    {
        _aggregationType = type;
    }

    public int getAggregationType()
    {
        return _aggregationType;
    }

    /**
     * @return True if any provides value matches the checks performed in
     *         {@link TimeSeriesArrayTools#isOHDMissingValue(float)}.
     */
    public boolean isAnyInputValueMissing()
    {
        for(final Float value: _inputValues)
        {
            if(TimeSeriesArrayTools.isOHDMissingValue(value))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * @return false if any input value in the window is missing, the period is not completely covered (the first is
     *         probably redundant with this), or there are no input values.
     */
    protected boolean isAnyValueIsMissingPeriodNotCoveredOrNoInput()
    {
        if(isAnyInputValueMissing() || !isPeriodCompletelyCovered() || (getInputCount() <= 0))
        {
            return true;
        }

        return false;
    }

    public void clear()
    {
        _inputValues.clear();
        _inputValueTimes.clear();
        _inputAffectedWindow.clear();
        _proportionsOfInputWindowInPeriod.clear();
        _proportionsOfPeriodCoveredByInputWindow.clear();
        _workingPeriod = null;
    }

    /**
     * This should only be called by the aggregation algorithm, not outside tools. Hence, it is protected.
     * 
     * @param workingPeriod The working period for current aggregation.
     */
    protected void setWorkingPeriod(final Period workingPeriod)
    {
        _workingPeriod = workingPeriod;
    }

    /**
     * Set to the time at which the computed aggregated value will be recorded. It should be a time within
     * {@link #_workingPeriod} and most likely either the end or mid point of that period.
     * 
     * @param time The time at which the aggregated value will be recorded.
     */
    protected void setComputationTime(final long time)
    {
        _computationTime = time;
    }

    /**
     * This method detects two types of missing values dictated by {@link TimeSeriesArrayTools#isOHDMissingValue(float)}
     * . In either case, it will store a NaN value!
     * 
     * @param value Value to add to the one-step aggregation.
     * @param time Its time.
     * @param propOfInputWindowInPeriod The proportion of the input value window within the aggregation period.
     * @param propOfPeriodCoveredByInputWindow The proportion of the aggregation period covered by the input value
     *            window.
     * @param affectedWindow The window "affected" by the input value (i.e., its applicable window, whatever term you
     *            want to use).
     */
    public void addInputValue(final float value,
                              final long time,
                              final double propOfInputWindowInPeriod,
                              final double propOfPeriodCoveredByInputWindow,
                              final Period affectedWindow)
    {
        _inputValueTimes.add(time);
        if(TimeSeriesArrayTools.isOHDMissingValue(value)) //Allow for two types of missing: NaN and -999.
        {
            _inputValues.add(Float.NaN);
        }
        else
        {
            _inputValues.add(value);
        }
        _inputAffectedWindow.add(affectedWindow);
        _proportionsOfInputWindowInPeriod.add(propOfInputWindowInPeriod);
        _proportionsOfPeriodCoveredByInputWindow.add(propOfPeriodCoveredByInputWindow);
    }

    protected Period getWorkingPeriod()
    {
        return _workingPeriod;
    }

    protected long getComputationTime()
    {
        return _computationTime;
    }

    protected int getInputCount()
    {
        return _inputValues.size();
    }

    protected float getInputValue(final int index)
    {
        return _inputValues.get(index);
    }

    protected long getInputValueTime(final int index)
    {
        return _inputValueTimes.get(index);
    }

    protected Period getInputAffectedWindow(final int index)
    {
        return _inputAffectedWindow.get(index);
    }

    protected double getProportionsOfInputWindowInPeriod(final int index)
    {
        return _proportionsOfInputWindowInPeriod.get(index);
    }

    protected double getProportionsOfPeriodCoveredByInputWindow(final int index)
    {
        return _proportionsOfPeriodCoveredByInputWindow.get(index);
    }

    /**
     * All subclasses must override this method. If a subclass depends upon the entire period being appropriately
     * covered when aggregating, then call the method {@link #isAnyValueIsMissingPeriodNotCoveredOrNoInput()} and return
     * {@link Float#NaN} if the returned boolean is true.
     * 
     * @return The aggregated value using the get methods as needed.
     * @throws TimeSeriesAggregationException
     */
    public abstract float aggregate() throws TimeSeriesAggregationException;

    /**
     * @return An appropriate name to display in log messages associated with the aggregation.
     */
    public abstract String getAggregationDisplayName();

    /**
     * Called immediately after {@link #aggregate()} is called. The returned milliseconds value will be recorded as a
     * date string within the comment field corresponding to the value in a {@link TimeSeriesArray}.
     * 
     * @return The value of {@link #_aggregatedValueValidTime}, which is the time associated with the value returned by
     *         the most recent call to {@link #aggregate()}.
     */
    public long getAggregatedValueValidTime()
    {
        return _aggregatedValueValidTime;
    }

    /**
     * For aggregators for which the computed value valid time is appropriate to record, such as the time assocaited
     * with a maximum or minimum value, the aggregator should call this method during {@link #aggregate()}.
     * 
     * @param validTime Time associated with an aggregated value.
     */
    protected void setAggregatedValueValidTime(final long validTime)
    {
        _aggregatedValueValidTime = validTime;
    }

}
