package ohd.hseb.hefs.utils.tsarrays.agg;

import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;

import javax.management.ListenerNotFoundException;

import nl.wldelft.util.Period;
import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.ParameterType;
import nl.wldelft.util.timeseries.SimpleEquidistantTimeStep;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeStep;
import ohd.hseb.hefs.utils.Dyad;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.xml.vars.XMLTimeStep;
import ohd.hseb.util.misc.HCalendar;

import com.google.common.collect.Lists;

/**
 * Uses an {@link OHDAggregator} to aggregate a time series. See the javadoc associated with the methods to see how it
 * works.
 * 
 * @author Hank.Herr
 */
public class TimeSeriesArrayAggregator
{

    /**
     * This must be initialized before calling {@link #computeAffectedWindow(TimeSeriesArray, int)}, as it records
     * already computed windows in this array indexed based on the time series to aggregate.
     */
    private Period[] _computedAffectedWindows;

    /**
     * Records warning messages generated by the aggregator, so that only one of each type of warning message is kept.
     */
    private final HashSet<String> _warningMessages = new HashSet<>();

    /**
     * For use by the {@link TimeSeriesArrayAggregator} class only.
     */
    protected Collection<String> getWarningMessages()
    {
        return _warningMessages;
    }

    /**
     * @param startTime The start time of the aggregation.
     * @param endTime The end time.
     * @param computationTimeStep The size of steps at which aggregated values are computed.
     * @param aggPeriod The period for each aggregated computation.
     * @param anchor The anchor, centered or ending, specifying how the aggregation period is positioned relative to the
     *            computation time step.
     * @return A {@link ListenerNotFoundException} of {@link Dyad}s each specifying the computation time ({@link Long})
     *         and the aggregation {@link Period} corresponding to that time.
     * @throws TimeSeriesAggregationException If the start time is after or on the end time.
     */
    private List<Dyad<Long, Period>> buildRangeArray(final long startTime,
                                                     final long endTime,
                                                     final ComputationTimeStep computationTimeStep,
                                                     final AggregationPeriod aggPeriod,
                                                     final PeriodAnchor anchor) throws TimeSeriesAggregationException
    {
        final List<Dyad<Long, Period>> periodList = Lists.newArrayList();
        Period period = null;

        if(startTime >= endTime)
        {
            throw new TimeSeriesAggregationException("Start time (" + HCalendar.buildDateTimeTZStr(startTime)
                + ") must be before the end time + (" + HCalendar.buildDateTimeTZStr(endTime) + ").");
        }

        //For a period type aggregation, the Period is start-to-end and the anchor time is based on the anchor.
        //The aggregation period has no meaning.
        if(XMLTimeStep.isPeriod(computationTimeStep))
        {
            period = new Period(startTime, endTime);
            periodList.add(new Dyad<Long, Period>(anchor.computeAnchorTime(startTime, endTime), period));
            return periodList;
        }

        //For a year aggregation, the first aggregation is the start time plus a number of years as specified.
        //The aggregation period is then determined such that the aggregated value will be computed for that time
        //given the anchor position and aggregation period parameters.  This should handle a start time landing on a 
        //leap year by looping the date to be Mar 1 and using that Mar 1 for all time steps that follow.  
        else if(computationTimeStep.isUnitYear())
        {
            // The first used calendar the current computationTime year 
            final Calendar workingCal = HCalendar.computeCalendarFromMilliseconds(startTime);
            workingCal.add(Calendar.YEAR, computationTimeStep.getMultiplier());
            while(workingCal.getTimeInMillis() <= endTime)
            {
                period = aggPeriod.computePeriod(startTime, workingCal.getTimeInMillis(), computationTimeStep, anchor);
                periodList.add(new Dyad<Long, Period>(workingCal.getTimeInMillis(), period));

                //Next iteration
                workingCal.add(Calendar.YEAR, computationTimeStep.getMultiplier());
            }
        }

        //For a month aggregation, the first computation time is the start time plus the specified number of months.
        //This adds one to the month field of a Calendar and lets the Calendar handle the implications.  This
        //means that if a time is for the 31st of a month. then when the 31st does not exist, the date will be
        //wrapped to the next month.  Hence, if user's want whole months, they should use the first of the next
        //month at hour 0.
        //The aggregation period is then determined such that the aggregated value will be computed for that time
        //given the anchor position and aggregation period parameters.
        else if(computationTimeStep.isUnitMonth())
        {
            // The first used calendar the current computationTime year 
            final Calendar workingCal = HCalendar.computeCalendarFromMilliseconds(startTime);
            workingCal.add(Calendar.MONTH, computationTimeStep.getMultiplier());
            while(workingCal.getTimeInMillis() <= endTime)
            {
                period = aggPeriod.computePeriod(startTime, workingCal.getTimeInMillis(), computationTimeStep, anchor);
                periodList.add(new Dyad<Long, Period>(workingCal.getTimeInMillis(), period));

                //Next iteration
                workingCal.add(Calendar.MONTH, computationTimeStep.getMultiplier());
            }
        }

        //All other units (week, day, hour) which have regular time steps.
        else
        {
            final long unitInMillis = computationTimeStep.computeIntervalInMillis();
            long workingTime = startTime + unitInMillis;
            while(workingTime <= endTime)
            {
                period = aggPeriod.computePeriod(startTime, workingTime, computationTimeStep, anchor);
                periodList.add(new Dyad<Long, Period>(anchor.computeAnchorTime(period), period));
                workingTime += unitInMillis;
            }
        }

        return periodList;
    }

//XXX See the XXX below!
//    private TimeSeriesArray calculateUserDefinedAccumulatedAgg(final TimeSeriesArray input,
//                                                               final ComputationTimeStep computationTimeStep,
//                                                               final Period overallPeriod,
//                                                               final AggregationPeriod aggregationPeriod,
//                                                               final PeriodAnchor periodAnchor,
//                                                               final OHDAggregator aggregator) throws Exception
//    {
//        final TimeSeriesArray preComputedAgg = calculateUserDefinedAggregation(input,
//                                                                               computationTimeStep,
//                                                                               overallPeriod,
//                                                                               new AggregationPeriod(),
//                                                                               new PeriodAnchor(),
//                                                                               aggregator);
//        for(int i = 1; i < preComputedAgg.size(); i++)
//        {
//            preComputedAgg.setValue(i,
//                                    aggregator.accumulateWithPreviousValue(preComputedAgg.getValue(i - 1),
//                                                                           preComputedAgg.getTime(i - 1)
//                                                                               - overallPeriod.getStartTime(),
//                                                                           i - 1,
//                                                                           preComputedAgg.getValue(i),
//                                                                           computationTimeStep.computeIntervalInMillis()));
//        }
//
//        return preComputedAgg;
//    }

    /**
     * This can only be called by an appropriate wrapper method which is responsible for setting the output header
     * appropriately after it is returned from this.
     * 
     * @param input The time series to aggregate.
     * @param outputTimeStep Target time step in String format.
     * @param period Overall aggregation period, start to end.
     * @param aggregator The {@link OHDAggregator} instance to use.
     * @return Output time series with the header set as a copy of the input. It must be changed outside this method to
     *         appropriate values!
     * @throws Exception
     */
    protected TimeSeriesArray calculateUserDefinedAggregation(final TimeSeriesArray input,
                                                              final ComputationTimeStep computationTimeStep,
                                                              final Period overallPeriod,
                                                              final AggregationPeriod aggregationPeriod,
                                                              final PeriodAnchor periodAnchor,
                                                              final boolean prefixWithZero,
                                                              final OHDAggregator aggregator) throws Exception
    {
        //XXX This old if below would call special methods in the aggregator to accumulate the aggregation,
        //theoretically preventing it from needing to start from time 0 in order to compute the number
        //for each step.  However, the performance improvement was too small to justify the additional code,
        //so I removed it.
        //
        //Special case for accumulated aggregation if the aggregator allows.  This should improve performance,
        //though perhaps not as much as I had hoped.
//        if(AggregationPeriod.isAccumulated(aggregationPeriod) && aggregator.canBeAccumulatedWithPrevioutValue())
//        {
//            return calculateUserDefinedAccumulatedAgg(input,
//                                                      computationTimeStep,
//                                                      overallPeriod,
//                                                      aggregationPeriod,
//                                                      periodAnchor,
//                                                      aggregator);
//        }

        //Used to track affected windows already computed. 
        _computedAffectedWindows = new Period[input.size()];
        Arrays.fill(_computedAffectedWindows, null);

        final TimeSeriesArray output = new TimeSeriesArray(new DefaultTimeSeriesHeader());
        TimeSeriesArrayTools.prepareHeader(input, output, null);

        //Check if the user-selected time step is valid.
        //This must call input.getHeader().getTimeStep(), because input.getTimeStep() may not be properly set 
        //if the input was itself an aggregated time series!
        checkTimeStepValidation(input.getHeader().getTimeStep(), computationTimeStep);

        //Determine the used period anchor using the aggregation period.
        final PeriodAnchor usedAnchor = aggregationPeriod.determinedUsedAnchor(periodAnchor);

        //Build the list of aggregation periods
        final List<Dyad<Long, Period>> rangeList = buildRangeArray(overallPeriod.getStartTime(),
                                                                   overallPeriod.getEndTime(),
                                                                   computationTimeStep,
                                                                   aggregationPeriod,
                                                                   usedAnchor);
        if(rangeList == null || rangeList.isEmpty())
        {
            throw new TimeSeriesAggregationException("Time series does not have enough data given the computation time step: "
                + computationTimeStep);
        }

        //Find first non-missing value.
        int firstNonMissingIndex = 0;
        while((firstNonMissingIndex < input.size())
            && (TimeSeriesArrayTools.isOHDMissingValue(input.getValue(firstNonMissingIndex))))
        {
            firstNonMissingIndex++;
        }
        if(firstNonMissingIndex == input.size())
        {
            throw new TimeSeriesAggregationException("Time series contains all missing data.");
        }

        //Adjust the first subperiod if needed for the case where there is no T0 data, but there is a T0 + 1 step value.
        //Be sure to use first non-missing and not just 0, because the first value in the ts could be NaN (i.e., input
        //is not trimmed!).
        Period affectedWindow = computeAffectedWindow(input, firstNonMissingIndex);
        if((rangeList.get(0).getSecond().getStartTime() < affectedWindow.getStartTime())
            && (rangeList.get(0).getSecond().getEndTime() >= affectedWindow.getStartTime()))
        {
            final Dyad<Long, Period> entry = rangeList.get(0);

            //Only adjust the first period if the difference between the affected window start time and
            //the range list start time is less than one time step.
            if(Math.abs(entry.getSecond().getStartTime() - affectedWindow.getStartTime()) <= input.getHeader()
                                                                                                  .getTimeStep()
                                                                                                  .getStepMillis())
            {
                rangeList.set(0, new Dyad<Long, Period>(entry.getFirst(), new Period(affectedWindow.getStartTime(),
                                                                                     entry.getSecond().getEndTime())));

            }
        }

        //To make the algorithm efficient, lastWorkingIndex for the next sub period will be determined when computing
        //this period so that searching does not need to be done again.
        int lastWorkingIndex = 0;
        int nextSubPeriodIndex = 0;

        //For each sub period...
        for(final Dyad<Long, Period> subPeriod: rangeList)
        {
            aggregator.clear();
            aggregator.setComputationTime(subPeriod.getFirst());
            aggregator.setWorkingPeriod(subPeriod.getSecond());

            //Determine the next sub period.  Null indicates no more periods after the working one.
            nextSubPeriodIndex++;
            Dyad<Long, Period> nextSubPeriod = null;
            if(nextSubPeriodIndex < rangeList.size())
            {
                nextSubPeriod = rangeList.get(nextSubPeriodIndex);
            }

            //Search the input time series for values affecting the subperiod.
            int valueIndex;
            for(valueIndex = Math.max(lastWorkingIndex, 0); valueIndex < input.size(); valueIndex++)
            {
                if(valueIndex == lastWorkingIndex)
                {
                    lastWorkingIndex = -1;
                }

                //Skip missing values.  This will allow for the aggregator.isPeriodCompletelyCovered()
                //call below to catch the problem.  Missing values are never passed down into the aggregator.
                //NOTE: aggregator.addInputValue(...) explicitly allows for missings to be given to it, but
                //  that causes problems some time.
                if(TimeSeriesArrayTools.isOHDMissingValue(input.getValue(valueIndex)))
                {
                    continue;
                }

                //Compute the affected window.
                affectedWindow = computeAffectedWindow(input, valueIndex);

                //If the affected windows is after or on the next sub period's window, then
                //set the last working index appropriately for the next sub period.
                if((nextSubPeriod != null) && (lastWorkingIndex < 0)
                    && (nextSubPeriod.getSecond().isAnyTimeCommon(affectedWindow)))
                {
                    lastWorkingIndex = valueIndex;
                }

                //Break out when the subperiod's end time is before the affected window.
                if(subPeriod.getSecond().getEndTime() <= affectedWindow.getStartTime())
                {
                    break;
                }

                //Continue to the next value and do nothing if the affected window is before the
                //desired working period.
                if(affectedWindow.getEndTime() <= subPeriod.getSecond().getStartTime())
                {
                    continue;
                }

                //Add to the aggregator.
                aggregator.addInputValue(input.getValue(valueIndex),
                                         input.getTime(valueIndex),
                                         computeProportionOfAffectedWindowInAggregationPeriod(affectedWindow,
                                                                                              subPeriod.getSecond()),
                                         computeProportionOfAggregationPeriodCoveredByTheAffectedWindow(affectedWindow,
                                                                                                        subPeriod.getSecond()),
                                         affectedWindow);
            }

            //Let the aggregator decide if the return value should be NaN or a number.
            output.putValue(subPeriod.getFirst(), aggregator.aggregate());

            //Record the aggregated value valid time, if specified, as a date string comment associated with the index of the just set value.
            if((aggregator.getAggregatedValueValidTime() != Long.MIN_VALUE)
                && (aggregator.getAggregatedValueValidTime() != Long.MAX_VALUE))
            {
                output.setComment(TimeSeriesArrayTools.getIndexOfTime(output, subPeriod.getFirst()),
                                  HCalendar.buildDateTimeStr(aggregator.getAggregatedValueValidTime()));
            }
        }

        //Handle the prefix with zero flag... The value will be placed one time step BEFORE the current first value.
        if(prefixWithZero)
        {
            output.putValue(output.getStartTime() - computationTimeStep.computeIntervalInMillis(), 0f);
        }

        //Sets the time step of the header appropriately, but does not change the parameterId.  Also, for
        //'period' time steps, the result of this method is a Long.MIN_VALUE time step.  
        //It also sets the parameter type to either accumulative or instanenous (if centered).  
        TimeSeriesArrayTools.prepareHeader(input,
                                           output,
                                           "",
                                           computationTimeStep.buildTimeStepStr(),
                                           computationTimeStep.computeIntervalInMillis());
        ((DefaultTimeSeriesHeader)output.getHeader()).setParameterType(ParameterType.ACCUMULATIVE);

        //Set the parameter type based on centered or not.
        if(usedAnchor.isCentered())
        {
            ((DefaultTimeSeriesHeader)output.getHeader()).setParameterType(ParameterType.INSTANTANEOUS);
        }

        //For a 'period' aggregation, create a time step that is equal to the size of the period, start to end, and
        //set it.  This will allow for the width of the period to be carried along as a time step even though the
        //series can have only a single time series value.
        if(XMLTimeStep.isPeriod(computationTimeStep))
        {
            final Dyad<Long, Period> subPeriod = rangeList.get(0); //It should not be possible for this to fail, given code above.
            ((DefaultTimeSeriesHeader)output.getHeader()).setTimeStep(SimpleEquidistantTimeStep.getInstance(subPeriod.getSecond()
                                                                                                                     .getDuration()));
        }

        //Collect the warning messages.
        _warningMessages.addAll(aggregator.getWarningMessages());

        return output;
    }

    /**
     * @param input The time series containing all values.
     * @param valueIndex The index of the value to compute the affected window for.
     * @return A {@link Period} specifying the time window that the value is assigned to. This assumes it is a regular
     *         time series. So, for instantaneous data, the window is +/- half of the step (the value is at the mid
     *         point). For accumulated or mean data, it is the time step ending at the value's measurement time.
     */
    private Period computeAffectedWindow(final TimeSeriesArray input, final int valueIndex)
    {
        //Compute the affected window, if needed.
        if(_computedAffectedWindows[valueIndex] == null)
        {
            long startTime;
            long endTime;
            if(input.getHeader().getParameterType().equals(ParameterType.INSTANTANEOUS))
            {
                startTime = input.getTime(valueIndex) - (long)(0.5 * input.getHeader().getTimeStep().getStepMillis());
                endTime = input.getTime(valueIndex) + (long)(0.5 * input.getHeader().getTimeStep().getStepMillis());
            }
            else
            {
                startTime = input.getTime(valueIndex) - input.getHeader().getTimeStep().getStepMillis();
                endTime = input.getTime(valueIndex);
            }
            _computedAffectedWindows[valueIndex] = new Period(startTime, endTime);
        }
        return _computedAffectedWindows[valueIndex];
    }

    /**
     * @return The proportion (i.e., 0 to 1) of the affected window contained within the provided period.
     */
    private double computeProportionOfAffectedWindowInAggregationPeriod(final Period affectedWindow,
                                                                        final Period aggregationPeriod)
    {
        //If the affected window is outside the aggregation period
        if((affectedWindow.getStartTime() >= aggregationPeriod.getEndTime())
            || (affectedWindow.getEndTime() <= aggregationPeriod.getStartTime()))
        {
            return 0D;
        }

        //If the affected window is completely contained in aggregation period.
        if((affectedWindow.getStartTime() >= aggregationPeriod.getStartTime())
            && (affectedWindow.getEndTime() <= aggregationPeriod.getEndTime()))
        {
            return 1.00D;
        }

        //If the aggregation period is completely contained in the affected window.
        //This should never happen because the target time step must be larger than the input time step, but I'll include
        //it anyway.
        if((affectedWindow.getStartTime() <= aggregationPeriod.getStartTime())
            && (affectedWindow.getEndTime() >= aggregationPeriod.getEndTime()))
        {
            return (double)aggregationPeriod.getDuration() / (double)affectedWindow.getDuration();
        }

        //If the affected window start time is within the aggregation period.  We know from previous ifs that the 
        //end time must be after the period for it to not be completed contained within the aggregation period.
        if(affectedWindow.getStartTime() > aggregationPeriod.getStartTime())
        {
            return (double)(aggregationPeriod.getEndTime() - affectedWindow.getStartTime())
                / (double)affectedWindow.getDuration();
        }

        //If the affected window end time is within the aggregation period.  We know from previous ifs that the 
        //start time must be before the period for it to not be completed contained within the aggregation period.
        if(affectedWindow.getEndTime() < aggregationPeriod.getEndTime())
        {
            return (double)(affectedWindow.getEndTime() - aggregationPeriod.getStartTime())
                / (double)affectedWindow.getDuration();
        }

        System.err.println("####>> NONE OF THE PROPORTION CONDITIONS IS SATISFIED... THIS SHOULD NEVER HAPPEN!");
        return 0D; //Should never happen!!!
    }

    /**
     * @return The proportion (i.e., 0 to 1) of the aggregation period covered by the affected window.
     */
    private double computeProportionOfAggregationPeriodCoveredByTheAffectedWindow(final Period affectedWindow,
                                                                                  final Period aggregationPeriod)
    {
        //If the affected window is outside the aggregation period
        if((affectedWindow.getStartTime() >= aggregationPeriod.getEndTime())
            || (affectedWindow.getEndTime() <= aggregationPeriod.getStartTime()))
        {
            return 0D;
        }

        //If the affected window is completely contained in aggregation period.
        if((affectedWindow.getStartTime() >= aggregationPeriod.getStartTime())
            && (affectedWindow.getEndTime() <= aggregationPeriod.getEndTime()))
        {
            return (double)affectedWindow.getDuration() / (double)aggregationPeriod.getDuration();
        }

        //If the aggregation period is completely contained in the affected window.
        //This should never happen because the target time step must be larger than the input time step, but I'll include
        //it anyway.
        if((affectedWindow.getStartTime() <= aggregationPeriod.getStartTime())
            && (affectedWindow.getEndTime() >= aggregationPeriod.getEndTime()))
        {
            return 1.0D;
        }

        //If the affected window start time is within the aggregation period.  We know from previous ifs that the 
        //end time must be after the period for it to not be completed contained within the aggregation period.
        if(affectedWindow.getStartTime() > aggregationPeriod.getStartTime())
        {
            return (double)(aggregationPeriod.getEndTime() - affectedWindow.getStartTime())
                / (double)aggregationPeriod.getDuration();
        }

        //If the affected window end time is within the aggregation period.  We know from previous ifs that the 
        //start time must be before the period for it to not be completed contained within the aggregation period.
        if(affectedWindow.getEndTime() < aggregationPeriod.getEndTime())
        {
            return (double)(affectedWindow.getEndTime() - aggregationPeriod.getStartTime())
                / (double)aggregationPeriod.getDuration();
        }

        System.err.println("####>> NONE OF THE PROPORTION CONDITIONS IS SATISFIED... THIS SHOULD NEVER HAPPEN!");
        return 0D; //Should never happen!!!
    }

    /**
     * @throws TimeSeriesAggregationException If the units are not compatible, likely due to the output time step being
     *             less that of the input time series, which should never be the case for an aggregation.
     */
    private void checkTimeStepValidation(final TimeStep inputTimeStep, final ComputationTimeStep computationTimeStep) throws TimeSeriesAggregationException
    {
        final long outputTS = computationTimeStep.computeIntervalInMillis();
        if(outputTS == Long.MIN_VALUE) //An irregular unit (month, year) or a special unit is being used.
        {
            //TODO Shouldn't this still do a check?
            return;
        }
        else
        {
            final long inputTS = inputTimeStep.getStepMillis();

            //case 1:
            //the output time step should be larger than the time step of input time series
            if(outputTS < inputTS)
            {
                throw new TimeSeriesAggregationException("Aggregation time step \"" + computationTimeStep
                    + "\" must be larger than the input time step \"" + inputTimeStep.toString() + "\"");
            }
        }
    }

}
