package ohd.hseb.hefs.utils.xml;

import java.util.StringTokenizer;

import org.xml.sax.Attributes;

/**
 * Paired with {@link ArrayXMLWriter}, it performs similarly to {@link CollectionXMLReader}, in that it directly
 * populates a storage array that has already been sized. If the number of elements read in does not match the size
 * exactly, an error will be thrown.<br>
 * <br>
 * This class handles the following array types:<br>
 * <br>
 * 1. primitives byte, chart, short, int, long, float, and double: read in via corresponding {@link XMLVariable}
 * subclasses.<br>
 * 2. Objects implementing {@link XMLReadable} (including {@link XMLVariable} subclasses): The {@link XMLReaderFactory}
 * must return instances of the object to store, which are read in directory via the {@link XMLReadable#getReader()}
 * returned value.<br>
 * 3. Objects read in via {@link XMLVariable}: the {@link XMLReaderFactory} must return instances of {@link XMLVariable}
 * , and the {@link XMLVariable#get()} return value, which must match the array component type, is stored in the array.<br>
 * <br>
 * For examples, see {@link ArrayXMLReaderWriterTest}.
 * 
 * @author Hank.Herr
 * @param <E>
 */
public class ArrayXMLReader<E extends XMLReadable> extends XMLReaderAdapter
{
    /**
     * The array being populated is stored as an {@link Object}.
     */
    private Object _backingArray;

    /**
     * The factory used to populate individual values.
     */
    private XMLReaderFactory<E> _factory;

    /**
     * The tag name for the values in the array.
     */
    private String _tagNameForValues;

    /**
     * Length of the array, used internally.
     */
    private int _arrayLength;

    /**
     * Used only while reading, it stores the working index.
     */
    private int _valueIndex;

    /**
     * Used only while reading, it stores the working {@link XMLReadable}.
     */
    private XMLReadable _temp;

    /**
     * Used only while reading, it stores the working {@link XMLReader} that the working {@link XMLReadable} provides.
     */
    private XMLReader _tempReader;

    /**
     * The delimiter used to separate items. If the main element has text for it, and the text includes _delimiter, then
     * that text is assumed to provide the array elements. Note that this may cause confusion with subelements, as they
     * will still be processed and will be assumed to add values beyond those added in the delimited text. Hence, use
     * one or the other, but NOT both!
     */
    private char _delimiter = '|';

    /**
     * If the array component type is an Object, not a primitive, then a factory must be provided to deal with it. One
     * exception: String arrays can handled via this constructor (to employ a factory) or directly.
     * 
     * @param tag
     * @param arrayToPopulate
     * @param factory Returns {@link XMLReader} instances to read in values to store in the array.
     */
    public ArrayXMLReader(final String tag, final Object[] arrayToPopulate, final XMLReaderFactory<E> factory)
    {
        super(tag);
        _factory = factory;
        _tagNameForValues = factory.get().getReader().getXMLTagName();
        _backingArray = arrayToPopulate;
        _arrayLength = arrayToPopulate.length;

        if(_factory == null)
        {
            throw new IllegalArgumentException("The factory cannot be null when and Object is used for the array type.");
        }
    }

    public ArrayXMLReader(final String tag, final String valueTagName, final byte[] arrayToPopulate)
    {
        super(tag);
        initForPrimitives(arrayToPopulate, arrayToPopulate.length, valueTagName);
    }

    public ArrayXMLReader(final String tag, final String valueTagName, final char[] arrayToPopulate)
    {
        super(tag);
        initForPrimitives(arrayToPopulate, arrayToPopulate.length, valueTagName);
    }

    public ArrayXMLReader(final String tag, final String valueTagName, final short[] arrayToPopulate)
    {
        super(tag);
        initForPrimitives(arrayToPopulate, arrayToPopulate.length, valueTagName);
    }

    public ArrayXMLReader(final String tag, final String valueTagName, final int[] arrayToPopulate)
    {
        super(tag);
        initForPrimitives(arrayToPopulate, arrayToPopulate.length, valueTagName);
    }

    public ArrayXMLReader(final String tag, final String valueTagName, final long[] arrayToPopulate)
    {
        super(tag);
        initForPrimitives(arrayToPopulate, arrayToPopulate.length, valueTagName);
    }

    public ArrayXMLReader(final String tag, final String valueTagName, final float[] arrayToPopulate)
    {
        super(tag);
        initForPrimitives(arrayToPopulate, arrayToPopulate.length, valueTagName);
    }

    public ArrayXMLReader(final String tag, final String valueTagName, final double[] arrayToPopulate)
    {
        super(tag);
        initForPrimitives(arrayToPopulate, arrayToPopulate.length, valueTagName);
    }

    public ArrayXMLReader(final String tag, final String valueTagName, final String[] arrayToPopulate)
    {
        super(tag);
        initForPrimitives(arrayToPopulate, arrayToPopulate.length, valueTagName);
    }

    private void initForPrimitives(final Object backingArray, final int arrayLength, final String tagNameForValues)
    {
        _factory = null;
        _tagNameForValues = tagNameForValues;
        _backingArray = backingArray;
        _arrayLength = arrayLength;
        if(arrayLength == 0)
        {
            throw new IllegalArgumentException("Cannot read a 0-length double array.");
        }
    }

    /**
     * Puts the value and increments {@link #_valueIndex}.
     * 
     * @param value String specifying the value in text..
     * @throws XMLReaderException If a problem is detected.
     */
    private void putPrimitiveValue(final String value) throws XMLReaderException
    {
        //I'm using placementIndex, because I wanted to increase _valueIndex here rather than at the bottom
        //of this method where it may be overlooked.
        final int placementIndex = _valueIndex;
        _valueIndex++;

        if(placementIndex >= _arrayLength)
        {
            throw new XMLReaderException("Array item index " + placementIndex
                + " is as long or longer than the array length " + _arrayLength + ".");
        }
        try
        {
            if(_backingArray instanceof byte[])
            {
                ((byte[])_backingArray)[placementIndex] = Byte.parseByte(value);
            }
            else if(_backingArray instanceof char[])
            {
                ((char[])_backingArray)[placementIndex] = value.charAt(0);
            }
            else if(_backingArray instanceof short[])
            {
                ((short[])_backingArray)[placementIndex] = Short.parseShort(value);
            }
            else if(_backingArray instanceof int[])
            {
                ((int[])_backingArray)[placementIndex] = Integer.parseInt(value);
            }
            else if(_backingArray instanceof long[])
            {
                ((long[])_backingArray)[placementIndex] = Long.parseLong(value);
            }
            else if(_backingArray instanceof float[])
            {
                ((float[])_backingArray)[placementIndex] = Float.parseFloat(value);
            }
            else if(_backingArray instanceof double[])
            {
                ((double[])_backingArray)[placementIndex] = Double.parseDouble(value);
            }
            else if(_backingArray instanceof String[])
            {
                ((String[])_backingArray)[placementIndex] = value;
            }
            else
            {
                throw new XMLReaderException("Invalid value class, " + value.getClass().getSimpleName());
            }
        }
        catch(final ArrayIndexOutOfBoundsException e)
        {
            throw new XMLReaderException("Unable extract a byte or char entry for a "
                + _backingArray.getClass().getSimpleName() + " from '" + value + "'.");
        }
        catch(final NumberFormatException e)
        {
            throw new XMLReaderException("Unable parse a number for a " + _backingArray.getClass().getSimpleName()
                + " from '" + value + "'.");
        }
    }

    private void putXMLReaderReadValue(final XMLReadable value) throws XMLReaderException
    {
        //I'm using placementIndex, because I wanted to increase _valueIndex here rather than at the bottom
        //of this method where it may be overlooked.
        final int placementIndex = _valueIndex;
        _valueIndex++;

        if(placementIndex >= _arrayLength)
        {
            throw new XMLReaderException("Array item index " + placementIndex
                + " is as long or longer than the array length " + _arrayLength + ".");
        }
        try
        {
            if(value instanceof XMLVariable)
            {
                final XMLVariable xmlVar = (XMLVariable)value;
                if(_backingArray.getClass().getComponentType().isInstance(xmlVar))
                {
                    ((Object[])_backingArray)[placementIndex] = xmlVar;
                }
                else if(_backingArray.getClass().getComponentType().isInstance(xmlVar.get()))
                {
                    ((Object[])_backingArray)[placementIndex] = xmlVar.get();
                }
                else
                {
                    throw new XMLReaderException("Value class " + value.getClass().getSimpleName()
                        + " does not match storage array "
                        + _backingArray.getClass().getComponentType().getSimpleName() + ".");
                }
            }
            else if(value instanceof XMLReadable)
            {
                ((Object[])_backingArray)[placementIndex] = value;
            }
        }
        catch(final ArrayIndexOutOfBoundsException e)
        {
            throw new XMLReaderException("Unable extract a byte or char entry for a "
                + _backingArray.getClass().getSimpleName() + " from '" + value + "'.");
        }
    }

    public Object getStorageObject()
    {
        return _backingArray;
    }

    /**
     * Sets the delimiter assumed when the XML is read.
     * 
     * @param delimiter The delimiter to use.
     */
    public void setDelimiter(final char delimiter)
    {
        _delimiter = delimiter;
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
    {
        super.readInPropertyFromXMLElement(elementName, attr);
        if(elementName.equals(getXMLTagName()))
        {
            _valueIndex = 0;
            _temp = null;
            _tempReader = null;
        }
        else if(elementName.equals(_tagNameForValues))
        {
            try
            {
                //Use a factory if provided.  Otherwise, this will read in the value within the setValue* method,
                //and it will only handle a primitive _backingArray component.
                if(_factory != null)
                {
                    if(_temp != null)
                    {
                        putXMLReaderReadValue(_temp);
                    }
                    _temp = _factory.get();
                    _tempReader = _temp.getReader();
                    return _tempReader;
                }
            }
            catch(final NullPointerException e)
            {
//                e.printStackTrace();
                throw new XMLReaderException("Array [" + getXMLTagName() + "] incorrectly formed.", e);
            }

            //For non-XMLReadables, this must do the parsing.
            return null;
        }
        else
        {
            throw new XMLReaderException("For element '" + getXMLTagName() + "', unrecognized subelement '"
                + elementName + "'.");
        }
        return null;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
        //This is doing the parsing, and this can only handle primitives.  So call the put method.
        if(elementName.equals(getXMLTagName()))
        {
            if(!value.trim().isEmpty())
            {
                final StringTokenizer tokenizer = new StringTokenizer(value, Character.toString(_delimiter));
                while(tokenizer.hasMoreElements())
                {
                    final String oneValue = tokenizer.nextToken();
                    putPrimitiveValue(oneValue);
                }
            }
        }
        else if(elementName.equals(_tagNameForValues))
        {
            putPrimitiveValue(value);
        }
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
        super.finalizeReading();

        //Put the last value if XMLReadables are being read!
        if(_factory != null)
        {
            if(_temp != null)
            {
                putXMLReaderReadValue(_temp);
            }
        }

        if(_valueIndex < _arrayLength)
        {
            throw new XMLReaderException("XML array had " + _valueIndex + " items, but expected " + _arrayLength
                + " items.");
        }
    }

}
