package ohd.hseb.hefs.utils.xml;

import java.text.NumberFormat;

import javax.xml.stream.XMLStreamWriter;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

/**
 * Writer of arrays to XML. {@link #getXMLTagName()} specifies the top level XML tag. The tag for each of the values of
 * the array is {@link #_tagNameForValues}. There is no indexing in the XML, so that the given array is output in order
 * and is expected to be input in order. Furthermore, the XML does not include any array size information. So, whatever
 * will read in the array later must know how many elements to expect. <br>
 * <br>
 * This class handles the following array component types:<br>
 * <br>
 * 1. Primitives byte, chart, short, int, long, float, and double: The XML is created using corresponding
 * {@link XMLVariable} instances.<br>
 * 2. Objects that implement {@link XMLWritable}: The XML is created via the {@link XMLWritable#getWriter()} returned
 * value. The {@link #_tagNameForValues} is ignored; the {@link XMLWriter} used dictates the tag name. <br>
 * 3. Other objects: The XML is created using the object toStrign value.<br>
 * <br>
 * For examples, see {@link GeneralXMLReaderWriterAdapterTest}.<br>
 * <br>
 * Note that if the array contains large amounts of data, it is recommended that the
 * {@link XMLStreamWriterWriter#writeXML(XMLStreamWriter)} method be used for writing, as it is much faster than the
 * {@link XMLWriter#writePropertyToXMLElement(Document)} method particularly for arrays of primitives.
 * 
 * @author Hank.Herr
 */
public class ArrayXMLWriter extends XMLWriterAdapter implements XMLStreamWriterWriter
{

    /**
     * The array to write, stored as an {@link Object}.
     */
    private Object _backingArray = null;

    /**
     * The length of the array to write, used internally.
     */
    private int _arrayLength = -1;

    /**
     * The XML tag to write for values in the array.
     */
    private String _tagNameForValues = null;

    /**
     * If true, then elements are delimited using {@link #_delimiter}, so that each array is listed as a delimited list
     * of array values.
     */
    private boolean _useDelimiter = false;

    /**
     * The delimiter to use. It MUST be a character that has no reasonable chance of being within a value of the array.
     * The default is '|', which should work for numerical arrays, but may not for String-based arrays.
     */
    private char _delimiter = '|';

    /**
     * If not null and if the backing array contains numerical components, short, int, long, float, and double, then
     * this will be used to format the Strings when output.
     */
    private NumberFormat _numberFormatter = null;

    public <E extends XMLWritable> ArrayXMLWriter(final String tag, final E[] values)
    {
        super(tag);
        _tagNameForValues = null;
        _backingArray = values;
        _arrayLength = values.length;
    }

    public ArrayXMLWriter(final String tag, final String valueTagName, final Object[] values)
    {
        super(tag);
        _tagNameForValues = valueTagName;
        _backingArray = values;
        _arrayLength = values.length;
    }

    public ArrayXMLWriter(final String tag, final String valueTagName, final byte[] values)
    {
        super(tag);
        init(valueTagName, values, values.length);
    }

    public ArrayXMLWriter(final String tag, final String valueTagName, final char[] values)
    {
        super(tag);
        init(valueTagName, values, values.length);
    }

    public ArrayXMLWriter(final String tag, final String valueTagName, final short[] values)
    {
        super(tag);
        init(valueTagName, values, values.length);
    }

    public ArrayXMLWriter(final String tag, final String valueTagName, final int[] values)
    {
        super(tag);
        init(valueTagName, values, values.length);
    }

    public ArrayXMLWriter(final String tag, final String valueTagName, final long[] values)
    {
        super(tag);
        init(valueTagName, values, values.length);
    }

    public ArrayXMLWriter(final String tag, final String valueTagName, final float[] values)
    {
        super(tag);
        init(valueTagName, values, values.length);
    }

    public ArrayXMLWriter(final String tag, final String valueTagName, final double[] values)
    {
        super(tag);
        init(valueTagName, values, values.length);
    }

    protected Object getBackingArray()
    {
        return _backingArray;
    }

    /**
     * If not null, the provided formatter will be used to format numerical-based arrays: short, int, long, float,
     * double.
     * 
     * @param format The {@link NumberFormat} to use.
     */
    public void setNumberFormatter(final NumberFormat format)
    {
        _numberFormatter = format;
    }

    /**
     * @param b If true, then array will be written as delimited list of numbers within the value of the wrapping XML
     *            element tag. The attribute {@link #_tagNameForValues} will not be used. This should save space in the
     *            resulting XML, but may force it to take more time to parse. Note that if the array components are
     *            themselves {@link XMLWritable} objects, then the delimiter will be ignored and the XML will be written
     *            as subelements of the top level element.
     */
    public void setUseDelimiter(final boolean b)
    {
        _useDelimiter = b;
    }

    /**
     * @param c The delimiter to use must be a single character.
     */
    public void setDelimiter(final char c)
    {
        _delimiter = c;
    }

    private void init(final String valueTagName, final Object values, final int length)
    {
        _tagNameForValues = valueTagName;
        _backingArray = values;
        _arrayLength = length;
    }

    public String getToString(final int i)
    {
        if(_backingArray instanceof byte[])
        {
            return Byte.toString(((byte[])_backingArray)[i]);
        }
        else if(_backingArray instanceof char[])
        {
            return Byte.toString((byte)((char[])_backingArray)[i]);
        }
        else if(_backingArray instanceof short[])
        {
            if(_numberFormatter != null)
            {
                return _numberFormatter.format(((short[])_backingArray)[i]);
            }
            return Integer.toString(((short[])_backingArray)[i]);
        }
        else if(_backingArray instanceof int[])
        {
            if(_numberFormatter != null)
            {
                return _numberFormatter.format(((int[])_backingArray)[i]);
            }
            return Integer.toString(((int[])_backingArray)[i]);
        }
        else if(_backingArray instanceof long[])
        {
            if(_numberFormatter != null)
            {
                return _numberFormatter.format(((long[])_backingArray)[i]);
            }
            return Long.toString(((long[])_backingArray)[i]);
        }
        else if(_backingArray instanceof float[])
        {
            if((_numberFormatter != null) && (!Float.isNaN(((float[])_backingArray)[i])))
            {
                return _numberFormatter.format(((float[])_backingArray)[i]);
            }
            return Float.toString(((float[])_backingArray)[i]);
        }
        else if(_backingArray instanceof double[])
        {
            if((_numberFormatter != null) && (!Double.isNaN(((double[])_backingArray)[i])))
            {
                return _numberFormatter.format(((double[])_backingArray)[i]);
            }
            return Double.toString(((double[])_backingArray)[i]);
        }
        return ((Object[])_backingArray)[i].toString();
    }

    private String createDelimitedString()
    {
        final StringBuffer result = new StringBuffer();
        for(int i = 0; i < _arrayLength; i++)
        {
            result.append(getToString(i));
            if(i < _arrayLength - 1)
            {
                result.append(_delimiter);
            }
        }
        return result.toString();
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        //This is a relatively slow mechanism for outputting XML.  It is not recommended this method be used.
        final Element mainElt = super.writePropertyToXMLElement(request);

        //Use the delimiter if the _useDelimiter is true and the array components are not XMLWritable.
        final boolean useDelimiter = _useDelimiter
            && (!XMLWritable.class.isAssignableFrom(_backingArray.getClass().getComponentType()));

        if(useDelimiter)
        {
            final Text valueTextNode = request.createTextNode(createDelimitedString());
            mainElt.appendChild(valueTextNode);
        }
        else
        {
            for(int i = 0; i < _arrayLength; i++)
            {
                if(XMLWritable.class.isAssignableFrom(_backingArray.getClass().getComponentType()))
                {
                    final Object o = ((Object[])_backingArray)[i];
                    mainElt.appendChild((((XMLWritable)o).getWriter()).writePropertyToXMLElement(request));
                }
                else
                {
                    final String txt = getToString(i);
                    mainElt.appendChild(XMLTools.createTextNodeElement(request, _tagNameForValues, txt));
                }
            }
        }
        return mainElt;
    }

    @Override
    public void writeXML(final XMLStreamWriter streamWriter) throws Exception
    {
        streamWriter.writeStartElement(getXMLTagName());
        getAttributes().write(streamWriter);

        //Use the delimiter if the _useDelimiter is true and the array components are not XMLWritable.
        final boolean useDelimiter = _useDelimiter
            && (!XMLWritable.class.isAssignableFrom(_backingArray.getClass().getComponentType()));

        if(useDelimiter)
        {
            final String txt = createDelimitedString();
            streamWriter.writeCharacters(txt);
        }
        else
        {
            for(int i = 0; i < _arrayLength; i++)
            {
                if(XMLWritable.class.isAssignableFrom(_backingArray.getClass().getComponentType()))
                {
                    final Object o = ((Object[])_backingArray)[i];
                    final XMLWriter writer = ((XMLWritable)o).getWriter();
                    if(writer instanceof XMLStreamWriterWriter)
                    {
                        ((XMLStreamWriterWriter)writer).writeXML(streamWriter);
                    }
                    else
                    {
                        XMLTools.writeXMLWriterToXMLStreamWriter(writer, streamWriter);
                    }
                }
                else
                {
                    final String txt = getToString(i);
                    streamWriter.writeStartElement(_tagNameForValues);
                    streamWriter.writeCharacters(txt);
                    streamWriter.writeEndElement();
                }
            }
        }
        streamWriter.writeEndElement();
    }
}
