package ohd.hseb.hefs.utils.xml;

import javax.xml.stream.XMLStreamWriter;

import org.w3c.dom.Element;

import com.google.common.base.Objects;

/**
 * Store a single XML attribute using one of the XMLVariable implementations. Two {@link Attribute} instances are equal
 * if their {@link #_name} is identical. It is not recommended you use this class directly. Instead, add
 * {@link Attribute} instances to an {@link AttributeList} using its
 * {@link AttributeList#addAttribute(String, Class, boolean)} method.<br>
 * <br>
 * The {@link #equals(Object)} should only be used to determine if two Attribute instances are for the same attribute;
 * e.g., in searching lists to determine if an attribute is already present in the list. Use the equals method on the
 * return of {@link #getStorageObject()} to determine if the values of the attributes are equal.
 * 
 * @author hank.herr
 */
public class Attribute<T> implements Cloneable
{
    private final String _name;

    private XMLVariable<T> _value;

    private final boolean _required;

    /**
     * @param name The name of the attribute.
     * @param storageClass The class of the storage object used. This must have an empty constructor.
     * @param required True if the attribute is a required one, false otherwise.
     */
    public Attribute(final String name, final Class<? extends XMLVariable<T>> storageClass, final boolean required)
    {
        _name = name;
        try
        {
            _value = (XMLVariable<T>)storageClass.getDeclaredConstructor().newInstance();                     
        }
        catch(final Exception e)
        {
            e.printStackTrace();
        }
        _required = required;
    }

    public Attribute(final String name, final XMLVariable<T> storageObject, final boolean required)
    {
        _name = name;
        _value = storageObject;
        _required = required;
    }

    public XMLVariable<T> getStorageObject()
    {
        return _value;
    }

    public boolean isFor(final String name)
    {
        return (_name.equals(name));
    }

    public boolean isSpecified()
    {
        return _value.get() != null;
    }

    public boolean isRequired()
    {
        return _required;
    }

    public void read(final String text) throws XMLReaderException
    {
        _value.setValueOfElement(_value.getXMLTagName(), text);
    }

    public void checkIfRequiredAndSet() throws XMLReaderException
    {
        if(isRequired() && (_value.get() == null))
        {
            throw new XMLReaderException("Required attribute, " + _name + ", is not set.");
        }
    }

    public void write(final Element xmlElement)
    {
        if(isSpecified() || isRequired())
        {
            if(_value.get() == null)
            {
                xmlElement.setAttribute(_name, null);
            }
            else
            {
                xmlElement.setAttribute(_name, _value.writeVariable(_value.get()));
            }
        }
    }

    /**
     * @param writer the {@link XMLStreamWriter} to which to write the attribute.
     * @throws Exception
     */
    public void write(final XMLStreamWriter writer) throws Exception
    {
        writer.writeAttribute(_name, _value.writeVariable(_value.get()));
    }

    @Override
    public int hashCode()
    {
        return _name.hashCode();
    }

    @Override
    public boolean equals(final Object obj)
    {
        if(this == obj)
        {
            return true;
        }

        if(!(obj instanceof Attribute<?>))
        {
            return false;
        }

        final Attribute<?> that = (Attribute<?>)obj;
        return Objects.equal(_name, that._name);
    }

    @Override
    public String toString()
    {
        return "Attribute: " + this._name + " = " + this._value.toString();
    }

    @Override
    public Attribute clone()
    {
        return new Attribute<T>(_name, _value.clone(), _required);
    }

}
