package ohd.hseb.hefs.utils.xml;

import java.util.ArrayList;

import javax.xml.XMLConstants;
import javax.xml.stream.XMLStreamWriter;

import ohd.hseb.hefs.utils.tools.ListTools;

import org.w3c.dom.Element;
import org.xml.sax.Attributes;

/**
 * Maintains a list of {@link Attribute} objects. The list can only be added to, not removed from as there does not
 * appear to be any need for removing. Recommended usage is to create an instance and then call the
 * {@link #addAttribute(String, Class, boolean)} method, recording the returned {@link XMLVariable}. The returned value
 * stores the attribute and is updated as the underlying attributes change (e.g., reading from an XML file).
 * 
 * @author hank.herr
 */
@SuppressWarnings("serial")
public class AttributeList extends ArrayList<Attribute> implements Cloneable
{
    public AttributeList()
    {
        super();
    }

    /**
     * The returned object is updated as the underlying attribute is updated; it is the exact object used to store the
     * attribute value.
     * 
     * @param name Name of the attribute
     * @param storageClass Storage class to use, which must extend {@link XMLVariable}.
     * @param required Indicates if the attribute must be specified.
     * @return The storage object used within the {@link Attribute} class.
     */
    @SuppressWarnings("unchecked")
    public <T extends XMLVariable> T addAttribute(final String name, final Class<T> storageClass, final boolean required)
    {
        @SuppressWarnings({"rawtypes"})
        final Attribute attr = new Attribute(name, storageClass, required);
        ListTools.addItemIfNotAlreadyInList(this, attr);
        return (T)retrieve(name).getStorageObject();
    }

    /**
     * Creates an attribute that uses the assigned storage object. Useful if you have an object already created and want
     * to assign it to this list.
     * 
     * @param name Name of the attribute
     * @param storageObject The object to use to store the attribute.
     * @param required Indicates if the attribute must be specified.
     */
    public <T extends XMLVariable> void addAttribute(final String name, final T storageObject, final boolean required)
    {
        @SuppressWarnings({"unchecked", "rawtypes"})
        final Attribute attr = new Attribute(name, storageObject, required);
        ListTools.addItemIfNotAlreadyInList(this, attr);
    }

    /**
     * @param name Name of {@link Attribute} to retrieve.
     * @return The Attribute with the given name or null if none is found.
     */
    public Attribute retrieve(final String name)
    {
        for(final Attribute attr: this)
        {
            if(attr.isFor(name))
            {
                return attr;
            }
        }
        return null;
    }

    /**
     * Given the XML {@link Attributes}, it reads them in and stores them.
     * 
     * @param attributes Attributes to read in.
     * @throws XMLReaderException If an attribute is found whose name is not recognized. If the unrecognized attribute
     *             starts with a namespace prefix, xmlns or xsi, or 'version', then no exception is thrown.
     */
    public void read(final Attributes attributes) throws XMLReaderException
    {
        for(int i = 0; i < attributes.getLength(); i++)
        {
            final String name = attributes.getQName(i);
            final Attribute attr = retrieve(name);

            if(attr == null)
            {
                //This is a fail-safe measure.  Theoretically, the GenericXMLReadingHandler startElement method
                //should never see any attributes dealing with namespaces.  However, for the *config.xml files used
                //for version info, it is seeing the namespace attributes.  Eventually that calls this method which
                //used to throw an exception because the name space attributes were not in the list.
                //
                //This if check does not throw exceptions for unfound attributes if they start with "xmlns" or "xsi",
                //two standard name space prefixes.
                if((!name.startsWith(XMLConstants.XMLNS_ATTRIBUTE)) && (!name.startsWith("xsi"))
                    && (!name.startsWith("version")))
                {
                    throw new XMLReaderException("Attribute with name '" + attributes.getQName(i)
                        + "' is not recognized.");
                }
            }
            else
            {
                attr.read(attributes.getValue(i));
            }
        }
    }

    /**
     * @throws XMLReaderException If one of the required attributes is not set (i.e., is null).
     */
    public void checkIfAllRequiredSet() throws XMLReaderException
    {
        for(final Attribute attr: this)
        {
            attr.checkIfRequiredAndSet();
        }
    }

    public void finalizeReading() throws XMLReaderException
    {
        for(final Attribute attr: this)
        {
            attr.getStorageObject().finalizeReading();
        }
    }

    /**
     * @param xmlElement Appends all attributes to the given {@link Element}.
     */
    public void write(final Element xmlElement)
    {
        for(final Attribute attr: this)
        {
            attr.write(xmlElement);
        }
    }

    /**
     * @param writer the {@link XMLStreamWriter} to which to write the attribute.
     * @throws Exception
     */
    public void write(final XMLStreamWriter writer) throws Exception
    {
        for(final Attribute attr: this)
        {
            attr.write(writer);
        }
    }

    /**
     * Clears the attribute values by setting the values within the storage objects to null. The contents of the list
     * are the same, but the values stored within those elements are nulled.
     */
    @SuppressWarnings("unchecked")
    public void clearValues()
    {
        for(final Attribute attr: this)
        {
            attr.getStorageObject().set(null);
        }
    }

    @Override
    public boolean equals(final Object other)
    {
        if(this == other)
        {
            return true;
        }

        if(!(other instanceof AttributeList))
        {
            return false;
        }

        final AttributeList otherList = (AttributeList)other;
        if(otherList.size() != size())
        {
            return false;
        }

        //The code below is brute force.  I don't feel like writing a one-sided check for AttributeList
        //(to be called twice) at this time because I don't see any other purpose for it than in this method.
        //One side -- order does not matter!
        for(final Attribute a: this)
        {
            final int index = otherList.indexOf(a);
            if(index < 0)
            {
                return false;
            }
            if(!a.getStorageObject().equals(otherList.get(index).getStorageObject()))
            {
                return false;
            }
        }

        //Other side
        for(final Attribute a: otherList)
        {
            final int index = this.indexOf(a);
            if(index < 0)
            {
                return false;
            }
            if(!a.getStorageObject().equals(this.get(index).getStorageObject()))
            {
                return false;
            }
        }

        return true;
    }

    @Override
    public AttributeList clone()
    {
        final AttributeList results = new AttributeList();
        for(final Attribute attr: this)
        {
            results.add(attr.clone());
        }
        return results;
    }

}
