package ohd.hseb.hefs.utils.xml;

import java.util.Collection;
import java.util.List;

import org.xml.sax.Attributes;

import com.google.common.collect.Lists;

/**
 * Takes an {@link XMLReaderFactory} and reads in a series of xml elements, creating a new factory for each one. At the
 * end it adds all the read items to the passed in collection.<br/>
 * For a similar but more generic class, one that allows the caller to specify a callback, see {@link ListXMLReader}.
 * 
 * @author alexander.garbarino
 * @param <E> The {@link XMLReadable} elements in the collection.
 */
public class CollectionXMLReader<E extends XMLReadable> extends XMLReaderAdapter
{
    private final Collection<E> _targetCollection;
    private final XMLReaderFactory<E> _factory;

    private List<E> _tempList = null;
    private E _temp = null;
    private XMLReader _tempReader = null;

    public CollectionXMLReader(final String tagName, final Collection<E> collection, final XMLReaderFactory<E> factory)
    {
        super(tagName);

        _targetCollection = collection;
        _factory = factory;
    }

    public static <E extends XMLReadable> CollectionXMLReader<E> create(final String tagName,
                                                                        final Collection<E> collection,
                                                                        final XMLReaderFactory<E> factory)
    {
        return new CollectionXMLReader<E>(tagName, collection, factory);
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
    {
        super.readInPropertyFromXMLElement(elementName, attr);
        if(elementName.equals(getXMLTagName()))
        {
            _targetCollection.clear();
            _tempList = Lists.newArrayList();
            _temp = _factory.get();
            _tempReader = _temp.getReader();
        }
        try
        {
            if(_tempReader.getXMLTagName().equals(elementName))
            {
                _tempList.add(_temp);
                final XMLReader reader = _tempReader;
                _temp = _factory.get();
                _tempReader = _temp.getReader();
                return reader;
            }
        }
        catch(final NullPointerException e)
        {
            throw new XMLReaderException("Collection [" + getXMLTagName() + "] incorrectly formed.", e);
        }
        return null;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
        super.finalizeReading();
        if(_tempList != null)
        {
            _targetCollection.addAll(_tempList);
        }
    }

}
