package ohd.hseb.hefs.utils.xml;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.utils.adapter.StateMetaInformation;

import org.xml.sax.Attributes;

import com.google.common.collect.Lists;

/**
 * Read a series of {@link XMLReader} instances as provided {@link XMLReadable} instances. A flag, {@link #_readInOrder}
 * , indicates if the components must be read in order. If unordered, then all components must have unique tag names,
 * otherwise only the first component will be read. For ordered components, the elements found must match the ordered
 * tag names.<br>
 * <br>
 * Note that it is possible for this reader to contain a growing list of elements. For example, see
 * {@link StateMetaInformation}.<br>
 * <br>
 * Note that the {@link #validate()} calls the same method within each reader within {@link #_componentReaders}.
 * However, the {@link #finalizeReading()} does nothing by default, since the component readers should have already been
 * finalized by the time the method here is called.
 * 
 * @author alexander.garbarino
 * @author hank.herr
 */
public class CompositeXMLReader extends XMLReaderAdapter
{
    private final List<XMLReadable> _components = new ArrayList<XMLReadable>();

    /**
     * Note that it is possible for elements in this list to be null (see NullControlOption class in HEFS).
     */
    private final List<XMLReader> _componentReaders = new ArrayList<XMLReader>();

    private boolean _readInOrder = false;

    private int _workingIndex = 0;

    public CompositeXMLReader(final String tag, final Iterable<? extends XMLReadable> components)
    {
        this(tag, false, Lists.newArrayList(components));
    }

    public CompositeXMLReader(final String tag, final XMLReadable... components)
    {
        this(tag, false, Lists.newArrayList(components));
    }

    public CompositeXMLReader(final String tag,
                              final boolean readInOrder,
                              final Iterable<? extends XMLReadable> components)
    {
        super(tag);
        _readInOrder = readInOrder;
        addComponents(components);
    }

    public CompositeXMLReader(final String tag, final boolean readInOrder, final XMLReadable... components)
    {
        this(tag, readInOrder, Lists.newArrayList(components));
    }

    public boolean isReadInOrder()
    {
        return _readInOrder;
    }

    public void setReadInOrder(final boolean readInOrder)
    {
        _readInOrder = readInOrder;
    }

    public void addComponent(final XMLReadable component)
    {
        _components.add(component);
        _componentReaders.add(component.getReader());
    }

    public void removeComponent(final int index)
    {
        this._components.remove(index);
        this._componentReaders.remove(index);
    }

    public void insertComponent(final int index, final XMLReadable component)
    {
        _components.add(index, component);
        _componentReaders.add(index, component.getReader());
    }

    public void addComponents(final Iterable<? extends XMLReadable> components)
    {
        for(final XMLReadable comp: components)
        {
            addComponent(comp);
        }
    }

    public void insertComponents(final int index, final Iterable<? extends XMLReadable> components)
    {
        int workingIndex = index;
        for(final XMLReadable comp: components)
        {
            insertComponent(workingIndex, comp);
            workingIndex++;
        }
    }

    public void addComponents(final XMLReadable... components)
    {
        addComponents(Lists.newArrayList(components));
    }

    public void insertComponents(final int index, final XMLReadable... components)
    {
        insertComponents(index, Lists.newArrayList(components));
    }

    public List<XMLReadable> getComponents()
    {
        return _components;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
    {
        super.readInPropertyFromXMLElement(elementName, attr);

        //Unordered list.
        if(!_readInOrder)
        {
            for(final XMLReader reader: _componentReaders)
            {
                if(reader != null && reader.getXMLTagName().equals(elementName))
                {
                    return reader;
                }
            }
        }
        //Ordered list.
        else
        {
            if(elementName.equals(this.getXMLTagName()))
            {
                _workingIndex = 0;
            }
            else if(_workingIndex >= _componentReaders.size())
            {
                throw new XMLReaderException("For ordered composite XML reader with tag '" + getXMLTagName()
                    + "', there should be no more compoennts, but one has be found with tag '" + elementName + "'.");
            }
            else if(elementName.equals(_componentReaders.get(_workingIndex).getXMLTagName()))
            {
                _workingIndex++;
                return _componentReaders.get(_workingIndex - 1);
            }
            else
            {
                throw new XMLReaderException("For ordered composite XML reader with tag '" + getXMLTagName()
                    + "', element read has name '" + elementName + "', but '"
                    + _componentReaders.get(_workingIndex).getXMLTagName() + "' is expected next.");
            }
        }
        return null;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
        //Do NOT finalize the component readers.  That may result in finalization being called multiple times.
    }

    @Override
    public void validate() throws XMLReaderException
    {
        for(final XMLReader reader: _componentReaders)
        {
            //Noted above, readers can be null (see HEFS NullControlOption).
            if(reader != null)
            {
                reader.validate();
            }
        }
    }
}
