package ohd.hseb.hefs.utils.xml;

import java.util.ArrayList;
import java.util.List;

import javax.xml.stream.XMLStreamWriter;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.google.common.collect.Lists;

/**
 * Write a series of xml writers as provided by {@link XMLWritable} instances.
 * 
 * @author alexander.garbarino
 */
public class CompositeXMLWriter extends XMLWriterAdapter implements XMLStreamWriterWriter
{
    private final List<XMLWritable> _components = new ArrayList<XMLWritable>();
    private final List<XMLWriter> _componentWriters = new ArrayList<XMLWriter>();

    public CompositeXMLWriter(XMLWritable... components)
    {
        this(Lists.newArrayList(components));
    }

    public CompositeXMLWriter(String tag, XMLWritable... components)
    {
        this(tag, Lists.newArrayList(components));
    }

    public CompositeXMLWriter(Iterable<? extends XMLWritable> components)
    {
        this(null, components);
    }

    public CompositeXMLWriter(String tag, Iterable<? extends XMLWritable> components)
    {
        super(tag);
        addComponents(components);
    }

    public void addComponent(XMLWritable component)
    {
        _components.add(component);
        _componentWriters.add(component.getWriter());
    }

    public void insertComponent(int index, XMLWritable component)
    {
        _components.add(index, component);
        _componentWriters.add(index, component.getWriter());
    }

    public void addComponents(Iterable<? extends XMLWritable> components)
    {
        for(XMLWritable comp: components)
        {
            addComponent(comp);
        }
    }

    public void insertComponents(int index, Iterable<? extends XMLWritable> components)
    {
        int workingIndex = index;
        for(XMLWritable comp: components)
        {
            insertComponent(workingIndex, comp);
            workingIndex++;
        }
    }

    public void addComponents(XMLWritable... components)
    {
        addComponents(Lists.newArrayList(components));
    }

    public void insertComponents(int index, XMLWritable... components)
    {
        insertComponents(index, Lists.newArrayList(components));
    }

    public void removeComponent(int index)
    {
        this._components.remove(index);
        this._componentWriters.remove(index);
    }

    public void addComponentsTo(CompositeXMLWriter target)
    {
        target.addComponents(_components);
    }

    public List<XMLWritable> getComponents()
    {
        return _components;
    }

    @Override
    public Element writePropertyToXMLElement(Document request) throws XMLWriterException
    {
        Element elt = super.writePropertyToXMLElement(request);
        return addToElement(request, elt);
    }

    /**
     * Appends this writer's components to the specified element.
     * 
     * @param request
     * @param element the element to append to
     * @return the same element
     * @throws XMLWriterException
     */
    public Element addToElement(Document request, Element element) throws XMLWriterException
    {
        for(XMLWriter component: _componentWriters)
        {
            if(component != null)
            {
                Element subElement = component.writePropertyToXMLElement(request);
                if(subElement != null)
                {
                    element.appendChild(subElement);
                }
            }
        }
        return element;
    }

    @Override
    public XMLWriter getWriter()
    {
        return this;
    }

    /**
     * Writes the XML specified by this element to a stream via an {@link XMLStreamWriter}. The method is recursive: if
     * a component specified is another {@link CompositeXMLWriter} instance, then its
     * {@link CompositeXMLWriter#writeXML(XMLStreamWriter)} method will be called. However, if the component is not such
     * an instance, a {@link Document} will be written containing the {@link XMLWriter} generated element and that will
     * be written to the file via {@link XMLStreamWriter}. <br>
     * <br>
     * The gain of this is memory usage: in order to write out via a {@link Document}, the entire XML is stored in
     * memory and then dumped. Now, if {@link CompositeXMLWriter} is used, only portions that are not
     * {@link CompositeXMLWriter} must be dumped. For example, {@link TableXMLWriter} uses a {@link CompositeXMLWriter}
     * wrapping {@link ArrayXMLWriter} instances. To write a table, only the data for one {@link ArrayXMLWriter} must be
     * held in memory at a time.
     * 
     * @param writer
     * @throws Exception
     */
    @Override
    public void writeXML(XMLStreamWriter writer) throws Exception
    {
        writer.writeStartElement(getXMLTagName());
        writeAttributes(writer);
        for(XMLWriter componentWriter: _componentWriters)
        {
            //Skip null writers... they represent parameters that should not be output but 
            //are part of the writer object for other reasons.
            if(componentWriter == null)
            {
                continue;
            }
            else if(componentWriter instanceof XMLStreamWriterWriter)
            {
                ((XMLStreamWriterWriter)componentWriter).writeXML(writer);
            }
            else
            {
                XMLTools.writeXMLWriterToXMLStreamWriter(componentWriter, writer);
            }
        }
        writer.writeEndElement();
        writer.flush();
    }
}
