/*
 * Created on Mar 19, 2008 To change the template for this generated file go to Window&gt;Preferences&gt;Java&gt;Code
 * Generation&gt;Code and Comments
 */
package ohd.hseb.hefs.utils.xml;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.GZIPInputStream;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import ohd.hseb.hefs.utils.tools.StreamTools;

import org.jvnet.fastinfoset.FastInfosetException;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;
import org.xml.sax.helpers.DefaultHandler;

import com.sun.xml.fastinfoset.sax.SAXDocumentParser;

/**
 * A generic DefaultHandler based XML handling mechanism. It extends DefaultHandler and provides some basic methods,
 * some of which must be overridden by any subclass.<br>
 * <br>
 * This classes uses the XMLReader structure. Such a reader looks at the XML and either parses it itself or returns a
 * nested XMLReader that can handle it. The possibility of nested readers leads to the creation of a
 * _workingReaderStack, the top element of which is the XMLReader currently being used.
 * 
 * @author hank
 */
public class GenericXMLReadingHandler extends DefaultHandler
{

    private final List<XMLReader> _workingReaderStack = new ArrayList<XMLReader>();

    /**
     * The highest level XMLReader that will be used.
     */
    private XMLReader _topXMLReader = null;

    /**
     * If the reading proces fails, this String must be populated with an appropriate message so that the
     * readXMLFromFile will know of the failure!
     */
    protected String _xmlFailureMessage = "";

    private String _currentValue = "";

    /**
     * @param topLevelXMLReader The top-level reader to be used in parsing the XML.
     */
    public GenericXMLReadingHandler(final XMLReader topLevelXMLReader)
    {
        this._topXMLReader = topLevelXMLReader;
    }

    /**
     * Call this to set the XML failure message.
     * 
     * @param message
     */
    public void setXMLFailureMessage(final String message)
    {
        _xmlFailureMessage = message;
    }

    /**
     * Call this to append to the existing XML failure message.
     * 
     * @param message
     */
    public void appendToXMLFailureMessage(final String message)
    {
        _xmlFailureMessage += message;
    }

    /**
     * Read XML from a string. This leads directly to calls to the startElement, endElement, and characters methods.
     * 
     * @param input The XML string to parse, which must NOT be fastInfoset or encrypted.
     * @throws GenericXMLReadingHandlerException if problems occur.
     */
    public void readXMLFromString(final String input) throws GenericXMLReadingHandlerException
    {
        if((input == null) || (input.length() == 0))
        {
            throw new GenericXMLReadingHandlerException("XML is null or zero-length.");
        }
        //Try to load into the working copy.
        try
        {
            _xmlFailureMessage = "";
            final SAXParserFactory factory = SAXParserFactory.newInstance();
            final SAXParser parser = factory.newSAXParser();

            final ByteArrayInputStream inputStream = new ByteArrayInputStream(input.getBytes());
            parser.parse(inputStream, this);
            if(_xmlFailureMessage.length() > 0)
            {
                throw new GenericXMLReadingHandlerException("Failed to read XML string: " + _xmlFailureMessage);
            }
        }
        catch(final ParserConfigurationException e)
        {
            throw new GenericXMLReadingHandlerException("Unable to configure XML parser: " + e.getMessage());
        }
        catch(final SAXException e)
        {
            throw new GenericXMLReadingHandlerException("Unable to build SAX parser: " + e.getMessage());
        }
        catch(final IOException e)
        {
            throw new GenericXMLReadingHandlerException("Unable to process XML string: " + e.getMessage());
        }
    }

    /**
     * Reads XML from the provided FastInfoset byte array.
     * 
     * @param input Bytes that are FastInfoset and must be parsed.
     * @throws GenericXMLReadingHandlerException
     */
    public void readXMLFromFISByteArray(final byte[] input) throws GenericXMLReadingHandlerException
    {
        if((input == null) || (input.length == 0))
        {
            throw new GenericXMLReadingHandlerException("Byte array is null or zero-length.");
        }
        //Try to load into the working copy.
        try
        {
            _xmlFailureMessage = "";
            final ByteArrayInputStream inputStream = new ByteArrayInputStream(input);
            final SAXDocumentParser reader = new SAXDocumentParser();
            reader.setContentHandler(this);
            reader.parse(inputStream);
            if(_xmlFailureMessage.length() > 0)
            {
                throw new GenericXMLReadingHandlerException("Failed to read XML string: " + _xmlFailureMessage);
            }
        }
        catch(final FastInfosetException e)
        {
            throw new GenericXMLReadingHandlerException("Unable to process FastInfoset byte[]: " + e.getMessage());
        }
        catch(final IOException e)
        {
            throw new GenericXMLReadingHandlerException("Unable to process FastInfoset byte[]: " + e.getMessage());
        }
        catch(final SAXException e)
        {
            throw new GenericXMLReadingHandlerException("Unable to process FastInfoset byte[]: " + e.getMessage());
        }
    }

    /**
     * This uses {@link XMLTools#isFastInfosetFile(String)} to determine if the resource file is a binary FastInfoset
     * and, if so, calls the appropriate methods. Otherwise, it is assumed to be a regular ASCII XML resource. The
     * resource is loaded via {@link ClassLoader#getSystemResourceAsStream(String)}.
     * 
     * @param resourceName Name of the resource to be read.
     * @throws GenericXMLReadingHandlerException If the {@link XMLReader} has a problem parsing the XML.
     */
    public void readXMLFromResource(final String resourceName) throws GenericXMLReadingHandlerException
    {
//        final InputStream xmlFileStream = ClassLoader.getSystemResourceAsStream(resourceName);
        final InputStream xmlFileStream = this.getClass().getClassLoader().getResourceAsStream(resourceName);
        if(xmlFileStream == null)
        {
            throw new GenericXMLReadingHandlerException("Unable to find resource with name '" + resourceName + "'.");
        }

        //Try to load into the working copy.
        _xmlFailureMessage = "";
        if(XMLTools.isFastInfosetFile(resourceName))
        {
            readXMLFromStreamAndClose(xmlFileStream, true);
        }
        else
        {
            readXMLFromStreamAndClose(xmlFileStream, false);
        }

    }

    /**
     * Calls {@link #readXMLFromFile(File, String)} passing in null for the encyrption key phrase.
     */
    public void readXMLFromFile(final File file) throws GenericXMLReadingHandlerException
    {
        readXMLFromFile(file, null);
    }

    /**
     * This uses {@link XMLTools#isFastInfosetFile(File)} to determine if the file is a binary FastInfoset and, if so,
     * calls the appropriate methods. Otherwise, it is assumed to be a regular ASCII XML file. Calls
     * {@link #readXMLFromStreamAndClose(InputStream, boolean)} once the input stream is constructed.
     * 
     * @param file File to read.
     * @param encryptionKeyPhrase Encryption key phrase to use (must match how it was written).
     * @throws GenericXMLReadingHandlerException If the {@link XMLReader} has a problem parsing the XML.
     */
    public void readXMLFromFile(final File file, final String encryptionKeyPhrase) throws GenericXMLReadingHandlerException
    {
        if(!file.exists())
        {
            throw new GenericXMLReadingHandlerException("File " + file.getAbsolutePath() + " does not exist.");
        }
        if(!file.canRead())
        {
            throw new GenericXMLReadingHandlerException("File " + file.getAbsolutePath()
                + " cannot be read (check permissions).");
        }

        //Try to load into the working copy.
        try
        {
            InputStream inputStream = new FileInputStream(file);

            if(XMLTools.isGZIPFile(file)) //file may be a gzip file.
            {
                inputStream = new GZIPInputStream(inputStream);
            }
            _xmlFailureMessage = "";
            if(XMLTools.isFastInfosetFile(file))
            {
                readXMLFromStreamAndClose(inputStream, true, encryptionKeyPhrase);
            }
            else
            {
                readXMLFromStreamAndClose(inputStream, false, encryptionKeyPhrase);
            }
        }
        catch(final IOException e)
        {
            throw new GenericXMLReadingHandlerException("Unable to process XML string: " + e.getMessage());
        }
    }

    /**
     * Calls {@link #readXMLFromStreamAndClose(InputStream, boolean, String)} passing in null for the encryption key
     * phrase.
     */
    public void readXMLFromStreamAndClose(final InputStream inputStream, final boolean fastInfoset) throws GenericXMLReadingHandlerException
    {
        readXMLFromStreamAndClose(inputStream, fastInfoset, null);
    }

    /**
     * This closes the stream after it is done reading!!! NOTE: Experimentation shows that the {@link SAXParser} which
     * is used for standard XML reading may close the stream upon completion.<br>
     * <br>
     * Calls {@link #readXMLFromStream(InputStream, boolean, boolean, String)} passing in true for the close stream
     * flag.
     */
    public void readXMLFromStreamAndClose(final InputStream inputStream,
                                          final boolean fastInfoset,
                                          final String encryptionKeyPhrase) throws GenericXMLReadingHandlerException
    {
        readXMLFromStream(inputStream, fastInfoset, true, encryptionKeyPhrase);
    }

    /**
     * Calls {@link #readXMLFromStream(InputStream, boolean, boolean)} passing in false for the close stream flag so
     * that the stream is not closed after reading.
     */
    public void readXMLFromStream(final InputStream inputStream, final boolean fastInfoset) throws GenericXMLReadingHandlerException
    {
        readXMLFromStream(inputStream, fastInfoset, false);
    }

    /**
     * This closes the stream after it is done reading!!!<br>
     * <br>
     * Calls {@link #readXMLFromStream(InputStream, boolean, boolean, String)} passing in null for the encryption key
     * phrase.
     */
    public void readXMLFromStream(final InputStream inputStream, final boolean fastInfoset, final boolean closeStream) throws GenericXMLReadingHandlerException
    {
        readXMLFromStream(inputStream, fastInfoset, closeStream, null);
    }

    /**
     * This closes the stream after it is done reading!!!
     * 
     * @param inputStream InputStream to read, which can be either ASCII XML or binary FastInfoset (based on next flag).
     * @param fastInfoset True if stream represents that output by fastInfoset.
     * @param closeStream If true, the stream is closed when reading is done.
     * @param encryptionKeyPhrase Encryption key phrase to use for reading (must match how it was written).
     * @throws GenericXMLReadingHandlerException If the XMLReader has a problem parsing the XML.
     */
    public void readXMLFromStream(InputStream inputStream,
                                  final boolean fastInfoset,
                                  final boolean closeStream,
                                  final String encryptionKeyPhrase) throws GenericXMLReadingHandlerException
    {
        inputStream = StreamTools.encryptInputStream(inputStream, encryptionKeyPhrase);

        //Try to load into the working copy.
        try
        {
            _xmlFailureMessage = "";
            if(fastInfoset)
            {
                final SAXDocumentParser reader = new SAXDocumentParser();
                reader.setContentHandler(this);
                reader.parse(inputStream);
            }
            else
            {
//This stuff below also appears to work, but I'm not sure if its any better, so I'll leave it commented.
//                org.xml.sax.XMLReader parser = org.xml.sax.helpers.XMLReaderFactory.createXMLReader();
//                parser.setContentHandler(this);
//                parser.parse(new InputSource(inputStream));
                final SAXParserFactory factory = SAXParserFactory.newInstance();
                final SAXParser parser = factory.newSAXParser();
                parser.parse(inputStream, this); //This closes the stream when done... not sure why.
            }
            if(_xmlFailureMessage.length() > 0)
            {
                throw new GenericXMLReadingHandlerException("Failed to read XML string: " + _xmlFailureMessage);
            }
        }
        catch(final ParserConfigurationException e)
        {
            e.printStackTrace();
            throw new GenericXMLReadingHandlerException("Unable to configure XML parser: " + e.getMessage(), e);
        }
        catch(final SAXException e)
        {
            e.printStackTrace();
            throw new GenericXMLReadingHandlerException("Unable to build SAX parser: " + e.getMessage(), e);
        }
        catch(final IOException e)
        {
            e.printStackTrace();
            throw new GenericXMLReadingHandlerException("Unable to process XML string: " + e.getMessage(), e);
        }
        catch(final FastInfosetException e)
        {
            e.printStackTrace();
            throw new GenericXMLReadingHandlerException("Unable to parse FastInfoset file: " + e.getMessage(), e);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            throw new GenericXMLReadingHandlerException("Unable to parse XML: " + e.getMessage(), e);
        }
        finally
        {
            if(closeStream)
            {
                try
                {
                    inputStream.close();
                }
                catch(final Exception e)
                {
                }
            }
        }

    }

    /**
     * @return Returns the reader that is currently on the top of the stack; i.e., the one that should be doing the
     *         reading.
     */
    public XMLReader getXMLReaderOnTopOfStack()
    {
        if(_workingReaderStack.size() == 0)
        {
            return null;
        }
        return this._workingReaderStack.get(this._workingReaderStack.size() - 1);
    }

    public XMLReader getTopXMLReader()
    {
        return this._topXMLReader;
    }

    private void pushXMLReaderOntoStack(final XMLReader reader)
    {
        _workingReaderStack.add(reader);
    }

    private void removeXMLReaderFromTopOfStack()
    {
        _workingReaderStack.remove(this._workingReaderStack.size() - 1);
    }

    private int getNumberOfReadersOnStack()
    {
        return this._workingReaderStack.size();
    }

    /**
     * Override if element values should be processed by something before being used. For example, if properties need to
     * be replaced.
     * 
     * @return Processed text value. By default, this returns the text it is given.
     */
    protected String preprocessElementValue(final String text) throws Exception
    {
        return text;
    }

    /**
     * Override if attribute values should be processed by something before being used. For example, if properties need
     * to be replaced.
     * 
     * @return Processed text value. By default, this returns the text it is given.
     */
    protected String preprocessAttributeValue(final String text) throws Exception
    {
        return text;
    }

    @Override
    public void startElement(final String namespaceURI,
                             final String localName,
                             final String qName,
                             final Attributes attributes)
    {
        _currentValue = "";

        //We use a copy contained in an AttributesImpl because this includes some methods, like remove, which may
        //be useful.  I don't know if this slows it down significantly.
        final AttributesImpl workingAttributes = new AttributesImpl(attributes);
        for(int i = 0; i < workingAttributes.getLength(); i++)
        {
            try
            {
                workingAttributes.setValue(i, preprocessAttributeValue(workingAttributes.getValue(i)));
            }
            catch(final Exception e)
            {
                this.appendToXMLFailureMessage("Unable to pre-process XML attribute " + workingAttributes.getQName(i)
                    + " with value " + workingAttributes.getValue(i) + ": " + e.getMessage() + "\n");
            }
        }

        //Set the working reader if needed.  This only works if no sub element of an element has the same
        //tag name as the element... which should obviously never happen.
        if(getNumberOfReadersOnStack() == 0)
        {
            if(this._topXMLReader.getXMLTagName().equals(qName))
            {
                this._workingReaderStack.add(_topXMLReader);
            }
            else
            {
                //This error will occur whenever the first element in the XML is not the _topXMLReader
                //xml tag.  It can also happen if you have extra XML after the _topXMLReader tagged section that
                //is not also for a _topXMLReader.  So, this requires that a the XML to read contains an initial
                //tag for the _topXMLReader and a closing tag for that same section with nothing else after!
                this.appendToXMLFailureMessage("Unable to process XML element '" + qName
                    + "':  element not recognized.\n");
                throw new IllegalArgumentException("Unable to process XML element " + qName
                    + ": element is not recognized.\n");
            }
        }

        //Call the working reader read method and deal with nested readers as needed.
        try
        {
            if(this.getNumberOfReadersOnStack() > 0)
            {
                XMLReader newReader = this.getXMLReaderOnTopOfStack().readInPropertyFromXMLElement(qName,
                                                                                                   workingAttributes);

                //Handle nesting in this loop
                while(newReader != null)
                {
                    this.pushXMLReaderOntoStack(newReader);
                    newReader = this.getXMLReaderOnTopOfStack().readInPropertyFromXMLElement(qName, workingAttributes);
                }
            }
        }
        catch(final XMLReaderException ple)
        {
            this.appendToXMLFailureMessage("Unable to process XML element '" + qName + "': " + ple.getMessage() + "\n");
        }
    }

    @Override
    public void endElement(final String namespaceUri, final String localName, final String qName)
    {
        if(this.getNumberOfReadersOnStack() > 0)
        {
            try
            {
                getXMLReaderOnTopOfStack().setValueOfElement(qName, preprocessElementValue(_currentValue));
                _currentValue = "";
            }
            catch(final Exception e)
            {
                this.appendToXMLFailureMessage("Unable to process XML element value " + qName + ": " + e.getMessage()
                    + "\n");
            }

            //If the element that just ended is the element being processed (not a subelement), then finalize it and
            //remove it from the stack.
            if(qName.equals(getXMLReaderOnTopOfStack().getXMLTagName()))
            {
                try
                {
                    getXMLReaderOnTopOfStack().validate();
                    getXMLReaderOnTopOfStack().finalizeReading();
                }
                catch(final XMLReaderException e)
                {
                    this.appendToXMLFailureMessage("Unable to process XML element value " + qName + ": "
                        + e.getMessage() + "\n");
                }
                this.removeXMLReaderFromTopOfStack();
            }
        }
    }

    @Override
    public void characters(final char[] chars, final int startIndex, final int endIndex)
    {
        _currentValue += new String(chars, startIndex, endIndex);
    }

    @Override
    public void endDocument()
    {
    }

}
