package ohd.hseb.hefs.utils.xml;

import java.util.List;
import java.util.Map;

import ohd.hseb.hefs.utils.tools.MapTools;

import org.xml.sax.Attributes;

import com.google.common.collect.Lists;

/**
 * Reads values into the supplied map from xml. Both K and V should have a zero argument constructor which is used for
 * reading in xml.
 * 
 * @author alexander.garbarino
 */
public class MapXMLReader<K extends XMLReadable, V extends XMLReadable> extends XMLReaderAdapter
{
    private final Map<K, V> _resultMap;
    private final XMLReaderFactory<K> _keyFactory;
    private final XMLReaderFactory<V> _valueFactory;
    private final List<K> _tempKeyList;
    private final List<V> _tempValList;

    private K _tempKey;
    private XMLReader _tempKeyReader;
    private V _tempValue;
    private XMLReader _tempValueReader;

    /**
     * Create a reader for the given map. Since a class at runtime doesn't know its generic parameterizations (K and V
     * here), you need to pass it in upon construction so it knows.
     * 
     * @param tagName the tag name to read from
     * @param map the map to load read data into
     * @param keyFactory factory for making new keys to be read into
     * @param valueFactory factory for making new values to be read into
     */
    public MapXMLReader(String tagName, Map<K, V> map, XMLReaderFactory<K> keyFactory, XMLReaderFactory<V> valueFactory)
    {
        super(tagName);

        _resultMap = map;
        _keyFactory = keyFactory;
        _valueFactory = valueFactory;

        _tempKeyList = Lists.newArrayList();
        _tempValList = Lists.newArrayList();

        _tempKey = _keyFactory.get();
        _tempValue = _valueFactory.get();
        _tempKeyReader = _tempKey.getReader();
        _tempValueReader = _tempValue.getReader();
    }

    @Override
    public void setValueOfElement(String elementName, String value) throws XMLReaderException
    {
        if("mapEntry".equals(elementName))
        {
            _tempKeyList.add(_tempKey);
            _tempValList.add(_tempValue);
            try
            {
                _tempKey = _keyFactory.get();
                _tempValue = _valueFactory.get();
                _tempKeyReader = _tempKey.getReader();
                _tempValueReader = _tempValue.getReader();
            }
            catch(Exception e)
            {
                throw new XMLReaderException("Map reader exception.");
            }
        }
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(String elementName, Attributes attr) throws XMLReaderException
    {
        super.readInPropertyFromXMLElement(elementName, attr);
        try
        {
            //We remove mapEntryPosition from the list of attributes, because it is handled here, not in the 
            //underlying reader!  It may cause problems if the underlying reader is an XMLVariable and it not 
            //initialized to handle the attribute.
            if(_tempKeyReader.getXMLTagName().equals(elementName) && attr.getValue("mapEntryPosition").equals("key"))
            {
                XMLTools.removeAttribute(attr, "mapEntryPosition");
                return _tempKeyReader;
            }
            else if(_tempValueReader.getXMLTagName().equals(elementName)
                && attr.getValue("mapEntryPosition").equals("value"))
            {
                XMLTools.removeAttribute(attr, "mapEntryPosition");
                return _tempValueReader;
            }
        }
        catch(NullPointerException e)
        {
            throw new XMLReaderException("Map [" + getXMLTagName() + "] incorrectly formed", e);
        }
        return null;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
        super.finalizeReading();
        MapTools.addToMap(_resultMap, _tempKeyList, _tempValList);
    }
}
