package ohd.hseb.hefs.utils.xml;

import java.util.AbstractMap;
import java.util.Map;
import java.util.Set;

import ohd.hseb.hefs.utils.xml.vars.XMLString;

import com.google.common.collect.Sets;

/**
 * Reworked java.util.Properties to fit in with this XML framework.
 * 
 * @author alexander.garbarino
 */
public class Properties extends java.util.Properties implements Cloneable, XMLReadable, XMLWritable
{
    private static final String DEFAULT_XML_TAG = "properties";

    private static final long serialVersionUID = 1591408688494589950L;

    private final String _tagName;

    private MapXMLReader<XMLString, XMLString> _reader;

    /**
     * Creates an empty properties map with the default tag.
     */
    public Properties()
    {
        this(DEFAULT_XML_TAG);
    }

    /**
     * Creates an empty properties map with the specified tag.
     * 
     * @param tag the xml tag to use
     */
    public Properties(String tag)
    {
        super();
        _tagName = tag;
        initializeReader();
    }

    /**
     * Creates a properties map with the given defaults and the default tag.
     * 
     * @param defaults the default properties to use
     */
    public Properties(Properties defaults)
    {
        this(DEFAULT_XML_TAG, defaults);
    }

    /**
     * Creates a properties map with the given defaults and the specified tag.
     * 
     * @param tag the xml tag to use
     * @param defaults the default properties to use
     */
    public Properties(String tag, Properties defaults)
    {
        super(defaults);
        _tagName = tag;
        initializeReader();
    }

    /**
     * Creates a properties map initialized to the given values, using the default tag.
     * 
     * @param base map of values to initialize this map with
     */
    public Properties(Map<String, String> base)
    {
        this(DEFAULT_XML_TAG, base);
    }

    /**
     * Creates a properties map with the specified xml tag, initialized to the given values
     * 
     * @param tag the xml tag to use
     * @param base map of values to initialize this map with
     */
    public Properties(String tag, Map<String, String> base)
    {
        super();
        _tagName = tag;
        this.putAll(base);
        initializeReader();
    }

    private void initializeReader()
    {
        // Makes a Map<XMLString, XMLString> which is backed by this object.
        // Doesn't implement the full map interface, only the parts we need.
        // May want to make a static method in XMLString.        
        Map<XMLString, XMLString> map = new AbstractMap<XMLString, XMLString>()
        {
            @Override
            public Set<Map.Entry<XMLString, XMLString>> entrySet()
            {
                return null;
            }

            @Override
            public XMLString put(XMLString key, XMLString value)
            {
                Properties.this.setProperty(key.get(), value.get());
                return null;
            }
        };

        try
        {
            _reader = new MapXMLReader<XMLString, XMLString>(getXMLTagName(),
                                                             map,
                                                             XMLString.makeFactory("keyString"),
                                                             XMLString.makeFactory("valString"));
        }
        catch(Exception e)
        {
            e.printStackTrace();
            throw new RuntimeException("Unexpected Error", e);
        }
    }

    @Override
    public Properties clone()
    {
        return (Properties)super.clone();
    }

    @Override
    public XMLWriter getWriter()
    {
        // Turns this from a Map<String, String> to a Map<XMLString, XMLString>.
        // May want to make a static method in XMLString.
        Map<XMLString, XMLString> map = new AbstractMap<XMLString, XMLString>()
        {
            @Override
            public Set<Map.Entry<XMLString, XMLString>> entrySet()
            {
                Set<Map.Entry<XMLString, XMLString>> set = Sets.newHashSet();
                for(Map.Entry<Object, Object> entry: Properties.this.entrySet())
                {
                    set.add(new AbstractMap.SimpleEntry<XMLString, XMLString>(new XMLString("keyString",
                                                                                            (String)entry.getKey()),
                                                                              new XMLString("valString",
                                                                                            (String)entry.getValue())));
                }
                return set;
            }
        };

        return new MapXMLWriter(getXMLTagName(), map);
    }

    public String getXMLTagName()
    {
        return _tagName;
    }

    @Override
    public XMLReader getReader()
    {
        return _reader;
    }

}
