package ohd.hseb.hefs.utils.xml;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.utils.xml.vars.XMLInteger;

import org.xml.sax.Attributes;

/**
 * Paired with {@link TableXMLWriter}, it directly populates a storage double array, or table, that has already been
 * sized (both dimensions). The first index in the double array is referred to as the row, and the second as the column
 * within this class. {@link #getXMLTagName()} specifies the top level XML tag. The tag for each of the rows of the
 * table (first index) is {@link #_tagNameForRow}. The tag names for the columns of the table (second index) is
 * {@link #_tagNameForColumn}. Each row element must specify the first index int the backing double array via a 'row'
 * attribute; if the row index is outside the range of the backing table, then an exception will be thrown. If any row
 * does not have the appropriate number of columns (array size is wrong relative to the backing array), then an
 * exception will be thrown by {@link ArrayXMLReader}.<br>
 * <br>
 * This class handles the same double array component types as {@link ArrayXMLReader} does for single array component
 * types. In fact, it uses {@link ArrayXMLReader} to read columns of the table. For examples, see the
 * {@link TableXMLReaderWriterTest}.
 * 
 * @author Hank.Herr
 * @param <E>
 */
public class TableXMLReader<E extends XMLReadable> extends XMLReaderAdapter
{

    private Object _backingArray;
    private XMLReaderFactory<E> _factory = null;
    private String _tagNameForRow = null;
    private String _tagNameForColumn = null;
    private int _numberOfRows;

    /**
     * Indicates which rows should be read. If empty. all rows are to be read.
     */
    private final List<Integer> _rowsToReadFromXML = new ArrayList<Integer>();

    /**
     * Passed through to {@link ArrayXMLReader}, which is used to read rows.
     */
    private char _delimiter = '|';

    /**
     * @param tag
     * @param doubleArrayToPopulate
     * @param factory
     */
    public TableXMLReader(final String tag,
                          final String rowTagName,
                          final Object[][] doubleArrayToPopulate,
                          final XMLReaderFactory<E> factory)
    {
        super(tag);
        _factory = factory;
        _tagNameForRow = rowTagName;
        _tagNameForColumn = factory.get().getReader().getXMLTagName();
        _backingArray = doubleArrayToPopulate;
        _numberOfRows = doubleArrayToPopulate.length;
        if(_numberOfRows == 0)
        {
            throw new IllegalArgumentException("Cannot read a 0-length double array.");
        }
    }

    /**
     * Only call this for primitive types
     */
    private void init(final String tag,
                      final String rowTagName,
                      final String columnTagName,
                      final Object backingArray,
                      final int firstDimensionLength)
    {
        _factory = null;
        _tagNameForRow = rowTagName;
        _tagNameForColumn = columnTagName;
        _backingArray = backingArray;
        _numberOfRows = firstDimensionLength;
    }

    public TableXMLReader(final String tag,
                          final String rowTagName,
                          final String columnTagName,
                          final byte[][] doubleArrayToPopulate)
    {
        super(tag);
        init(tag, rowTagName, columnTagName, doubleArrayToPopulate, doubleArrayToPopulate.length);
    }

    public TableXMLReader(final String tag,
                          final String rowTagName,
                          final String columnTagName,
                          final char[][] doubleArrayToPopulate)
    {
        super(tag);
        init(tag, rowTagName, columnTagName, doubleArrayToPopulate, doubleArrayToPopulate.length);
    }

    public TableXMLReader(final String tag,
                          final String rowTagName,
                          final String columnTagName,
                          final short[][] doubleArrayToPopulate)
    {
        super(tag);
        init(tag, rowTagName, columnTagName, doubleArrayToPopulate, doubleArrayToPopulate.length);
    }

    public TableXMLReader(final String tag,
                          final String rowTagName,
                          final String columnTagName,
                          final int[][] doubleArrayToPopulate)
    {
        super(tag);
        init(tag, rowTagName, columnTagName, doubleArrayToPopulate, doubleArrayToPopulate.length);
    }

    public TableXMLReader(final String tag,
                          final String rowTagName,
                          final String columnTagName,
                          final long[][] doubleArrayToPopulate)
    {
        super(tag);
        init(tag, rowTagName, columnTagName, doubleArrayToPopulate, doubleArrayToPopulate.length);
    }

    public TableXMLReader(final String tag,
                          final String rowTagName,
                          final String columnTagName,
                          final float[][] doubleArrayToPopulate)
    {
        super(tag);
        init(tag, rowTagName, columnTagName, doubleArrayToPopulate, doubleArrayToPopulate.length);
    }

    public TableXMLReader(final String tag,
                          final String rowTagName,
                          final String columnTagName,
                          final double[][] doubleArrayToPopulate)
    {
        super(tag);
        init(tag, rowTagName, columnTagName, doubleArrayToPopulate, doubleArrayToPopulate.length);
    }

    /**
     * @see ArrayXMLReader#setDelimiter(char).
     */
    public void setDelimiter(final char delimiter)
    {
        _delimiter = delimiter;
    }

    @SuppressWarnings({"unchecked", "rawtypes"})
    public ArrayXMLReader getArrayXMLReader(final int i) throws ArrayIndexOutOfBoundsException
    {
        if(_backingArray instanceof byte[][])
        {
            return new ArrayXMLReader(_tagNameForRow, _tagNameForColumn, ((byte[][])_backingArray)[i]);
        }
        else if(_backingArray instanceof char[][])
        {
            return new ArrayXMLReader(_tagNameForRow, _tagNameForColumn, ((char[][])_backingArray)[i]);
        }
        else if(_backingArray instanceof short[][])
        {
            return new ArrayXMLReader(_tagNameForRow, _tagNameForColumn, ((short[][])_backingArray)[i]);
        }
        else if(_backingArray instanceof int[][])
        {
            return new ArrayXMLReader(_tagNameForRow, _tagNameForColumn, ((int[][])_backingArray)[i]);
        }
        else if(_backingArray instanceof long[][])
        {
            return new ArrayXMLReader(_tagNameForRow, _tagNameForColumn, ((long[][])_backingArray)[i]);
        }
        else if(_backingArray instanceof float[][])
        {
            return new ArrayXMLReader(_tagNameForRow, _tagNameForColumn, ((float[][])_backingArray)[i]);
        }
        else if(_backingArray instanceof double[][])
        {
            return new ArrayXMLReader(_tagNameForRow, _tagNameForColumn, ((double[][])_backingArray)[i]);
        }
        return new ArrayXMLReader(_tagNameForRow, ((Object[][])_backingArray)[i], _factory);
    }

    /**
     * Add a row for reading. Note that though only the rows for reading are read, the size of the table is not altered.
     * 
     * @param row The row to read. Counting starts at 0, of course.
     */
    public void addRowToRead(final int row)
    {
        _rowsToReadFromXML.add(row);
    }

    /**
     * Clear rows to read, so that all rows are read.
     */
    public void clearRowsToRead()
    {
        _rowsToReadFromXML.clear();
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
    {
        super.readInPropertyFromXMLElement(elementName, attr);
        if(elementName.equals(getXMLTagName()))
        {
            //Used to reset a row counter, but now does nothing since the row is specified as an attribute.
        }
        else if(elementName.equals(_tagNameForRow))
        {
            int row = -1;
            try
            {
                row = Integer.parseInt(attr.getValue("row"));
            }
            catch(final NumberFormatException e)
            {
                throw new XMLReaderException("Attribute 'row' with value '" + attr.getValue("row")
                    + "', is not an integer.");
            }
            if((row < 0) || (row >= _numberOfRows))
            {
                throw new XMLReaderException("Attribute specified table row index " + row
                    + " is negative or as is as large or larger than the the number of rows in the table "
                    + _numberOfRows + ".");
            }

            if((!_rowsToReadFromXML.isEmpty()) && (!_rowsToReadFromXML.contains(row)))
            {
                return null; //Skip the row.
            }

            final ArrayXMLReader arrayReader = getArrayXMLReader(row);
            arrayReader.setDelimiter(_delimiter);
            arrayReader.addAttribute(new XMLInteger("row"), true);
            return arrayReader;
        }
        else
        {
            throw new XMLReaderException("For element '" + getXMLTagName() + "', unrecognized subelement '"
                + elementName + "'.");
        }
        return null;
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
        super.finalizeReading();
    }

}
