package ohd.hseb.hefs.utils.xml;

import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;

import javax.xml.stream.XMLStreamWriter;

import ohd.hseb.hefs.utils.xml.vars.XMLInteger;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.google.common.base.Strings;

/**
 * Writes double arrays, or tables, to XML. {@link #getXMLTagName()} specifies the top level XML tag. The tag for each
 * of the rows of the table (first index) is {@link #_tagNameForRow}. The tag names for the columns of the table (second
 * index) is {@link #_tagNameForColumn}. Each row element of the table is indexed using a 'row' attribute. The XML does
 * not include any array size information. So, whatever will read in the array later must know how many elements to
 * expect. <br>
 * <br>
 * This class handles the same types as {@link ArrayXMLWriter}. In fact, it uses {@link ArrayXMLWriter} to write out the
 * column values for a row. Note that if the table is large it is recommended that the
 * {@link XMLStreamWriterWriter#writeXML(XMLStreamWriter)} method be used for writing, as it is much faster than the
 * {@link XMLWriter#writePropertyToXMLElement(Document)} method.<br>
 * <br>
 * To save space, if necessary, rows can be skipped. For example, if a row contains all missing data, it can be left out
 * of the XML. However, this means upon reading the XML, the rows must be initialized to contain all missing data so
 * that any row not read in will be left with all missing. To setup a check for skipping a row, call
 * {@link #setSkipChecker(TableWriterRowSkipChecker)} after construction and pass in an implementer of
 * {@link TableWriterRowSkipChecker}.<br>
 * <br>
 * Another restriction that can be made is to only write out specific rows of the table. This can be done by calling
 * {@link #addRowToWrite(int)}. If this is not called, then the internal list {@link #_rowsToWriteToXML} will be empty,
 * so that all rows are written. If {@link #addRowToWrite(int)} is called at least once, then only those rows added to
 * {@link #_rowsToWriteToXML} will be written ({@link #_skipChecker} still applies). Call {@link #clearRowsToWrite()} if
 * this object will be used again in order to reset {@link #_rowsToWriteToXML}. For examples, see
 * {@link TableXMLReaderWriterTest}.
 * 
 * @author Hank.Herr
 */
public class TableXMLWriter extends XMLWriterAdapter implements XMLStreamWriterWriter
{

    /**
     * The table, or double array, to write is stored as an {@link Object}.
     */
    private Object _backingArray = null;

    /**
     * The number of rows (first dimension length) in the table, used internally.
     */
    private int _numberOfRows = -1;

    /**
     * The tag name for the rows to output. This is passed to an {@link ArrayXMLWriter} as the top level tag name.
     */
    private String _tagNameForRow = null;

    /**
     * The tag name for the columns to output. This is passed to an {@link ArrayXMLWriter} as the value tag name.
     */
    private String _tagNameForColumn = null;

    private TableWriterRowSkipChecker _skipChecker = null;

    /**
     * Passed through to the used {@link ArrayXMLWriter}.
     */
    private boolean _useDelimiter = false;

    /**
     * Passed through to the used {@link ArrayXMLWriter}.
     */
    private char _delimiter = '|';

    /**
     * Passed through to the used {@link ArrayXMLWriter}.
     */
    private NumberFormat _numberFormatter = null;

    /**
     * Indicates which rows should be written. If empty. all rows are to be written.
     */
    private final List<Integer> _rowsToWriteToXML = new ArrayList<Integer>();

    public <E extends XMLWritable> TableXMLWriter(final String tag, final String tagNameForRow, final E[][] values)
    {
        super(tag);
        _tagNameForRow = tagNameForRow;
        _tagNameForColumn = null;
        _backingArray = values;
        _numberOfRows = values.length;
        if(_numberOfRows == 0)
        {
            throw new IllegalArgumentException("Array of values to write is empty.");
        }
    }

    public TableXMLWriter(final String tag,
                          final String tagNameForFirstIndex,
                          final String tagNameForSecondIndex,
                          final Object[][] values)
    {
        super(tag);
        _tagNameForRow = tagNameForFirstIndex;
        _tagNameForColumn = tagNameForSecondIndex;
        _backingArray = values;
        _numberOfRows = values.length;
        if(_numberOfRows == 0)
        {
            throw new IllegalArgumentException("Array of values to write is empty.");
        }
    }

    public TableXMLWriter(final String tag,
                          final String tagNameForFirstIndex,
                          final String tagNameForSecondIndex,
                          final byte[][] values)
    {
        super(tag);
        if(values.length == 0)
        {
            throw new IllegalArgumentException("Array of values to write is empty.");
        }
        init(tagNameForFirstIndex, tagNameForSecondIndex, values, values.length, values[0].length);
    }

    public TableXMLWriter(final String tag,
                          final String tagNameForFirstIndex,
                          final String tagNameForSecondIndex,
                          final char[][] values)
    {
        super(tag);
        if(values.length == 0)
        {
            throw new IllegalArgumentException("Array of values to write is empty.");
        }
        init(tagNameForFirstIndex, tagNameForSecondIndex, values, values.length, values[0].length);
    }

    public TableXMLWriter(final String tag,
                          final String tagNameForFirstIndex,
                          final String tagNameForSecondIndex,
                          final short[][] values)
    {
        super(tag);
        if(values.length == 0)
        {
            throw new IllegalArgumentException("Array of values to write is empty.");
        }
        init(tagNameForFirstIndex, tagNameForSecondIndex, values, values.length, values[0].length);
    }

    public TableXMLWriter(final String tag,
                          final String tagNameForFirstIndex,
                          final String tagNameForSecondIndex,
                          final int[][] values)
    {
        super(tag);
        if(values.length == 0)
        {
            throw new IllegalArgumentException("Array of values to write is empty.");
        }
        init(tagNameForFirstIndex, tagNameForSecondIndex, values, values.length, values[0].length);
    }

    public TableXMLWriter(final String tag,
                          final String tagNameForFirstIndex,
                          final String tagNameForSecondIndex,
                          final long[][] values)
    {
        super(tag);
        if(values.length == 0)
        {
            throw new IllegalArgumentException("Array of values to write is empty.");
        }
        init(tagNameForFirstIndex, tagNameForSecondIndex, values, values.length, values[0].length);
    }

    public TableXMLWriter(final String tag,
                          final String tagNameForFirstIndex,
                          final String tagNameForSecondIndex,
                          final float[][] values)
    {
        super(tag);
        if(values.length == 0)
        {
            throw new IllegalArgumentException("Array of values to write is empty.");
        }
        init(tagNameForFirstIndex, tagNameForSecondIndex, values, values.length, values[0].length);
    }

    public TableXMLWriter(final String tag,
                          final String tagNameForFirstIndex,
                          final String tagNameForSecondIndex,
                          final double[][] values)
    {
        super(tag);
        if(values.length == 0)
        {
            throw new IllegalArgumentException("Array of values to write is empty.");
        }
        init(tagNameForFirstIndex, tagNameForSecondIndex, values, values.length, values[0].length);
    }

    public void setSkipChecker(final TableWriterRowSkipChecker skipChecker)
    {
        _skipChecker = skipChecker;
    }

    private void init(final String firstIndexTagName,
                      final String secondIndexTagName,
                      final Object values,
                      final int firstDimensionLength,
                      final int secondDimensionLength)
    {
        _tagNameForRow = firstIndexTagName;
        _tagNameForColumn = secondIndexTagName;
        _backingArray = values;
        _numberOfRows = firstDimensionLength;
    }

    public ArrayXMLWriter getArrayXMLWriter(final int i)
    {
        if(_backingArray instanceof byte[][])
        {
            return new ArrayXMLWriter(_tagNameForRow, _tagNameForColumn, ((byte[][])_backingArray)[i]);
        }
        else if(_backingArray instanceof char[][])
        {
            return new ArrayXMLWriter(_tagNameForRow, _tagNameForColumn, ((char[][])_backingArray)[i]);
        }
        else if(_backingArray instanceof short[][])
        {
            return new ArrayXMLWriter(_tagNameForRow, _tagNameForColumn, ((short[][])_backingArray)[i]);
        }
        else if(_backingArray instanceof int[][])
        {
            return new ArrayXMLWriter(_tagNameForRow, _tagNameForColumn, ((int[][])_backingArray)[i]);
        }
        else if(_backingArray instanceof long[][])
        {
            return new ArrayXMLWriter(_tagNameForRow, _tagNameForColumn, ((long[][])_backingArray)[i]);
        }
        else if(_backingArray instanceof float[][])
        {
            return new ArrayXMLWriter(_tagNameForRow, _tagNameForColumn, ((float[][])_backingArray)[i]);
        }
        else if(_backingArray instanceof double[][])
        {
            return new ArrayXMLWriter(_tagNameForRow, _tagNameForColumn, ((double[][])_backingArray)[i]);
        }
        return new ArrayXMLWriter(_tagNameForRow, _tagNameForColumn, ((Object[][])_backingArray)[i]);
    }

    /**
     * @see ArrayXMLWriter#setNumberFormatter(NumberFormat).
     */
    public void setNumberFormatter(final NumberFormat format)
    {
        _numberFormatter = format;
    }

    /**
     * @see ArrayXMLWriter#setUseDelimiter(boolean).
     */
    public void setUseDelimiter(final boolean b)
    {
        _useDelimiter = b;
    }

    /**
     * @see ArrayXMLWriter#setDelimiter(char).
     */
    public void setDelimiter(final char c)
    {
        _delimiter = c;
    }

    /**
     * Add a row for writing to {@link #_rowsToWriteToXML}. If any row is added, only those rows indicated will be
     * written. If {@link #_rowsToWriteToXML} is empty, meaning this method was never called, then all rows will be
     * written. Note that {@link #_skipChecker} always applies regardless of the number of rows to be written.
     * 
     * @param row The row to write. Counting starts at 0, of course.
     */
    public void addRowToWrite(final int row)
    {
        _rowsToWriteToXML.add(row);
    }

    /**
     * Clear rows to write, so that all rows are read.
     */
    public void clearRowsToWrite()
    {
        _rowsToWriteToXML.clear();
    }

    @Override
    public Element writePropertyToXMLElement(final Document request) throws XMLWriterException
    {
        //This is slow!  The recommended way to write this is via the writeXML call below.
        if(Strings.isNullOrEmpty(getXMLTagName()))
        {
            throw new IllegalStateException("Tag name is not specified.");
        }

        //A composite writer is used to store all of the ArrayXMLWriters and then generate the element.
        //I cannot have this returned via getWriter(), because this extends XMLWriterAdapter, and such 
        //things must always return this via getWriter() or the writing tools get confused.
        final CompositeXMLWriter proxyWriter = new CompositeXMLWriter(getXMLTagName());
        proxyWriter.addAllAttributes(this.getAttributes());
        for(int i = 0; i < _numberOfRows; i++)
        {
            if(_rowsToWriteToXML.isEmpty() || (_rowsToWriteToXML.contains(i)))
            {
                final ArrayXMLWriter rowWriter = getArrayXMLWriter(i);
                rowWriter.setUseDelimiter(_useDelimiter);
                rowWriter.setDelimiter(_delimiter);
                rowWriter.setNumberFormatter(_numberFormatter);
                if((_skipChecker == null) || (!_skipChecker.skipRow(rowWriter.getBackingArray())))
                {
                    rowWriter.addAttribute(new XMLInteger("row", i), true);
                    proxyWriter.addComponent(rowWriter);
                }
            }
        }
        return proxyWriter.writePropertyToXMLElement(request);
    }

    @Override
    public void writeXML(final XMLStreamWriter streamWriter) throws Exception
    {
        streamWriter.writeStartElement(getXMLTagName());
        getAttributes().write(streamWriter);
        for(int i = 0; i < _numberOfRows; i++)
        {
            if(_rowsToWriteToXML.isEmpty() || (_rowsToWriteToXML.contains(i)))
            {
                final ArrayXMLWriter rowWriter = getArrayXMLWriter(i);
                rowWriter.setUseDelimiter(_useDelimiter);
                rowWriter.setDelimiter(_delimiter);
                rowWriter.setNumberFormatter(_numberFormatter);
                if((_skipChecker == null) || (!_skipChecker.skipRow(rowWriter.getBackingArray())))
                {
                    rowWriter.addAttribute(new XMLInteger("row", i), true);
                    rowWriter.writeXML(streamWriter);
                }
            }
        }
        streamWriter.writeEndElement();
    }

}
