package ohd.hseb.hefs.utils.xml;

import org.xml.sax.Attributes;

/**
 * General interface that can be used within the {@link GenericXMLReadingHandler} for parsing XML. Note that every
 * XMLReader is also {@link XMLReadable}, usually by returning this for {@link XMLReadable#getReader()}.
 * 
 * @author hank.herr
 */
public interface XMLReader extends XMLReadable
{
    /**
     * @return The tag name of the root element that this reader reads. This is the element name which the handler will
     *         assume starts and ends this XMLReader instance's reading.
     */
    public String getXMLTagName();

    /**
     * Called when the end of an element is reached. It will be passed the value for that element (as opposed to
     * attributes).
     * 
     * @param elementName Name of the element.
     * @param value Value of the element.
     * @throws XMLReaderException If the elementName or its value is invalid, throw an exception.
     */
    public void setValueOfElement(String elementName, String value) throws XMLReaderException;

    /**
     * While an XML file is being parsed, this method will be called whenever an element is found that must be processed
     * by the XMLReader instance. This includes the first element of the XMLReader instance (i.e. the root element for
     * that reader) and all sub-elements.
     * 
     * @param elementName The name of the element being processed. The first time it is called, it will always be equal
     *            to the XML tagname (see AbstractProperty).
     * @param attr The org.xml.sax.Attributes associated with the element.
     * @return XMLReader, if the elementName implies a nested XMLReader needs to be read in, then it will be returned.
     *         If null, then the property was read in completely and there were no more nested ChartProperty instances.
     * @throws XMLReaderException If the elementName or one of its attributes is invalid.
     */
    public XMLReader readInPropertyFromXMLElement(String elementName, Attributes attr) throws XMLReaderException;

    /**
     * Called when the end of document/element is reached, but after validation. This is intended to take the already
     * validated XML read-in and perform any additional steps required to complete processing. For example, if a map
     * needs to be populated based on read-in XML, it can be done through this method.<br>
     * <br>
     * Note that {@link #finalizeReading()} should only be called one time when done reading XML of some kind. It
     * *should* never be called more than one time. If the {@link XMLVariable} framework is used, this will not be
     * called more that one time per element read. However, since it is always possible for someone to screw up, it
     * might be best to account for the possibility this may be called more than once.
     * 
     * @throws XMLReaderException can be thrown if the finalization discovers an error.
     */
    public void finalizeReading() throws XMLReaderException;

    /**
     * Called when the end of document/element is reached, but BEFORE {@link #finalizeReading()}. It should check to
     * make sure that everything required was read and is valid. <br>
     * <br>
     * Note that {@link #finalizeReading()} will be called only when done reading XML of some kind, whereas this
     * validate method may be called even when an element is not read (for an example, see {@link CompositeXMLReader}
     * which validates all component {@link XMLVariable} instances, which in turn check to make sure required elements
     * were read in correctly).
     * 
     * @throws XMLReaderException
     */
    public void validate() throws XMLReaderException;
}
