package ohd.hseb.hefs.utils.xml;

import org.xml.sax.Attributes;

import com.google.common.base.Strings;

/**
 * An adapter class for creating {@link XMLReader} instances. This stores a tag name and an {@link AttributeList}.
 * 
 * @author alexander.garbarino
 */
public abstract class XMLReaderAdapter implements XMLReader
{
    protected String _tagName;

    private final AttributeList _attributes = new AttributeList();

    public XMLReaderAdapter(final String tag)
    {
        _tagName = tag;
    }

    /**
     * Wraps {@link AttributeList#addAttribute(String, Class, boolean)}.
     */
    public <T extends XMLVariable> T addAttribute(final String name, final Class<T> storageClass, final boolean required)
    {
        return _attributes.addAttribute(name, storageClass, required);
    }

    /**
     * Wraps {@link AttributeList#addAttribute(String, T, boolean)}.
     */
    public <T extends XMLVariable> void addAttribute(final String name, final T storageObject, final boolean required)
    {
        _attributes.addAttribute(name, storageObject, required);
    }

    /**
     * Wraps {@link AttributeList#addAttribute(String, T, boolean)} passing in the storageObject
     * {@link XMLVariable#getXMLTagName()} as the attribute name.
     */
    public <T extends XMLVariable> void addAttribute(final T storageObject, final boolean required)
    {
        _attributes.addAttribute(storageObject.getXMLTagName(), storageObject, required);
    }

    /**
     * Copies the attribute list elements from a {@link XMLWriterAdapter} to allow for both to use the same storage
     * objects.
     * 
     * @param writer
     */
    public void copyAttributesFrom(final XMLWriterAdapter writer)
    {
        _attributes.clear();
        _attributes.addAll(writer.getAttributes());
    }

    public AttributeList getAttributes()
    {
        return _attributes;
    }

    @Override
    public String getXMLTagName()
    {
        return _tagName;
    }

    @Override
    public void setValueOfElement(final String elementName, final String value) throws XMLReaderException
    {
    }

    @Override
    public XMLReader readInPropertyFromXMLElement(final String elementName, final Attributes attr) throws XMLReaderException
    {
        if(Strings.isNullOrEmpty(getXMLTagName()))
        {
            throw new IllegalStateException("Tag name is not specified.");
        }
        if(elementName.equals(getXMLTagName()))
        {
            _attributes.clearValues();
            _attributes.read(attr);
            _attributes.checkIfAllRequiredSet();
        }
        return null; //ALWAYS RETURN NULL!!!
    }

    @Override
    public void finalizeReading() throws XMLReaderException
    {
    }

    @Override
    public void validate() throws XMLReaderException
    {
    }

    @Override
    public XMLReader getReader()
    {
        return this;
    }
}
