package ohd.hseb.hefs.utils.xml;

import java.awt.Color;
import java.awt.Font;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.util.Collection;
import java.util.zip.GZIPOutputStream;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.sax.SAXResult;
import javax.xml.transform.stream.StreamResult;

import nl.wldelft.util.IndentingXMLStreamWriter;
import ohd.hseb.hefs.utils.datetime.HEFSDateTools;
import ohd.hseb.hefs.utils.tools.StreamTools;
import ohd.hseb.util.data.DataSet;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HString;

import org.jfree.ui.RectangleInsets;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.Text;
import org.xml.sax.Attributes;
import org.xml.sax.helpers.AttributesImpl;

import com.google.common.base.Function;
import com.sun.xml.fastinfoset.sax.SAXDocumentSerializer;
import com.sun.xml.fastinfoset.stax.StAXDocumentSerializer;

public abstract class XMLTools
{

    public static final String FASTINFOSET_EXTENSION = "fi";
    public static final String XML_EXTENSION = "xml";
    public static final String GZIP_EXTENSION = "gz";

    public static final Function<XMLReadable, XMLReader> GET_READER = new Function<XMLReadable, XMLReader>()
    {
        @Override
        public XMLReader apply(final XMLReadable input)
        {
            return input.getReader();
        }
    };

    /**
     * Uses HEFSDateTools to compute a fixed date string in milliseconds.
     * 
     * @param date Date string.
     * @param time Time string.
     * @return Date in millis based on the two input strings concatenated together with a space in the middle.
     */
    public static long computeDateInMillisFromStrings(final String date, final String time)
    {
        final String datetimeStr = date + " " + time;

        final Long millis = HEFSDateTools.computeFixedDate(datetimeStr).getTimeInMillis();

        if(millis == null)
        {
            return Long.MIN_VALUE;
        }

        return millis;
    }

    /**
     * Computes the date from XML attributes that contain values 'date' and 'time'.
     * 
     * @param attr Attributes which must contain 'date' and 'time' values.
     * @return Result of computeDateInMillisFromStrings passing in date value and time value.
     * @throws XMLReaderException If either value is null (both are required).
     */
    public static long computeDateFromAttrubtes(final Attributes attr) throws XMLReaderException
    {
        final String dateStr = attr.getValue("date");
        if(dateStr == null)
        {
            throw new XMLReaderException("Missing date attribute within startDate tag.");
        }

        final String timeStr = attr.getValue("time");
        if(timeStr == null)
        {
            throw new XMLReaderException("Missing time attribute within startDate tag.");
        }

        final long millis = computeDateInMillisFromStrings(dateStr, timeStr);
        if(millis == Long.MIN_VALUE)
        {
            throw new XMLReaderException("Poorly format date or time: date = '" + dateStr + "'; time = " + timeStr
                + ".");
        }
        return millis;
    }

    /**
     * If the attributes specify a relative date, it is returned. Otherwise, it builds a date string (i.e., 'CCYY-MM-DD
     * hh:mm:ss') from XML attributes that contain values 'date' and 'time'.
     * 
     * @param attr Attributes which must contain 'date' and 'time' values or 'relativeDate'.
     * @return The date string, '[date] [time]' or relative date.
     * @throws XMLReaderException If relativeDate is null and one of date and time are null values, or if the resulting
     *             string is invalid, implying either the value of date, time, or relativeDate is invalid.
     */
    public static String computeDateStringFromAttributes(final Attributes attr) throws XMLReaderException
    {
        String fullDateStr = null;
        if(attr.getValue("relativeDate") != null)
        {
            fullDateStr = attr.getValue("relativeDate");
        }
        else
        {
            final String dateStr = attr.getValue("date");
            if(dateStr == null)
            {
                throw new XMLReaderException("Missing date attribute within date tag.");
            }

            final String timeStr = attr.getValue("time");
            if(timeStr == null)
            {
                throw new XMLReaderException("Missing time attribute within date tag.");
            }

            fullDateStr = dateStr + " " + timeStr;
        }

        if(!HEFSDateTools.isDateStringValid(fullDateStr))
        {
            throw new XMLReaderException("Attribute date value is badly formatted: " + fullDateStr);
        }
        return fullDateStr;
    }

    /**
     * The error checking in this method is not complete, so it is best if the program design makes invalid attributes
     * impossible. For example, if the unit is year or month, the resulting string will not be checked for validity at
     * all, so that '1xv4 years' is valid. However, testing is done for smaller units (weeks to hours). Furthermore, for
     * year and month, the unit must only contain the strings 'year' or 'month'. So '1 yearhank' is considered valid.<br>
     * <br>
     * The return is a single string to be used to store the time step. The string can be parsed later on to acquire the
     * size of the time step in millis.
     * 
     * @param multStr The multiplier as a String. This can be null only if unit is 'period' and period is valid.
     * @param unitStr The unit as a String. This canot be null.
     * @param specialUnits {@link Collection} of special units that are acceptable.
     * @return A time step string of the format '[quantity] [unit]'.
     * @throws XMLReaderException If an expected attribute is missing, the unit is not recognized, or the multiplier
     *             attribute is not an integer.
     */
    public static String computeTimeStepStringFromComponents(final String multStr,
                                                             final String unitStr,
                                                             final Collection<String> specialUnits) throws XMLReaderException
    {
        if(unitStr == null)
        {
            throw new XMLReaderException("Missing unit attribute within time step tag.");
        }

        if(specialUnits.contains(unitStr))
        {
            return "1 " + unitStr;
        }
        else if(unitStr.toLowerCase().contains("year") || unitStr.toLowerCase().contains("month"))
        {
            if(multStr == null)
            {
                throw new XMLReaderException("Missing multiplier attribute within time step tag.");
            }

            return multStr + " " + unitStr;
        }
        else
        {
            if(multStr == null)
            {
                throw new XMLReaderException("Missing multiplier attribute within time step tag.");
            }

            final String timeStepStr = multStr + " " + unitStr;
            final long timeStepLong = HCalendar.computeIntervalValueInMillis(timeStepStr);
            if(timeStepLong == (long)DataSet.MISSING)
            {
                throw new XMLReaderException("Either a non-integer multiplier or unrecognized unit.");
            }

            return timeStepStr;
        }
    }

    /**
     * Calls {@link #computeTimeStepStringFromComponents(String, String, Collection)}, passing in attributes from the
     * given {@link Attributes} object.
     */
    public static String computeTimeStepStringFromAttributes(final Attributes attr,
                                                             final Collection<String> specialUnits) throws XMLReaderException
    {
        return computeTimeStepStringFromComponents(attr.getValue("multiplier"), attr.getValue("unit"), specialUnits);
    }

    /**
     * Calls {@link #computeTimeStepStringFromComponents(String, String, Collection)}, passing in attributes from the
     * given {@link AttributeList} object.
     */
    public static String computeTimeStepStringFromAttributes(final AttributeList attr,
                                                             final Collection<String> specialUnits) throws XMLReaderException

    {
        final Attribute unit = attr.retrieve("unit");
        final Attribute multiplier = attr.retrieve("multiplier");
        return computeTimeStepStringFromComponents((String)unit.getStorageObject().get(),
                                                   (String)multiplier.getStorageObject().get(),
                                                   specialUnits);
    }

    /**
     * Creates a date XML element, with attributes of either 'relativeDate' or both 'date' and 'time'.
     * 
     * @param request Document for which createElement is called.
     * @param xmlTag The XML tag for the element.
     * @param validFullDateString A valid relative ([base] +/- [quan] [unit] [quan] [unit] ...) or fixed (CCYYMMDD
     *            hh:mm:ss) date string.
     * @return An element that can be added to the XML document.
     */
    public static Element createDateElement(final Document request,
                                            final String xmlTag,
                                            final String validFullDateString)
    {
        final Element element = request.createElement(xmlTag);

        if(HEFSDateTools.isRelativeDate(validFullDateString))
        {
            element.setAttribute("relativeDate", validFullDateString);
        }
        else
        {
            element.setAttribute("date", validFullDateString.substring(0, 10));
            element.setAttribute("time", validFullDateString.substring(11, 19));
        }
        return element;
    }

    /**
     * Creates a time step XML element with attributes 'unit' and 'multiplier'.
     * 
     * @param request Document for which createElement is called.
     * @param xmlTag The XML tag for the element.
     * @param validFullTimeStepString A valid time step string of hte format '[multiplier] [unit]' where unit is one of
     *            year(s), month(s), week(s), day(s), or hour(s).
     * @return An element that can be added to the XML document.
     */
    public static Element createTimeStepElement(final Document request,
                                                final String xmlTag,
                                                final String validFullTimeStepString)
    {
        final Element element = request.createElement(xmlTag);

        final int pos = HString.findFirstNonNumericCharacter(validFullTimeStepString);
        element.setAttribute("unit", validFullTimeStepString.substring(pos + 1, validFullTimeStepString.length()));
        element.setAttribute("multiplier", validFullTimeStepString.substring(0, pos));

        return element;
    }

    /**
     * Shorthand method to create a text node, or an element with only a value and no attributes.
     * 
     * @param request Document for which createElement is called.
     * @param xmlTag The XML tag for the element.
     * @param value The value of the element to create.
     * @return An element with the appropriate text value.
     */
    public static Element createTextNodeElement(final Document request, final String tag, final String value)
    {
        final Text valueTextNode = request.createTextNode(value);
        final Element element = request.createElement(tag);
        element.appendChild(valueTextNode);
        return element;
    }

    /**
     * Strips name space information from an XML. This is useful if one XML loaded from a file is to be incorporated
     * within another larger XML. If the namespace is still in the smaller XML, it will cause validation errors.
     * 
     * @param xmlString The string from which to strip the namespace info.
     * @return Stripped string.
     */
    public static String removeNameSpaceInformationFromXML(final String xmlString)
    {
        if(xmlString.startsWith("<?"))
        {
            final int endIndex = xmlString.indexOf("?>");
            if(endIndex < xmlString.length() - 2)
            {
                return xmlString.substring(endIndex + 2);
            }
        }
        return xmlString;
    }

    /**
     * @param tagName The name for the tag.
     * @param request Document to use for createElement.
     * @param font The font to create an Element for.
     * @return Element specifying the font using attributes 'family', 'style', and 'size'.
     */
    public static Element createFontElement(final String tagName, final Document request, final Font font)
    {
        final Element fontElement = request.createElement(tagName);
        fontElement.setAttribute("family", font.getFamily());
        fontElement.setAttribute("style", "" + font.getStyle());
        fontElement.setAttribute("size", "" + font.getSize());
        return fontElement;
    }

    /**
     * @param attr Attributes object from which to extract the font, which must include values for 'family', 'style',
     *            and 'size'.
     * @return Font specified by attr.
     * @throws XMLToolsException if a problem occurs while trying to translate it. This expects a fully specified font:
     *             family, style and size.
     */
    public static Font extractFontFromAttributes(final Attributes attr) throws XMLToolsException
    {
        final String family = attr.getValue("family");
        final String style = attr.getValue("style");
        final String size = attr.getValue("size");

        if((family == null) || (style == null) || (size == null))
        {
            throw new XMLToolsException("Missing either family, style, or size attribute.");
        }

        try
        {
            final int styleInt = Integer.parseInt(style);
            final int sizeInt = Integer.parseInt(size);
            return new Font(family, styleInt, sizeInt);
        }
        catch(final NumberFormatException nfe)
        {
            throw new XMLToolsException("Either style or size attribute is not a number.");
        }
    }

    /**
     * @param tagName The name for the tag.
     * @param request Document to use for createElement.
     * @param RectangleInsets The RectangleInsets to create an element for.
     * @return Element specifying the insets using attributes 'top', 'left', 'bottom' and 'right'.
     */
    public static Element createInsetsElement(final String tagName, final Document request, final RectangleInsets insets)
    {
        final Element insetsElement = request.createElement(tagName);
        insetsElement.setAttribute("top", "" + insets.getTop());
        insetsElement.setAttribute("left", "" + insets.getLeft());
        insetsElement.setAttribute("bottom", "" + insets.getBottom());
        insetsElement.setAttribute("right", "" + insets.getRight());
        return insetsElement;
    }

    /**
     * @param attr Attributes object from which to extract the Insets object, which must include all attributes 'top',
     *            'left', 'bottom', and 'right'.
     * @return RectangleInsets specified by attr.
     * @throws XMLToolsException if a problem occurs while trying to translate it. This expects a fully specified
     *             Insets: top, left, bottom, right (all integers).
     */
    public static RectangleInsets extractInsetsFromAttributes(final Attributes attr) throws XMLToolsException
    {
        final String topStr = attr.getValue("top");
        final String leftStr = attr.getValue("left");
        final String bottomStr = attr.getValue("bottom");
        final String rightStr = attr.getValue("right");

        if((topStr == null) || (leftStr == null) || (bottomStr == null) || (rightStr == null))
        {
            throw new XMLToolsException("One of the attributes specifying the insets, top, left, bottom, or right, is missing.");
        }

        try
        {
            final RectangleInsets insets = new RectangleInsets(Double.parseDouble(topStr),
                                                               Double.parseDouble(leftStr),
                                                               Double.parseDouble(bottomStr),
                                                               Double.parseDouble(rightStr));
            return insets;
        }
        catch(final Exception e)
        {
            throw new XMLToolsException("At least one of the attributes specifying the insets is not an integer; "
                + "the attribute values are '" + topStr + "', '" + leftStr + "', '" + bottomStr + "', and '" + rightStr
                + "'.");
        }
    }

    /**
     * @param tagName The name for the tag.
     * @param request Document to use for createElement.
     * @param color The color to create an Element for.
     * @return Element specifying the color using attributes 'red', 'green', 'blue', and 'alpha' (transparency).
     */
    public static Element createColorElement(final String tagName, final Document request, final Color color)
    {
        final Element colorElement = request.createElement(tagName);
        colorElement.setAttribute("red", "" + color.getRed());
        colorElement.setAttribute("green", "" + color.getGreen());
        colorElement.setAttribute("blue", "" + color.getBlue());
        colorElement.setAttribute("alpha", "" + color.getAlpha());
        return colorElement;
    }

    /**
     * @param attr Attributes object from which to extract the color, which must contain values for 'red', 'green', and
     *            'blue'; 'alpha' is an optional attribute.
     * @return Color specified by attr.
     * @throws XMLToolsException if a problem occurs while trying to translate it. This expects a fully specified color:
     *             r, g, b, alpha (alpha is optional!).
     */
    public static Color extractColorFromAttributes(final Attributes attr) throws XMLToolsException
    {
        final String red = attr.getValue("red");
        final String green = attr.getValue("green");
        final String blue = attr.getValue("blue");
        final String alpha = attr.getValue("alpha");

        if((red == null) || (green == null) || (blue == null))
        {
            throw new XMLToolsException("Missing either red, green, or blue attribute.");
        }

        try
        {
            if(alpha != null)
            {
                return new Color(Integer.parseInt(red),
                                 Integer.parseInt(green),
                                 Integer.parseInt(blue),
                                 Integer.parseInt(alpha));
            }
            else
            {
                return new Color(Integer.parseInt(red), Integer.parseInt(green), Integer.parseInt(blue));
            }
        }
        catch(final NumberFormatException nfe)
        {
            throw new XMLToolsException("At least one attribute is not a number.");
        }
    }

    /**
     * Shorthand for Boolean.parseBoolean which throws an XMLRederException.
     * 
     * @param value Value to parse.
     * @return Parsed value.
     * @throws XMLReaderException If neither true not false.
     */
    public static Boolean processBoolean(final String value) throws XMLReaderException
    {
        try
        {
            return Boolean.parseBoolean(value.trim());
        }
        catch(final NumberFormatException e)
        {
            throw new XMLReaderException("Boolean string '" + value.trim() + "' is neither true nor false.");
        }
    }

    /**
     * Shorthand for Integer.parseInt which throws an XMLRederException.
     * 
     * @param value Value to parse.
     * @return Parsed value.
     * @throws XMLReaderException If not an int.
     */
    public static int processInt(final String value) throws XMLReaderException
    {
        try
        {
            return Integer.parseInt(value.trim());
        }
        catch(final NumberFormatException e)
        {
            throw new XMLReaderException("Integer string '" + value.trim() + "' is not a number.");
        }
    }

    /**
     * @param element Element to check
     * @return value of element.hasAttributes() || element.hasChildNodes().
     */
    public static boolean doesElementHaveAttributesOrChildren(final Element element)
    {
        return (element.hasAttributes() || element.hasChildNodes());
    }

    /**
     * Add the child to the parent, but only if the child is not empty (see doesElementHaveAttributesOrChildren).
     * 
     * @param parent Parent node.
     * @param child Child to add.
     */
    public static void appendElementIfNotEmpty(final Element parent, final Element child)
    {
        if(doesElementHaveAttributesOrChildren(child))
        {
            parent.appendChild(child);
        }
    }

    /**
     * Shorthand that uses DocumentBuilderFactory and DocumentBuilder to create a Document request.
     * 
     * @return
     * @throws ParserConfigurationException
     */
    private static Document createGenericXMLDocument() throws ParserConfigurationException
    {
        final DocumentBuilderFactory builderFactory = DocumentBuilderFactory.newInstance();
        final DocumentBuilder builder = builderFactory.newDocumentBuilder();
        final Document request = builder.newDocument();
        return request;
    }

    private static void appendElementToXMLDocumentFromXMLWriter(final Document xmlDocument, final XMLWriter writer) throws XMLWriterException
    {
        final Element element = writer.writePropertyToXMLElement(xmlDocument);
        xmlDocument.appendChild(element);
    }

    private static byte[] writeFISByteArray(final Document xmlDocument) throws TransformerException,
                                                                       UnsupportedEncodingException
    {
        final ByteArrayOutputStream out = new ByteArrayOutputStream();
        writeXMLToOutputStream(xmlDocument, out, false, true);
        return out.toByteArray();
    }

    private static byte[] writeFISByteArray(final XMLStreamWriterWriter writer) throws Exception
    {
        final ByteArrayOutputStream out = new ByteArrayOutputStream();
        writeXMLToOutputStream(writer, out, false, true);
        return out.toByteArray();
    }

    /**
     * @param xmlDocument Document to write to a string.
     * @param index True for indentation.
     * @return A String that contains the XML.
     * @throws TransformerException
     * @throws UnsupportedEncodingException
     */
    public static String writeXMLString(final Document xmlDocument, final boolean indent) throws TransformerException,
                                                                                         UnsupportedEncodingException
    {
        final ByteArrayOutputStream out = new ByteArrayOutputStream();
        writeXMLToOutputStream(xmlDocument, out, indent, false);
        return out.toString().replace("\r\n", "\n");
    }

    /**
     * @param xmlDocument Document to write to a string.
     * @param index True for indentation.
     * @return A String that contains the XML.
     * @throws TransformerException
     * @throws UnsupportedEncodingException
     */
    private static String writeXMLString(final XMLStreamWriterWriter writer, final boolean indent) throws Exception
    {
        final ByteArrayOutputStream out = new ByteArrayOutputStream();
        writeXMLToOutputStream(writer, out, indent, false);

        return out.toString().replace("\r\n", "\n");
    }

    /**
     * calls {@link #writeXMLToOutputStream(Document, OutputStream, boolean, boolean, String)} with null passed in for
     * the encryption key phrase to prevent encryption.
     */
    public static void writeXMLToOutputStream(final Document xmlDocument,
                                              final OutputStream output,
                                              final boolean indent,
                                              final boolean fastInfoSet) throws TransformerException,
                                                                        UnsupportedEncodingException
    {
        writeXMLToOutputStream(xmlDocument, output, indent, fastInfoSet, null);
    }

    /**
     * Writes the XML contained in the document to an output stream.
     * 
     * @param xmlDocument {@link Document} to write.
     * @param output {@link OutputStream} to write to.
     * @param indent Indicates if indentation should be included.
     * @param fastInfoSet Indicates if the document is fast info set. If this is true, the indent flag is ignored.
     * @param encryptionKeyPhrase Encryption passphrase passed to
     *            {@link StreamTools#encryptOutputStream(OutputStream, String)} to encrypt the file. Null means no
     *            encryption.
     * @throws TransformerException
     * @throws UnsupportedEncodingException
     */
    public static void writeXMLToOutputStream(final Document xmlDocument,
                                              OutputStream output,
                                              final boolean indent,
                                              final boolean fastInfoSet,
                                              final String encryptionKeyPhrase) throws TransformerException,
                                                                               UnsupportedEncodingException
    {
        Result outputResult = null;
        output = StreamTools.encryptOutputStream(output, encryptionKeyPhrase);

        //Regular XML outputResult...
        if(!fastInfoSet)
        {
            final OutputStreamWriter write = new OutputStreamWriter(output, "utf-8");
            outputResult = new StreamResult(write);
        }
        //fastInfoset outputResult...
        else
        {
            final SAXDocumentSerializer saxDocumentSerializer = new SAXDocumentSerializer();
            saxDocumentSerializer.setOutputStream(output);
            outputResult = new SAXResult(saxDocumentSerializer);
            ((SAXResult)outputResult).setLexicalHandler(saxDocumentSerializer);
        }

        //Setup the transformer factory.
        final String oldValue = System.setProperty("javax.xml.transform.TransformerFactory",
                                                   "com.sun.org.apache.xalan.internal.xsltc.trax.TransformerFactoryImpl");
        final TransformerFactory xformFactory = TransformerFactory.newInstance();
        if(oldValue == null)
        {
            System.clearProperty("javax.xml.transform.TransformerFactory");
        }
        else
        {
            System.setProperty("javax.xml.transform.TransformerFactory", oldValue);
        }

        //Create a transformer
        if((!fastInfoSet) && (indent))
        {
            xformFactory.setAttribute("indent-number", 4);
        }
        final Transformer idTransform = xformFactory.newTransformer();
        if((!fastInfoSet) && (indent))
        {
            idTransform.setOutputProperty(OutputKeys.METHOD, "xml");
            idTransform.setOutputProperty(OutputKeys.INDENT, "yes");
        }

        //Write the output stream.
        final Source input = new DOMSource(xmlDocument);
        idTransform.transform(input, outputResult);
    }

    /**
     * @param outputStream The {@link OutputStream} to which to write the XML.
     * @param indent True if the XML is to be indented.
     * @param fastInfoset True if the XML is to be fast infoset.
     * @return An {@link XMLStreamWriter} ready for writing.
     * @throws Exception
     */
    public static XMLStreamWriter initializeXMLStreamWriter(final OutputStream outputStream,
                                                            final boolean indent,
                                                            final boolean fastInfoset) throws Exception
    {
        XMLStreamWriter streamWriter = null;
        if(fastInfoset)
        {
            streamWriter = new StAXDocumentSerializer(outputStream);
        }
        else if(indent)
        {
            streamWriter = new IndentingXMLStreamWriter(XMLOutputFactory.newInstance()
                                                                        .createXMLStreamWriter(outputStream, "UTF-8"));
            ((IndentingXMLStreamWriter)streamWriter).setIndentStep("    ");
        }
        else
        {
            streamWriter = XMLOutputFactory.newInstance().createXMLStreamWriter(outputStream, "UTF-8");
        }
        return streamWriter;
    }

    /**
     * Uses {@link XMLStreamWriter} to write XML more efficiently in terms of memory usage. The stream will not be
     * closed when this is done, through it will be flushed.
     * 
     * @param writer What to write.
     * @param outputStream The stream to write to.
     * @param indent
     * @param fastInfoset If true, indent is ignored.
     * @throws Exception
     */
    public static void writeXMLToOutputStream(final XMLStreamWriterWriter writer,
                                              final OutputStream outputStream,
                                              final boolean indent,
                                              final boolean fastInfoset) throws Exception
    {
        //Initialize the stream writer
        final XMLStreamWriter streamWriter = initializeXMLStreamWriter(outputStream, indent, fastInfoset);

        //Write the document
        try
        {
            streamWriter.writeStartDocument("UTF-8", "1.0");
            writer.writeXML(streamWriter);
            streamWriter.writeEndDocument();
        }
        finally
        {
            //I'm not sure that its necessary to flush and close the outputStream after doing so for the
            //streamWriter, but by debugging I saw that the closed flag in outputStream was false after 
            //closing streamWriter, so I figured I better.
            streamWriter.flush();
            streamWriter.close();
            outputStream.flush();
        }
    }

    /**
     * @param fastInfoset True for a fastInfoset file extension, false for regular XML.
     * @return {@link #FASTINFOSET_EXTENSION} if fastInfoset is true, or {@link #XML_EXTENSION} if false.
     */
    public static String getXMLFileExtension(final boolean fastInfoset)
    {
        if(fastInfoset)
        {
            return FASTINFOSET_EXTENSION;
        }
        else
        {
            return XML_EXTENSION;
        }
    }

    /**
     * @param file File whose name is to be checked.
     * @return True if the file name ends with extension .{@link #FASTINFOSET_EXTENSION} or .
     *         {@link #FASTINFOSET_EXTENSION}.{@link #GZIP_EXTENSION}.
     */
    public static boolean isFastInfosetFile(final File file)
    {
        return isFastInfosetFile(file.getName());
    }

    /**
     * @param fileName Name of file.
     * @return True if the file name ends with extension .{@link #FASTINFOSET_EXTENSION} or .
     *         {@link #FASTINFOSET_EXTENSION}.{@link #GZIP_EXTENSION}.
     */
    public static boolean isFastInfosetFile(final String fileName)
    {
        return fileName.endsWith("." + FASTINFOSET_EXTENSION)
            || (fileName.endsWith("." + FASTINFOSET_EXTENSION + "." + GZIP_EXTENSION));
    }

    /**
     * @param file File whose name is to be checked.
     * @return True if the file name ends with extension .{@link #XML_EXTENSION} or . {@link #XML_EXTENSION}.
     *         {@link #GZIP_EXTENSION}.
     */
    public static boolean isXMLFile(final File file)
    {
        return isXMLFile(file.getName());
    }

    /**
     * @param file File whose name is to be checked.
     * @return True if the file name ends with extension .{@link #XML_EXTENSION} or . {@link #XML_EXTENSION}.
     *         {@link #GZIP_EXTENSION}.
     */
    public static boolean isXMLFile(final String fileName)
    {
        return fileName.endsWith("." + XML_EXTENSION)
            || (fileName.endsWith("." + XML_EXTENSION + "." + GZIP_EXTENSION));
    }

    /**
     * @param file {@link File} to check
     * @return True if the file name ends with extenstion {@link #GZIP_EXTENSION}.
     */
    public static boolean isGZIPFile(final File file)
    {
        return isGZIPFile(file.getName());
    }

    /**
     * @param fileName Name of file
     * @return True if the file name ends with extenstion {@link #GZIP_EXTENSION}.
     */
    public static boolean isGZIPFile(final String fileName)
    {
        return fileName.endsWith("." + GZIP_EXTENSION);
    }

    /**
     * Calls {@link #writeXMLFileFromXMLWriter(File, XMLWritable, boolean)} passing in false for the indentation flag.
     * It also assumes no encryption.
     */
    public static void writeXMLFileFromXMLWriter(final File file, final XMLWritable writable) throws Exception
    {
        writeXMLFileFromXMLWriter(file, writable, false);
    }

    /**
     * Calls {@link #writeXMLFileFromXMLWriter(File, XMLWritable, boolean, String)} with null passed in for the
     * encryption key phrase so that there is no encryption.
     */
    public static void writeXMLFileFromXMLWriter(final File file,
                                                 final XMLWritable writable,
                                                 final boolean withIndentation) throws Exception
    {
        writeXMLFileFromXMLWriter(file, writable, withIndentation, null);
    }

    /**
     * If writable's {@link XMLWritable#getWriter()} is a {@link XMLStreamWriterWriter}, then a StAX XML writer will be
     * used, which is less memory intensive and faster. For all others, a {@link Document} will be constructed in memory
     * to store the entire document prior to writing via a more standard {@link TransformerFactory} provided
     * {@link Transformer}.
     * 
     * @param file File to generate. Will be a FastInfoset file if {@link #isFastInfosetFile(File)} returns true.
     *            Otherwise, XML.
     * @param writable XMLWritable providing the output.
     * @param withIndentation True if an XML file generated should be properly indented and formatted.
     * @param encryptionKeyPhrase Encryption passphrase passed to either
     *            {@link #writeXMLToOutputStream(Document, OutputStream, boolean, boolean, String)} or
     *            {@link #writeXMLToOutputStream(XMLStreamWriterWriter, OutputStream, boolean, boolean, String)}.
     * @throws Exception FastInfoset files only.
     */
    @SuppressWarnings("resource")
    public static void writeXMLFileFromXMLWriter(final File file,
                                                 final XMLWritable writable,
                                                 final boolean withIndentation,
                                                 final String encryptionKeyPhrase) throws Exception
    {
        //Output stream can be gzipped.
        final FileOutputStream fileOut = new FileOutputStream(file);
        OutputStream output = fileOut;
        if(isGZIPFile(file))
        {
            output = new GZIPOutputStream(fileOut);
        }

        //Handle encryption, if any.
        if(encryptionKeyPhrase != null)
        {
            //This line generates a resource leak warning because fileOut is not closed at this type, but it will be closed
            //as a byproduct of closing the cipher stream, so who cares?
            output = StreamTools.encryptOutputStream(output, encryptionKeyPhrase);
        }

        //For fast writing that outputs immediately to a stream...
        if(writable.getWriter() instanceof XMLStreamWriterWriter)
        {
            final XMLStreamWriterWriter writer = (XMLStreamWriterWriter)writable.getWriter();
            if(isFastInfosetFile(file))
            {
                writeXMLToOutputStream(writer, output, false, true);
            }
            else if(withIndentation)
            {
                writeXMLToOutputStream(writer, output, true, false);
            }
            else
            {
                writeXMLToOutputStream(writer, output, false, false);
            }
        }
        //For slow writing that requires constructing an entire Document first...
        else
        {
            final Document doc = XMLTools.createGenericXMLDocument();
            XMLTools.appendElementToXMLDocumentFromXMLWriter(doc, writable.getWriter());
            if(isFastInfosetFile(file))
            {
                writeXMLToOutputStream(doc, output, false, true);
            }
            else if(withIndentation)
            {
                writeXMLToOutputStream(doc, output, true, false);
            }
            else
            {
                writeXMLToOutputStream(doc, output, false, false);
            }
        }
        output.flush();
        output.close();
    }

    /**
     * See {@link #writeXMLFileFromXMLWriter(File, XMLWritable, boolean)} for instructions on how the XML is written.
     * 
     * @param writable The {@link XMLWritable} that provides the {@link XMLWriter} that provides the output to place in
     *            the XML String.
     * @return A String containing the XML.
     * @throws Exception
     */
    public static String writeXMLStringFromXMLWriter(final XMLWritable writable) throws Exception
    {
        if(writable.getWriter() instanceof XMLStreamWriterWriter)
        {
            return XMLTools.writeXMLString((XMLStreamWriterWriter)writable.getWriter(), false);
        }
        else
        {
            final Document doc = XMLTools.createGenericXMLDocument();
            XMLTools.appendElementToXMLDocumentFromXMLWriter(doc, writable.getWriter());
            return XMLTools.writeXMLString(doc, false);
        }
    }

    /**
     * See {@link #writeXMLFileFromXMLWriter(File, XMLWritable, boolean)} for instructions on how the XML is written.
     * 
     * @param writable The {@link XMLWritable} that provides the {@link XMLWriter} that provides the output to go into
     *            the String.
     * @return String containing the XML with indentation and formatting.
     * @throws Exception
     */
    public static String writeXMLStringFromXMLWriterWithIndentation(final XMLWritable writable) throws Exception
    {
        if(writable.getWriter() instanceof XMLStreamWriterWriter)
        {
            return XMLTools.writeXMLString((XMLStreamWriterWriter)writable.getWriter(), true);
        }
        else
        {
            final Document doc = XMLTools.createGenericXMLDocument();
            XMLTools.appendElementToXMLDocumentFromXMLWriter(doc, writable.getWriter());
            return XMLTools.writeXMLString(doc, true);
        }
    }

    /**
     * See {@link #writeXMLFileFromXMLWriter(File, XMLWritable, boolean)} for instructions on how the XML is written.
     * 
     * @param writable Provides the {@link XMLWriter} that provides the output to go into the byte[]
     * @return byte[] containing FastInfoset output.
     * @throws Exception
     */
    public static byte[] writeFISByteArrayFromXMLWriter(final XMLWritable writable) throws Exception
    {
        if(writable.getWriter() instanceof XMLStreamWriterWriter)
        {
            return XMLTools.writeFISByteArray((XMLStreamWriterWriter)writable.getWriter());
        }
        else
        {
            final Document doc = XMLTools.createGenericXMLDocument();
            XMLTools.appendElementToXMLDocumentFromXMLWriter(doc, writable.getWriter());
            return XMLTools.writeFISByteArray(doc);
        }
    }

    /**
     * Calls {@link #readXMLFromFile(File, XMLReadable, String)} passing in null for the encryption key phrase so that
     * no de-encryption is done.
     */
    public static void readXMLFromFile(final File file, final XMLReadable readable) throws GenericXMLReadingHandlerException
    {
        readXMLFromFile(file, readable, null);
    }

    /**
     * Reads the XML from the file using the XMLReadable given and a GenericXMLReadingHandler.
     * 
     * @param file File to read.
     * @param readable The reader provider.
     * @param encryptionKeyPhrase Encryption passphrase passed to
     *            {@link GenericXMLReadingHandler#readXMLFromStreamAndClose(InputStream, boolean, String)}. Null implies
     *            the stream is not encrypted.
     * @throws GenericXMLReadingHandlerException If the handler encounters a problem.
     */
    public static void readXMLFromFile(final File file, final XMLReadable readable, final String encryptionKeyPhrase) throws GenericXMLReadingHandlerException
    {
        final GenericXMLReadingHandler handler = new GenericXMLReadingHandler(readable.getReader());
        handler.readXMLFromFile(file, encryptionKeyPhrase);
    }

    /**
     * Calls {@link #readXMLFromStreamAndClose(InputStream, boolean, XMLReadable, String)} passing in null for the
     * encryption key phrase so that de-encryption is not performed.
     */
    public static void readXMLFromStreamAndClose(final InputStream stream,
                                                 final boolean isFastInfosetStream,
                                                 final XMLReadable readable) throws GenericXMLReadingHandlerException
    {
        readXMLFromStreamAndClose(stream, isFastInfosetStream, readable, null);
    }

    /**
     * Reads the XML from given input stream using the XMLReadable given and a GenericXMLReadingHandler. This closes the
     * stream after it is done reading (via {@link GenericXMLReadingHandler} readXMLFromStream).
     * 
     * @param file File to read.
     * @param isFastInfosetStream Treu if the stream is fastInfoset.
     * @param readable The reader provider.
     * @param encryptionKeyPhrase Encryption passphrase passed to
     *            {@link GenericXMLReadingHandler#readXMLFromStreamAndClose(InputStream, boolean, String)}. Null implies
     *            the stream is not encrypted.
     * @throws GenericXMLReadingHandlerException If the handler encounters a problem.
     */
    public static void readXMLFromStreamAndClose(final InputStream stream,
                                                 final boolean isFastInfosetStream,
                                                 final XMLReadable readable,
                                                 final String encryptionKeyPhrase) throws GenericXMLReadingHandlerException
    {
        final GenericXMLReadingHandler handler = new GenericXMLReadingHandler(readable.getReader());
        handler.readXMLFromStreamAndClose(stream, isFastInfosetStream, encryptionKeyPhrase);
    }

    /**
     * Calls {@link #readXMLFromStream(InputStream, boolean, XMLReadable, String)} passing in null for the encryption
     * key phrase so that de-encryption is not performed.
     */
    public static void readXMLFromStream(final InputStream stream,
                                         final boolean isFastInfosetStream,
                                         final XMLReadable readable) throws GenericXMLReadingHandlerException
    {
        final GenericXMLReadingHandler handler = new GenericXMLReadingHandler(readable.getReader());
        handler.readXMLFromStream(stream, isFastInfosetStream);
    }

    /**
     * Reads the XML from given input stream using the XMLReadable given and a GenericXMLReadingHandler. This will not
     * close the stream when done.
     * 
     * @param stream {@link InputStream} to read.
     * @param isFastInfosetStream Treu if the stream is fastInfoset.
     * @param readable The reader provider.
     * @param encryptionKeyPhrase Encryption passphrase passed to
     *            {@link GenericXMLReadingHandler#readXMLFromStream(InputStream, boolean, String)}. Null implies the
     *            stream is not encrypted.
     * @throws GenericXMLReadingHandlerException If the handler encounters a problem.
     */
    public static void readXMLFromStream(final InputStream stream,
                                         final boolean isFastInfosetStream,
                                         final XMLReadable readable,
                                         final String encryptionKeyPhrase) throws GenericXMLReadingHandlerException
    {
        final GenericXMLReadingHandler handler = new GenericXMLReadingHandler(readable.getReader());
        handler.readXMLFromStream(stream, isFastInfosetStream, false, encryptionKeyPhrase); //False is the close flag.
    }

    /**
     * Reads the XML from given input stream using the XMLReadable given and a GenericXMLReadingHandler. This closes the
     * stream after it is done reading (via {@link GenericXMLReadingHandler} readXMLFromStream). Assumes no encryption.
     * 
     * @param resourceName Name of the system resource to read.
     * @param readable The reader provider.
     * @throws GenericXMLReadingHandlerException If the handler encounters a problem.
     */
    public static void readXMLFromResource(final String resourceName, final XMLReadable readable) throws GenericXMLReadingHandlerException
    {
        final GenericXMLReadingHandler handler = new GenericXMLReadingHandler(readable.getReader());
        handler.readXMLFromResource(resourceName);
    }

    /**
     * Reads the XML from the string using the XMLReader given and a GenericXMLReadingHandler. Assumes no encryption.
     * 
     * @param xmlString String containing the XML to read.
     * @param readable The {@link XMLReadable} that returns the reader.
     * @throws GenericXMLReadingHandlerException If the handler encounters a problem.
     */
    public static void readXMLFromString(final String xmlString, final XMLReadable readable) throws GenericXMLReadingHandlerException
    {
        final GenericXMLReadingHandler handler = new GenericXMLReadingHandler(readable.getReader());
        handler.readXMLFromString(xmlString);
    }

    /**
     * Removes an attribute from the given {@link Attributes}.
     * 
     * @param base
     * @param qNameToRemove
     */
    public static void removeAttribute(final Attributes base, final String qNameToRemove)
    {
        final int index = base.getIndex(qNameToRemove);
        if(index >= 0)
        {
            if(base instanceof AttributesImpl)
            {
                ((AttributesImpl)base).removeAttribute(index);
            }
        }
    }

    /**
     * Creates an {@link Element} from the {@link XMLWriter} via the
     * {@link XMLWriter#writePropertyToXMLElement(Document)} method and then writes that {@link Element} to an
     * {@link XMLStreamWriter} via {@link #writeElementToXMLStreamWriter(Element, XMLStreamWriter)}.
     * 
     * @param writer The {@link XMLWriter} to write.
     * @param streamWriter The {@link XMLStreamWriter} to write it to.
     * @throws Exception
     */
    public static void writeXMLWriterToXMLStreamWriter(final XMLWriter writer, final XMLStreamWriter streamWriter) throws Exception
    {
        final Element element = writer.writePropertyToXMLElement(createGenericXMLDocument());
        XMLTools.writeElementToXMLStreamWriter(element, streamWriter);
    }

    /**
     * Writes an {@link Element} to an {@link XMLStreamWriter}
     * 
     * @param element {@link Element} to write.
     * @param writer {@link XMLStreamWriter} to write to.
     * @throws Exception
     */
    public static void writeElementToXMLStreamWriter(final Element element, final XMLStreamWriter writer) throws Exception
    {
        if((!element.hasChildNodes()) && (!element.hasAttributes()))
        {
            writer.writeEmptyElement(element.getTagName());
        }
        else
        {
            writer.writeStartElement(element.getTagName());
            final NamedNodeMap attrMap = element.getAttributes();
            if(element.hasAttributes())
            {
                for(int i = 0; i < attrMap.getLength(); i++)
                {
                    final Node node = attrMap.item(i);
                    writer.writeAttribute(node.getNodeName(), node.getNodeValue());
                }
            }
            if(element.hasChildNodes())
            {
                for(int i = 0; i < element.getChildNodes().getLength(); i++)
                {
                    final Node node = element.getChildNodes().item(i);

                    if(node.getNodeType() == Node.TEXT_NODE)
                    {
                        writer.writeCharacters(node.getTextContent());
                    }
                    else if(node.getNodeType() == Node.ELEMENT_NODE)
                    {
                        writeElementToXMLStreamWriter((Element)node, writer);
                    }
                }
            }
            writer.writeEndElement();
        }
        writer.flush();
    }

}
